package cn.hperfect.nbquerier.core.querier;

import cn.hperfect.nbquerier.annotation.NbView;
import cn.hperfect.nbquerier.annotation.TableJoin;
import cn.hperfect.nbquerier.config.NbQuerierConfigFactory;
import cn.hperfect.nbquerier.config.NbQuerierConfiguration;
import cn.hperfect.nbquerier.config.NbQuerierCons;
import cn.hperfect.nbquerier.core.components.builder.INbQueryBuilder;
import cn.hperfect.nbquerier.core.components.executor.INbExecutor;
import cn.hperfect.nbquerier.core.components.type.INbQueryType;
import cn.hperfect.nbquerier.core.conditions.ISqlSegment;
import cn.hperfect.nbquerier.core.conditions.segments.MergeSegments;
import cn.hperfect.nbquerier.core.metedata.*;
import cn.hperfect.nbquerier.core.metedata.field.ClassNbField;
import cn.hperfect.nbquerier.core.metedata.inter.INbField;
import cn.hperfect.nbquerier.core.metedata.inter.INbTable;
import cn.hperfect.nbquerier.core.metedata.table.VirtualTable;
import cn.hperfect.nbquerier.core.querier.execute.DoFunction;
import cn.hperfect.nbquerier.core.querier.execute.DoUpdate;
import cn.hperfect.nbquerier.core.transaction.INbTransaction;
import cn.hperfect.nbquerier.enums.NbFieldType;
import cn.hperfect.nbquerier.enums.NbOrderType;
import cn.hperfect.nbquerier.enums.QueryType;
import cn.hperfect.nbquerier.toolkit.SqlUtils;
import cn.hperfect.nbquerier.toolkit.StringPool;
import cn.hperfect.nbquerier.toolkit.support.SFunction;
import cn.hutool.core.collection.CollUtil;
import cn.hutool.core.collection.ListUtil;
import cn.hutool.core.lang.Assert;
import cn.hutool.core.util.BooleanUtil;
import cn.hutool.core.util.StrUtil;
import cn.hutool.extra.spring.SpringUtil;
import org.springframework.lang.Nullable;

import java.sql.Connection;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;
import java.util.function.Consumer;

/**
 * @author huanxi
 * @version 1.0
 * @date 2021/11/24 9:12 上午
 */
public interface NbQuerier<T> extends ISqlSegment, DoFunction<T>, DoUpdate<T>, IResultMap<T> {
    static INbExecuteBatch startBatch(int size) {
        return new NbExecuteBatch(size);
    }

    static INbExecutor newExecutor() {
        return SpringUtil.getBean(NbQuerierConfiguration.class).newNbExecutor();
    }

    /**
     * 获取条件表达式
     *
     * @return
     */
    MergeSegments getExpression();


    /**
     * 根据类中注解构建querier对象
     *
     * @param clazz
     * @param <T>
     * @return
     */
    static <T> NbQuerier<T> table(Class<T> clazz) {
        //获取builder 构建INbQuerier
        return table(clazz, null);
    }

    static NbQuerier<?> wrapper(NbQuerier<?> fromQuerier) {
        return wrapper(fromQuerier, fromQuerier.getTableAlias(fromQuerier.getTableName()));
    }

    /**
     * 包装一层 select xxx from ( ${ fromQuerier } )  ${alias}
     *
     * @param fromQuerier
     * @param alias
     * @return
     */
    static NbQuerier<?> wrapper(NbQuerier<?> fromQuerier, String alias) {
        INbQueryBuilder builder = SpringUtil.getBean(INbQueryBuilder.class);
        DefaultNbQuerier<?> result = builder.newNbQuerier();
        NbQueryInfo queryInfo = new NbQueryInfo();
        queryInfo.setQueryType(QueryType.FORM_QUERY);
        queryInfo.setAlias(alias);
        String virtualTableName = String.format(NbQuerierCons.SUB_SQL, alias);
        //字段复制
        List<ClassNbField> nbFields = new ArrayList<>();
        VirtualTable virtualTable = new VirtualTable(virtualTableName, null, new PrimaryKey(), nbFields);
        for (INbField field : fromQuerier.getQueryInfo().getFields()) {
            //忽略删除字段
            if (field.getFieldType() == NbFieldType.DELETE) {
                continue;
            }
            ClassNbField nbField = new ClassNbField();
            nbFields.add(nbField);
            nbField.setName(field.getName());
            nbField.setFieldType(field.getFieldType());
            nbField.setTable(virtualTable);
            if (field.getFieldType() == NbFieldType.ORDER) {
                //添加
                String tableAlias = fromQuerier.getTableAliasMap().get(field.getTableName());
                fromQuerier.additionField(SqlUtils.withAlias(tableAlias, field.getName()));
            }
        }
        //构建字段的时候,是通过表名来的
        queryInfo.setTable(virtualTable);
        queryInfo.setFromQuerier(fromQuerier);
        result.setQueryInfo(queryInfo);
        DefaultNbQuerier defaultNbQuerier = (DefaultNbQuerier) fromQuerier;
        result.setParamBuilder(defaultNbQuerier.getParamBuilder());
        return result;
    }

    /**
     * beta 版本,不稳定
     *
     * @param fromQuerier
     * @param alias
     * @return
     */
    static NbQuerier<?> from(NbQuerier<?> fromQuerier, String alias) {
        INbQueryBuilder builder = SpringUtil.getBean(INbQueryBuilder.class);
        DefaultNbQuerier<?> result = builder.newNbQuerier();
        NbQueryInfo queryInfo = new NbQueryInfo();
        queryInfo.setQueryType(QueryType.FORM_QUERY);
        queryInfo.setAlias(alias);
        queryInfo.setFromQuerier(fromQuerier);
        NbQueryInfo fromQuerierInfo = fromQuerier.getQueryInfo();
        queryInfo.setTable(fromQuerierInfo.getTable());
        result.setQueryInfo(queryInfo);
        return result;
    }

    static <T> NbQuerier<T> table(INbTable table) {
        //获取builder 构建INbQuerier
        return table(table, null, null);
    }

    static <T> NbQuerier<T> table(NbQuerier<?> querier, Class<T> clazz) {
        INbQueryBuilder builder = SpringUtil.getBean(INbQueryBuilder.class);
        return builder.build(querier, clazz);
    }

    static <T> NbQuerier<T> table(INbTable table, @Nullable Class<T> clazz) {
        //获取builder 构建INbQuerier
        return table(table, clazz, null);
    }

    static <T> NbQuerier<T> table(INbTable table, String alias) {
        //获取builder 构建INbQuerier
        return table(table, null, alias);
    }

    static <T> NbQuerier<T> table(Class<T> clazz, String alias) {
        //获取builder 构建INbQuerier(兼容quarkus)
        INbQueryBuilder builder = NbQuerierConfigFactory.INSTANCE.getConfiguration().getNbQueryBuilder();
        return builder.build(clazz, alias);
    }

    static <T> NbQuerier<T> table(INbTable table, @Nullable Class<T> tableClazz, String alias) {
        INbQueryBuilder builder = SpringUtil.getBean(INbQueryBuilder.class);
        return builder.build(table, tableClazz, alias);
    }

    static <T> NbQuerier<T> table(NbQueryInfo queryInfo) {
        INbQueryBuilder builder = SpringUtil.getBean(INbQueryBuilder.class);
        return builder.build(queryInfo);
    }

    @Deprecated
    static <T> NbQuerier<T> view(Class<T> viewClazz, Class<?> tableClazz, String alias) {
        INbQueryBuilder builder = SpringUtil.getBean(INbQueryBuilder.class);
        NbQueryInfo queryInfo = builder.buildQueryInfo(tableClazz, alias);
        NbQuerier<T> querier = builder.build(queryInfo);
        return querier.setResultClass(viewClazz);
    }

    @Deprecated
    static <T> NbQuerier<T> view(Class<T> view) {
        NbView nbView = view.getAnnotation(NbView.class);
        Assert.notNull(nbView, "@NbView不能为空");
        NbQuerier<T> querier = NbQuerier.view(view, nbView.masterTable(), nbView.alias());
        TableJoin[] joins = nbView.joins();
        if (joins.length > 0) {
            for (TableJoin join : joins) {
                querier.leftJoin(join.joinTable(), join.alias());
            }
        }
        //视图字段构建->
        return querier;
    }

    /**
     * 添加行号(一般用于分组查询)
     *
     * @param partitionBy
     * @param orderField
     * @param sort
     * @return
     */
    NbQuerier<T> addRowNum(String partitionBy, String orderField, String sort);

    /**
     * 附加字段,不会检测合法性，注意sql注入
     *
     * @param field
     * @return
     */
    NbQuerier<T> additionField(String field);

    NbQuerier<T> additionFields(List<String> fields);

    /**
     * 字段设置
     *
     * @param field
     * @return
     */
    NbQuerier<T> field(String field);
    default NbQuerier<T> emptyField(){
        return field(StringPool.EMPTY);
    }

    default NbQuerier<T> field(SFunction<T, ?> field) {
        return field(getFieldName(field));
    }

    @SuppressWarnings("unchecked")
    default NbQuerier<T> field(SFunction<T, ?>... fields) {
        return field(CollUtil.join(Arrays.asList(getFieldNames(Arrays.asList(fields))), ","));
    }

    default NbQuerier<T> field(List<SFunction<T, ?>> fields) {
        return field(CollUtil.join(Arrays.asList(getFieldNames(fields)), ","));
    }


    /**
     * 左连接
     *
     * @param clazz
     * @param alias
     * @param on
     * @return
     */
    NbQuerier<T> leftJoin(Class<?> clazz, String alias, String on);

    /**
     * 左连接
     *
     * @param clazz
     * @param alias
     * @param tableAField
     * @param tableBField
     * @param <F>
     * @param <E>         返回值(字段类型)
     * @return
     */
    <F, E> NbQuerier<T> leftJoin(Class<F> clazz, String alias, SFunction<F, E> tableAField, SFunction<T, E> tableBField);


    /**
     * 自动左连
     */
    NbQuerier<T> leftJoin(Class<?> clazz, String alias);

    NbQuerier<T> leftJoin(NbQueryInfo queryInfo, String on, boolean lateral);

    default NbQuerier<T> leftJoin(NbQueryInfo queryInfo, String on) {
        return this.leftJoin(queryInfo, on, false);
    }

    @Deprecated
    NbQuerier<T> leftJoin(NbQuerier<?> querier, String alias);

    /**
     * 表子查询
     *
     * @param querier
     * @param alias
     * @param on
     * @return
     */

    default NbQuerier<T> leftJoin(NbQuerier<?> querier, String alias, String on) {
        return leftJoin(querier, alias, on, false);
    }

    NbQuerier<T> leftJoin(NbQuerier<?> querier, String alias, String on, boolean lateral);

    /**
     * 从表中找字段
     *
     * @param name
     * @param table
     * @return
     */
    QueryField findQueryField(String name, String table);

    QueryField findQueryField(String name);


    /**
     * 支持注入,对当前对象操作,一般用于链式查询
     *
     * @param consumer
     * @return
     */
    default NbQuerier<T> with(Consumer<NbQuerier<T>> consumer) {
        consumer.accept(this);
        return this;
    }

    /**
     * 添加查询字段
     * 作用于自己构建的字段,避免报错早不到字段，和执行类型
     *
     * @param name
     * @param type
     * @return
     */
    default NbQuerier<T> addQueryField(String name, INbQueryType type) {
        getQueryFields().add(new QueryField().setName(name).setType(type));
        return this;
    }

    default NbQuerier<T> with(Boolean condition, Consumer<NbQuerier<T>> consumer) {
        if (BooleanUtil.isTrue(condition)) {
            return with(consumer);
        }
        return this;
    }

    List<QueryValParam> getParams();

    /**
     * 条件聚合
     *
     * @return
     */
    List<ConditionAgg> getConditionAggList();

    List<JoinTableRule> getJoins();

    /**
     * 格式化变量
     *
     * @param type   数据类型
     * @param value
     * @param update 是否更新,更新和查询不一样, false 会设置成(xx,xx)
     * @return
     */
    String formatVariable(INbQueryType type, Object value, boolean update);

    String formatVariable(QueryValParam param);

    String formatVariable(INbQueryType type, Object value, boolean update, boolean toArray);

    /**
     * 设置权限值
     *
     * @param dataScope
     * @return
     */
    default NbQuerier<T> dataScope(List<Object> dataScope) {
        Assert.notEmpty(dataScope, "数据权限不能为空");
        getQueryInfo().setDataScope(dataScope);
        return this;
    }

    default NbQuerier<T> dataPerm(Object dataScope) {
        getQueryInfo().setPermValue(dataScope);
        return this;
    }

    default NbQuerier<T> dataScope(Object dataScope) {
        return dataScope(ListUtil.toList(dataScope));
    }

    /**
     * 是否有where条件
     *
     * @return
     */
    boolean hasWhere();

    /**
     * 带删除数据,不加软删除后缀
     *
     * @return
     */
    default NbQuerier<T> withDelete() {
        getQueryInfo().setWithDelete(true);
        return this;
    }

    /**
     * 设置主表别名
     *
     * @param alias
     * @return
     */
    NbQuerier<T> alias(String alias);

    /**
     * 查询json数组作为一个字段
     *
     * @param fieldName
     * @param querier
     * @return
     */
    NbQuerier<T> withMany(String fieldName, NbQuerier<?> querier);

    NbQuerier<T> withOne(String fieldName, NbQuerier<?> querier);

    default NbQuerier<T> withOne(SFunction<T, ?> fieldName, NbQuerier<?> querier) {
        return withOne(getFieldName(fieldName), querier);
    }

    /**
     * 获取sql语句 并且连接参数
     *
     * @param querier
     * @return
     */
    String buildConnect(NbQuerier<?> querier);


    String connectParam(NbQuerier<?> querier, String sql);


    default NbQuerier<T> withMany(SFunction<T, List<?>> fieldName, NbQuerier<?> querier) {
        return withMany(fieldName, querier, true);
    }

    default NbQuerier<T> withMany(SFunction<T, List<?>> fieldName, NbQuerier<?> querier, boolean inline) {
        //不带别名
        return withMany(getFieldName(fieldName, false), querier);
    }

    /**
     * 一对多数量查询
     *
     * @param fieldName
     * @param querier
     * @return
     */
    NbQuerier<T> withManyCount(String fieldName, NbQuerier<?> querier);

    default NbQuerier<T> withManyCount(SFunction<T, ?> fieldName, NbQuerier<?> querier) {
        return withManyCount(getFieldName(fieldName, false), querier);
    }


    NbQuerier<T> union(NbQuerier<T> querier);

    NbQuerier<T> unionAll(NbQuerier<T> querier);

    NbQuerier<T> having(String sqlHaving);

    NbQuerier<T> group(String... columns);

    default NbQuerier<T> group(SFunction<T, ?> column) {
        return group(getFieldName(column));
    }

    /**
     * 自动替换 xx.*
     *
     * @param val
     * @return
     */
    default NbQuerier<T> fieldReplace(boolean val) {
        getQueryInfo().setFieldReplace(val);
        return this;
    }

    /**
     * 自动排序
     * 排序字段 desc
     * pk desc
     *
     * @return
     */
    NbQuerier<T> autoOrder();

    /**
     * 设置表达式更新
     *
     * @param s
     * @return
     */
    NbQuerier<T> updateExpr(String s);

    /**
     * sql 表达式更新
     * username = ?,?为占位符
     *
     * @param sql
     * @param params
     * @return
     */
    default NbQuerier<T> updateExpr(String sql, QueryValParam... params) {
        return this.updateExpr(parseSqlParam(sql, params));
    }

    @Deprecated
    NbQuerier<T> updateExpr(String sql, Object... param);

    /**
     * 查找字段
     *
     * @param fieldName
     * @return
     */
    QueryField findField(String fieldName);

    /**
     * 数组类型
     * 移除数组元素
     *
     * @param fieldFunc
     * @return
     */
    default NbQuerier<T> updateRemoveArrayItem(SFunction<T, ?> fieldFunc, Object item) {
        String fieldName = getFieldName(fieldFunc);
        QueryField field = findField(fieldName);
        INbQueryType type = field.getType();
        Assert.isTrue(type.isArray(), "只支持数组类型移除元素");
        String param = formatVariable(type.getArraySubType(), item, true);
        String sql = StrUtil.format("{}=array_remove({},{})", field.getName(), field.getName(), param);
        return updateExpr(sql);
    }

    /**
     * 结果集class
     *
     * @param resultClass
     * @return
     */

    NbQuerier<T> setResultClass(Class<T> resultClass);

    /**
     * 指定主表的schema
     *
     * @param tempSchemaName
     * @return
     */
    NbQuerier<T> tableSchema(String tempSchemaName);

    /**
     * 获取表名
     *
     * @return
     */
    default String getTableName() {
        return getQueryInfo().getTableName();
    }

    /**
     * 多字段排序
     *
     * @param defaultOrderInfos
     * @return
     */
    default NbQuerier<T> order(List<OrderInfoStr> defaultOrderInfos) {
        for (OrderInfoStr defaultOrderInfo : defaultOrderInfos) {
            this.order(NbOrderType.parse(defaultOrderInfo.getOrder()), defaultOrderInfo.getNullsPosition(), defaultOrderInfo.getOrderBy());
        }
        return this;
    }

    default NbQuerier<T> allowSetNull() {
        getQueryInfo().setAllowSetNull(true);
        return this;
    }

    /**
     * 获取数据源名称
     *
     * @return
     */
    String getDsName();

    /**
     * 加锁 其他事务无法获取 该事务查询数据
     * for update
     *
     * @return
     */
    NbQuerier<T> lock();

    /**
     * 设置链接
     *
     * @param connection
     * @return
     */

    NbQuerier<T> setConnection(Connection connection);

    /**
     * 设置事务
     *
     * @param tx
     * @return
     */
    NbQuerier<T> setTx(INbTransaction tx);

    /**
     * 批量提交
     *
     * @param nbBatch
     * @return
     */

    NbQuerier<T> batch(INbExecuteBatch nbBatch);
}
