package cn.hperfect.nbquerier.core.querier;

import cn.hperfect.nbquerier.config.NbQuerierConfiguration;
import cn.hperfect.nbquerier.config.NbQuerierCons;
import cn.hperfect.nbquerier.config.properties.NbQuerierProperties;
import cn.hperfect.nbquerier.core.components.builder.INbQueryBuilder;
import cn.hperfect.nbquerier.core.components.builder.INbSqlBuilder;
import cn.hperfect.nbquerier.core.components.builder.impl.ParamBuilder;
import cn.hperfect.nbquerier.core.components.dialect.IDialectWare;
import cn.hperfect.nbquerier.core.components.executor.INbExecutor;
import cn.hperfect.nbquerier.core.components.executor.options.DoUpdateOptions;
import cn.hperfect.nbquerier.core.components.type.INbQueryType;
import cn.hperfect.nbquerier.core.components.type.NbQueryType;
import cn.hperfect.nbquerier.core.conditions.ILaterQuerier;
import cn.hperfect.nbquerier.core.conditions.ISqlSegment;
import cn.hperfect.nbquerier.core.conditions.segments.MergeSegments;
import cn.hperfect.nbquerier.core.func.BuildInFunc;
import cn.hperfect.nbquerier.core.metedata.*;
import cn.hperfect.nbquerier.core.metedata.custom.BaseCustomEntity;
import cn.hperfect.nbquerier.core.metedata.custom.CustomEntityBuilder;
import cn.hperfect.nbquerier.core.metedata.inter.INbField;
import cn.hperfect.nbquerier.core.metedata.querier.UpdateData;
import cn.hperfect.nbquerier.core.metedata.table.ClassNbTable;
import cn.hperfect.nbquerier.core.metedata.table.ThisTable;
import cn.hperfect.nbquerier.core.metedata.table.VirtualTable;
import cn.hperfect.nbquerier.core.querier.qrs.QueryExprParser;
import cn.hperfect.nbquerier.core.registry.NbInterceptorRegistry;
import cn.hperfect.nbquerier.core.transaction.INbTransaction;
import cn.hperfect.nbquerier.enums.*;
import cn.hperfect.nbquerier.exceptions.NbSQLException;
import cn.hperfect.nbquerier.exceptions.NbSQLMessageException;
import cn.hperfect.nbquerier.toolkit.*;
import cn.hperfect.nbquerier.toolkit.map.UnderlineCaseMap;
import cn.hperfect.nbquerier.toolkit.support.SFunction;
import cn.hperfect.nbquerier.toolkit.support.SerializedLambda;
import cn.hutool.core.bean.BeanUtil;
import cn.hutool.core.collection.CollUtil;
import cn.hutool.core.collection.ListUtil;
import cn.hutool.core.lang.Assert;
import cn.hutool.core.map.BiMap;
import cn.hutool.core.map.MapUtil;
import cn.hutool.core.util.ArrayUtil;
import cn.hutool.core.util.ObjectUtil;
import cn.hutool.core.util.StrUtil;
import cn.hutool.json.JSONObject;
import lombok.Getter;
import lombok.Setter;
import lombok.experimental.Accessors;
import lombok.experimental.Delegate;
import lombok.extern.slf4j.Slf4j;
import org.springframework.lang.Nullable;

import java.lang.reflect.InvocationTargetException;
import java.sql.Connection;
import java.time.LocalDateTime;
import java.util.*;
import java.util.function.Consumer;
import java.util.stream.Collectors;

import static cn.hperfect.nbquerier.enums.SqlKeyword.GROUP_BY;

/**
 * 查询构造器
 * 1. 查询构造器的构建器
 * 2. 执行器
 * 3. sql构建器
 * 4. 查询信息
 * 1. 表单信息(子查询sql),schema,ds,alias ...
 * 2. 子查询字段
 * 5. 连表信息
 * 6. union todo 待实现
 * 7. 查询完之后的操作
 * 8. 查询条件
 * 待优化
 * 1. 方言,尽量再执行时选择,否在再构建sql 是需要判断数据库类型
 * 2. 脱离spring环境,通过springboot-stater 加载配置 完成初始化
 * 3. 一级缓存(同一个事务内,命中率太低),与二级缓存cache方法(如何清除缓存?缓存key?)
 * 4. 函数class 字段,如 createBy -> createUser(目前只支持视图),如果需要筛选需要再包一层
 * 5. 多数据源事务bug修复
 * 6. 权限设计
 * 7. 字段
 * - 更加灵活的构建，查询target
 * - class函数字段
 * - 直接指定禁止编辑
 *
 * @author huanxi
 * @version 1.0
 * @date 2021/11/24 9:36 上午
 */
@Getter
@Setter
@Slf4j
@Accessors(chain = true)
public class DefaultNbQuerier<T> implements NbQuerier<T> {
    private final INbSqlBuilder sqlBuilder;
    private final INbQueryBuilder queryBuilder;
    private final NbQuerierProperties config;
    private final NbInterceptorRegistry interceptorChain;
    private final IDialectWare dialectWare;
    private final NbQuerierConfiguration configuration;
    /**
     * 查询信息
     */
    private NbQueryInfo queryInfo;

    private INbExecuteBatch batch;

    /**
     * 反射结果class
     */
    private Class<T> resultClass;
    /**
     * 连表信息
     */
    private final List<JoinTableRule> joins = new ArrayList<>();
    /**
     * 条件聚合
     */

    private List<ConditionAgg> conditionAggList;

    /**
     * 所有查询字段
     */
    protected QueryFields queryFields = new QueryFields();

    @Delegate
    private ParamBuilder paramBuilder = new ParamBuilder();

    /**
     * 条件sql片段(复制mybatis plus)
     */
    protected MergeSegments expression = new MergeSegments();

    protected String lastSql;
    /**
     * 数据库连接
     */
    private Connection connection;
    /**
     * 事务
     */
    private INbTransaction tx;
    /**
     * 结果集处理
     */

    private ResultConsumer<T> resultConsumer = new ResultConsumer<>();

    /**
     * 表别名映射 tableName -> alias
     */
    private BiMap<String, String> tableAliasMap = new BiMap<>(new HashMap<>());
    /**
     * 更新数据
     */

    private UpdateData updateData;

    public UpdateData safeGetUpdateData() {
        if (updateData == null) {
            updateData = new UpdateData();
        }
        return updateData;
    }

    public List<ConditionAgg> safeGetConditionAggList() {
        if (conditionAggList == null) {
            conditionAggList = new ArrayList<>();
        }
        return conditionAggList;
    }


    /**
     * 设置事务,使得所有操作都在一个事务内
     * todo 事务
     */
//    private INbTransaction transaction;
    public DefaultNbQuerier(NbQuerierConfiguration configuration) {
        this.configuration = configuration;
        try {
            this.config = configuration.getConfig();
            this.queryBuilder = configuration.getNbQueryBuilder();
            Assert.notNull(this.queryBuilder, "queryBuilder不能为空");
            this.sqlBuilder = configuration.getNbSqlBuilder();
            Assert.notNull(this.queryBuilder, "SqlBuilder不能为空");
            this.interceptorChain = configuration.getNbInterceptorRegistry();
            Assert.notNull(this.interceptorChain, "interceptorChain不能为空");
            this.dialectWare = configuration.getDialectWare();
            Assert.notNull(this.interceptorChain, "dialectWare 不能为空");
        } catch (Exception throwable) {
            throw new NbSQLMessageException("创建查询对象失败:" + throwable.getMessage());
        }
    }

    public void setQueryInfo(NbQueryInfo nbQueryInfo) {
        this.queryInfo = nbQueryInfo;
        if (ThisTable.ME.equals(nbQueryInfo.getTable())) {
            return;
        }

        //todo 设置类型
//        nbQueryInfo.setDbType();
        //别名处理
        Assert.isFalse(tableAliasMap.containsKey(nbQueryInfo.getSql()), "已存在该别名");
        tableAliasMap.put(nbQueryInfo.getTableName(), nbQueryInfo.getAlias());
        Assert.notNull(nbQueryInfo.getFields(), "字段列表不能为空");
        //字段处理
        List<QueryField> list = nbQueryInfo
                .getFields()
                .stream()
                .filter(INbField::isQuery)
                .map(i -> new QueryField(i.getQueryType(),
                        i.getName(),
                        ObjectUtil.defaultIfBlank(i.getTableName(),
                                nbQueryInfo.getTableName()),
                        null, i.getRedirect()))
                .collect(Collectors.toList());
        this.queryFields.puts(list);
    }



    /*-----------------------------------------where start-----------------------------------------*/


    /**
     * 匿名使用wrapper 查询(嵌套查询)
     * 创建一个临时的MergeSegments,替换当前类
     *
     * @param
     */
    @Override
    public NbQuerier<T> where(Consumer<NbQuerier<T>> consumer) {
        MergeSegments old = this.expression;
        this.expression = new MergeSegments();
        consumer.accept(this);
        if (this.hasWhere()) {
            old.add(WrapperKeyword.APPLY, this.expression);
        }
        this.expression = old;
        return this;
    }

    public NbQuerier<T> whereOr(Consumer<NbQuerier<T>> consumer) {
        MergeSegments old = this.expression;
        this.expression = new MergeSegments();
        consumer.accept(this);
        if (this.hasWhere()) {
            MergeSegments temp = this.expression;
            this.expression = old;
            this.or().getExpression().add(WrapperKeyword.APPLY, temp);
        } else {
            this.expression = old;
        }
        return this;
    }

    @Override
    public NbQuerier<T> whereExists(NbQuerier<?> querier) {
        String subSql = buildConnect(querier);
        expression.add(QueryRuleEnum.EXISTS, (ISqlSegment) () -> StringPool.LEFT_BRACKET + subSql + StringPool.RIGHT_BRACKET);
        return this;
    }

    @Override
    public NbQuerier<T> whereIn(String field, NbQuerier<?> querier) {
        // field in (sql)
        String subSql = buildConnect(querier);
        return where(field, QueryRuleEnum.IN, (ISqlSegment) () -> StringPool.LEFT_BRACKET + subSql + StringPool.RIGHT_BRACKET, true);
    }


    @Override
    public NbQuerier<T> whereNotIn(String field, NbQuerier<?> querier) {
        // field in (sql)
        String subSql = buildConnect(querier);
        return where(field, QueryRuleEnum.NOT_IN, (ISqlSegment) () -> StringPool.LEFT_BRACKET + subSql + StringPool.RIGHT_BRACKET, true);
    }


    /**
     * 解析lambda字段名称
     */
    @Override
    public <G, K> String getFieldName(SFunction<G, K> function, boolean withAlias) {
        SerializedLambda resolve = LambdaUtils.resolve(function);
        Class<?> implClass = resolve.getImplClass();
        if (resultClass != null && implClass.isAssignableFrom(resultClass)) {
            implClass = resultClass;
        }
        String tableName = queryBuilder.parseTableInfo(implClass).getTableName();
        String alias = tableAliasMap.get(tableName);
        String field = StrUtil.removePrefix(resolve.getImplMethodName(), "get");
        String underField = StrUtil.toUnderlineCase(field);
        return withAlias ? SqlUtils.withAlias(alias, underField) : underField;
    }

    @Override
    public NbQuerier<T> or() {
        expression.add(SqlKeyword.OR);
        return this;
    }

    public NbQuerier<T> setConnection(Connection connection) {
        this.connection = connection;
        return this;
    }

    @Override
    public NbQuerier<T> batch(INbExecuteBatch nbBatch) {
        this.batch = nbBatch;
        return this;
    }

    @Override
    public NbQuerier<T> not() {
        expression.add(SqlKeyword.NOT);
        return this;
    }

    @Override
    public NbQuerier<T> where(String sql) {
        expression.add(WrapperKeyword.APPLY, () -> sql);
        return this;
    }

    @Override
    public NbQuerier<T> whereQrs(JSONObject qrs) {
        if (qrs != null) {
            QueryExprParser.parseJsonObjectToCondition(this, qrs);
        }
        return this;
    }


    /**
     * where 入口
     *
     * @param field
     * @param ruleEnum
     * @param value
     * @return
     */
    @Override
    public NbQuerier<T> where(String field, QueryRuleEnum ruleEnum, Object value, boolean notNull) {
        Assert.isFalse(notNull && value == null, "条件查询,字段:{}值不能为空", field);
        Assert.notEmpty(field, "查询条件字段不能为空");
        QueryItem queryItem = new QueryItem(field);
        return where(queryItem, ruleEnum, value, notNull);
    }

    @Override
    public NbQuerier<T> where(IQueryItem iQueryItem, QueryRuleEnum ruleEnum, Object value, boolean notNull) {
        INbQueryType type = null;
        if (iQueryItem instanceof QueryItem) {
            QueryItem queryItem = (QueryItem) iQueryItem;
            //查询字段
            QueryField queryField = BuildUtils.findOne(queryFields, queryItem, queryInfo.getTableName(), tableAliasMap);
            assert queryField != null : "不存在字段:" + queryItem.getName();
            if (StrUtil.isNotBlank(queryField.getWhereRedirect())) {
                //重定向
                queryItem.setSub(true);
                queryItem.setSql(queryField.getWhereRedirect());
            } else if (StrUtil.isNotBlank(queryField.getRedirect())) {
                //重定向
                queryItem.setName(queryField.getRedirect());
                return where(queryItem.getExpr(), ruleEnum, value, notNull);
            }
            queryItem.setTableAlias(tableAliasMap.get(queryField.getTableName()));
            type = queryField.getType();
            //枚举数组类型,或者枚举map类型
            if (type == NbQueryType.MAP_ENUM_LIST && getQueryInfo().isClass()) {
                value = NbEnumUtils.getEnumListNewValue(queryInfo.getFields(), (Collection<?>) value, queryItem.getName());
            } else if (type == NbQueryType.MAP_ENUM && getQueryInfo().isClass()) {
                value = NbEnumUtils.getEnumNewValue(queryInfo.getFields(), value, queryItem.getName());
            }
        } else if (iQueryItem instanceof JsonPropQueryItem) {
            JsonPropQueryItem jsonPropQueryItem = (JsonPropQueryItem) iQueryItem;
            type = jsonPropQueryItem.getQueryType();
        } else {
            throw new NbSQLMessageException("不支持该类型");
        }
        String name = iQueryItem.getExpr();
        //不同类型用不同的处理方式(设置where 条件)
        type.setWhereCondition(this, expression, ruleEnum, name, value);
        return this;
    }

    /**
     * 判断是否有值
     *
     * @param value
     * @return
     */
    private boolean hasValue(Object value) {
        if (value == null) {
            return false;
        }
        return !(value instanceof CharSequence && StrUtil.isBlank((CharSequence) value));
    }



    /*-----------------------------------------where end-----------------------------------------*/


    /**
     * 查某列数据
     *
     * @param field
     * @param <F>
     * @return
     */
    @Override
    @SuppressWarnings("unchecked")
    public <F> List<F> column(String field) {
        field = StrUtil.toUnderlineCase(field);
        this.field(field);
        List<F> list = new ArrayList<>();
        List<LinkedHashMap<String, Object>> linkedHashMaps = this.selectMap();
        if (CollUtil.isEmpty(linkedHashMaps)) {
            return list;
        }
        for (LinkedHashMap<String, Object> map : linkedHashMaps) {
            if (map == null) {
                continue;
            }
            for (Map.Entry<String, Object> entry : map.entrySet()) {
                list.add((F) entry.getValue());
            }
        }
        return list;
    }


    @Override
    public NbQuerier<T> page(int pageNo, int pageSize) {
        this.queryInfo.setPageInfo(new PageInfo(pageNo, pageSize));
        return this;
    }

    @Override
    public NbQuerier<T> limit(int offset, int length) {
        dialectWare.limit(this, offset, length);
        return this;
    }

    @Override
    public NbQuerier<T> last(String last) {
        this.lastSql = StrUtil.concat(true, this.lastSql, StringPool.SPACE, last);
        return this;
    }


    @Override
    public NbQuerier<T> union(NbQuerier<T> querier) {
        throw new NbSQLMessageException("union 方法待实现");
    }

    @Override
    public NbQuerier<T> unionAll(NbQuerier<T> querier) {
        List<NbQuerier<?>> unionAllQueries = getQueryInfo().getUnionAllQueries();
        if (unionAllQueries == null) {
            unionAllQueries = new ArrayList<>();
            queryInfo.setUnionAllQueries(unionAllQueries);
        }
        unionAllQueries.add(querier);
        return this;
    }

    /**
     * 将一个对象存在的值作为查询条件,需要注入对象默认值
     *
     * @param t
     * @return
     */
    @Override
    public NbQuerier<T> where(T t) {
        Assert.notNull(this.resultClass, "tableClazz 不存在");
        Map<String, Object> map = beanToMap(t);
        //获取所有字段值-> 加入where条件
        if (MapUtil.isNotEmpty(map)) {
            for (Map.Entry<String, Object> entry : map.entrySet()) {
                if (entry.getValue() != null) {
                    this.where(entry.getKey(), entry.getValue());
                }
            }
        }
        return this;
    }

    @Override
    public NbQuerier<T> having(String sqlHaving) {
        expression.add(SqlKeyword.HAVING, () -> sqlHaving);
        return this;
    }

    /*----------------------------join start------------------------------*/
    @Override
    public NbQuerier<T> leftJoin(Class<?> clazz, String alias, String on) {
        return leftJoin(getJoinQueryInfo(clazz, alias), on);
    }


    @Override
    public <F, E> NbQuerier<T> leftJoin(Class<F> clazz, String alias, SFunction<F, E> tableAField, SFunction<T, E> tableBField) {
        NbQueryInfo joinQueryInfo = getJoinQueryInfo(clazz, alias);
        String tableAFieldName = getFieldName(tableAField);
        String tableBFieldName = getFieldName(tableBField);
//        StrUtil.format("{}={}", tableAFieldName, tableBFieldName)
        return leftJoin(joinQueryInfo, tableAFieldName + '=' + tableBFieldName);
    }


    @Override
    public NbQuerier<T> leftJoin(Class<?> clazz, String alias) {
        //解析on
        NbQueryInfo join = getJoinQueryInfo(clazz, alias);
        return leftJoin(join, parseJoinOn(alias, join));
    }


    @Override
    public NbQuerier<T> leftJoin(NbQuerier<?> querier, String alias) {
        Assert.isTrue(querier.getQueryInfo().isClass(), "只支持class 类型自动连表");
        ClassNbTable table = (ClassNbTable) querier.getQueryInfo().getTable();
        NbQueryInfo join = getJoinQueryInfo(table.getTableClass(), alias);
        return leftJoin(queryInfo, parseJoinOn(alias, join));
    }

    private NbQueryInfo getJoinQueryInfo(Class<?> clazz, String alias) {
        NbQueryInfo join = queryBuilder.buildQueryInfo(clazz, alias);
        Assert.isFalse(CollUtil.contains(tableAliasMap.values(), i -> i.equals(alias)),
                "该查询构造器中已存在别名:{},class:{}", alias, clazz);
        tableAliasMap.put(join.getTableName(), alias);
        return join;
    }

    /**
     * 自动解析join on (待优化)
     * 1. a->left join-> b 外键再->b
     *
     * @param alias
     * @param join
     * @return
     */
    private String parseJoinOn(String alias, NbQueryInfo join) {
        //获取外键字段
        INbField one = CollUtil.findOne(this.queryInfo.getFields(), i -> i.isFk() &&
                i.getJoinTable() != null &&
                StrUtil.equals(i.getJoinTable().getTableName(), join.getTableName()));
        if (one == null) {
            //尝试从关联表中查找关联当前表的字段
            one = CollUtil.findOne(join.getFields(), i -> i.getFieldType() == NbFieldType.FK &&
                    i.getJoinTable()
                            .getTableName().equals(queryInfo.getTableName()));

            if (one != null) {
                String masterAlias = tableAliasMap.get(this.queryInfo.getTableName());
                return StrUtil.format("{}={}", SqlUtils.withAlias(alias, getQueryInfo().getPkName()), SqlUtils.withAlias(masterAlias, one.getName()));
            }
        }
        String on = null;
        //连表情况
        if (one == null && CollUtil.isNotEmpty(joins)) {
            for (JoinTableRule joinTableRule : joins) {
                //递归解析, todo 判断循环递归
                on = parseJoinOn(joinTableRule.getQueryInfo().getAlias(), joinTableRule.getQueryInfo());
                if (StrUtil.isNotBlank(on)) {
                    break;
                }
            }
        }
        if (one != null) {
            String masterAlias = tableAliasMap.get(this.queryInfo.getTableName());
            final String fieldName = one.getName();
            on = StrUtil.format("{}={}", SqlUtils.withAlias(alias, join.getPkName()), SqlUtils.withAlias(masterAlias, fieldName));
        }

        Assert.notBlank(on, "表单{},未找到关联表单:{}的外键字段", this.queryInfo.getTableName(), join.getTableName());
        return on;
    }

    public NbQuerier<T> leftJoin(NbQueryInfo queryInfo, String on, boolean lateral) {
        this.joins.add(new JoinTableRule(queryInfo, JoinType.LEFT_JOIN, on, lateral));
        this.tableAliasMap.put(queryInfo.getTableName(), queryInfo.getAlias());
        if (queryInfo.isSub()) {
            this.queryFields.puts(queryInfo.getQueryFields());
        } else {
            List<QueryField> fields = queryInfo
                    .getFields()
                    .stream().map(i -> new QueryField(i.getQueryType(), i.getName(),
                            queryInfo.getTableName(), null, i.getRedirect()))
                    .collect(Collectors.toList());
            this.queryFields.puts(fields);
        }
        return this;
    }

    /**
     * 连表子查询
     *
     * @param querier
     * @param alias
     * @param on
     * @return
     */
    @Override
    public NbQuerier<T> leftJoin(NbQuerier<?> querier, String alias, String on, boolean lateral) {
        String sql = this.buildConnect(querier);
        //添加表
        NbQueryInfo info = queryBuilder.buildQueryInfo(querier, alias);
        //字段
        info.setSql(sql);
        info.setAlias(alias);
        info.setQueryType(QueryType.SUB);
        String virtualTableName = String.format(NbQuerierCons.SUB_SQL, alias);
        //字段的设置
        //理论上应该复制参数 不然对原来的字段
        info.setQueryFields(
                new QueryFields(querier
                        .getQueryFields()
                        .fields()
                        .stream()
                        .map(i -> new QueryField(i.getType(), i.getName(), i.getTableName(), virtualTableName, null))
                        .collect(Collectors.toList()))
        );
        //虚拟表名
        tableAliasMap.put(virtualTableName, alias);
        //复制表别名

        Set<String> tableSet = querier.getQueryFields()
                .fields()
                .stream()
                .map(QueryField::getTableName)
                .collect(Collectors.toSet());
        for (String s : tableSet) {
            tableAliasMap.put(s, alias);
        }
        NbQueryInfo q = querier.getQueryInfo();
        VirtualTable virtualTable = new VirtualTable(virtualTableName,
                q.getTableName(),
                q.getPk(),
                ListUtil.empty());
        info.setVirtualTable(virtualTable);
        return leftJoin(info, on, lateral);
    }

    /*----------------------------join end------------------------------*/

    @Override
    public NbQuerier<T> addRowNum(String partitionBy, String orderField, String sort) {
        //todo 待实现
        return this;
    }

    @Override
    public NbQuerier<T> field(String fieldStr) {
        //字段设置
        this.queryInfo.setSetField(fieldStr);
        return this;
    }


    @Override
    public NbQuerier<T> order(NbOrderType sort, @Nullable NullsPosition nullsPosition, String... fields) {
        if (ArrayUtil.isEmpty(fields)) {
            return this;
        }
        for (String field : fields) {
            //判断字段是否存在
            QueryItem queryItem = getColumnName(field, true);
            INbQueryType type = queryItem.getQueryField().getType();
            Assert.isTrue(type.orderAble(), "字段:{},类型:{},暂时不支持排序", field, type.getTitle());
            ArrayList<ISqlSegment> list = ListUtil.toList(SqlKeyword.ORDER_BY, queryItem, sort.getKeyword());
            if (nullsPosition != null) {
                list.add(SqlKeyword.NULLS);
                list.add(nullsPosition);
            }
            expression.add(list.toArray(new ISqlSegment[]{}));
        }
        return this;
    }


    @Override
    public <F> List<F> select(Class<F> clazz) {
        Assert.isTrue(clazz != null, "clazz 为空，请使用selectMap方法");
        //反射tableClass实例返回
        List<LinkedHashMap<String, Object>> linkedHashMaps = this.selectMap();
        //结果处理->直接map-> 对象
        return linkedHashMaps.stream().map(map -> mapToBean(map, clazz)).collect(Collectors.toList());
    }

    @Override
    public T mapToBean(Map<String, Object> map) {
        Assert.notNull(resultClass, "查询对象结果的class不能为空");
        return mapToBean(map, resultClass);
    }

    /**
     * map 封装成对象返回
     *
     * @param map
     * @return
     */
    private <F> F mapToBean(Map<String, Object> map, Class<F> clazz) {
        F instance = null;
        if (map == null) {
            return null;
        }
        mapFieldAlias(map, false);
        boolean isCustomEntity = BaseCustomEntity.class.isAssignableFrom(clazz);
        try {
            if (isCustomEntity) {
                instance = CustomEntityBuilder.newInstance(clazz, map);
                //自定义实体类
            } else {
                instance = clazz.getDeclaredConstructor().newInstance();
                instance = BeanUtil.fillBeanWithMap(map, instance, true, true);
            }
        } catch (InstantiationException | IllegalAccessException | InvocationTargetException |
                 NoSuchMethodException e) {
            log.error("反射对象失败:", e);
            //(提示:如果有@NotNull的对象需要加@NoArgsConstructor)
            throw new NbSQLMessageException("class:{}反射查询对象出现错误", clazz.getSimpleName());
        }
        return instance;
    }

    /**
     * 查询通用map
     *
     * @return
     */
    @Override
    public List<LinkedHashMap<String, Object>> selectMap() {
        INbExecutor executor = newExecutor();
        Assert.notNull(executor, "该构造器不能查询");
        String sql = buildQuerySql();
        List<LinkedHashMap<String, Object>> resultList = executor.doQuery(getDsName(), sql, this.getParams(), ResultType.LIST);

        //自动类型转换
        if (CollUtil.isNotEmpty(resultList)) {
            resultConsumer.doMapList(resultList);
        }
        return resultList;
    }

    @Override
    public String buildFuncSql(String funcName) {
        return sqlBuilder.buildFuncSql(this, funcName, false);
    }

    @Override
    public String buildQuerySql() {
        return sqlBuilder.buildQuerySql(this);
    }

    @Override
    public List<T> select() {
        List<T> list = this.select(this.resultClass);
        if (CollUtil.isNotEmpty(list)) {
            resultConsumer.doBeanList(list);
        }
        return list;
    }


    @Override
    public int update(boolean isDelete) {
        INbExecutor executor = newExecutor();
        Assert.notNull(updateData, "没有更新数据");
        Map<String, Object> value = new HashMap<>();
        updateData.getUpdateDataMap().forEach((k, v) -> value.put(k.getName(), v));
        if (!isDelete) {
            this.interceptorChain.doBeforeUpdate(getQueryInfo(), ListUtil.toList(value));
        }
        String sql = sqlBuilder.buildUpdateSql(this, updateData);
        return this.executeUpdate(sql, executor);
    }

    @Override
    public int delete() {
        if (queryInfo.isSoftDelete()) {
            UpdateData ud = safeGetUpdateData();
            ud.clear();
            ud.put(queryInfo.getDeleteField(), LocalDateTime.now());
            return update(true);
        }

        String sql = sqlBuilder.buildDeleteSql(this);
        return newExecutor().doUpdate(getDsName(), sql, getParams());
    }


    public INbExecutor newExecutor() {
        INbTransaction executeTx = null;
        if (batch != null) {
            executeTx = batch.getTx(configuration);
        } else if (this.tx != null) {
            executeTx = this.tx;
        } else {
            executeTx = connection == null ? configuration.newTx() : configuration.newTx(connection);
        }
        return configuration.newNbExecutor(executeTx);
    }


    @Override
    public NbQuerier<T> set(String field, Object value) {
        String tableName = queryInfo.getTableName();
        String uF = StrUtil.toUnderlineCase(field);
        INbField baseNbField = queryInfo.findField(uF);
        if (baseNbField == null) {
            return this;
        }
        Assert.notNull(baseNbField, "无法更新字段:{},原因:表:{},不存在该字段", uF, tableName);
        this.safeGetUpdateData().put(baseNbField, value);
        return this;
    }

    @Override
    @Nullable
    public QueryField findQueryField(String name, String table) {
        return this.queryFields.find(name, table);
    }

    @Override
    @Nullable
    public QueryField findQueryField(String name) {
        Collection<QueryField> fields = this.queryFields.findByName(name);
        if (CollUtil.isNotEmpty(fields)) {
            return fields.stream().iterator().next();
        }
        return null;
    }

    /**
     * 保存数据
     *
     * @param data
     * @return
     */
    @Override
    public int save(Map<String, Object> data) {
        if (MapUtil.isEmpty(data)) {
            return 0;
        }
        interceptorChain.doBeforeSave(this.queryInfo, ListUtil.toList(data));
        String sql = sqlBuilder.buildSaveSql(this, new UnderlineCaseMap<>(data));
        INbExecutor executor = newExecutor();
        return executeUpdate(sql, executor);
    }

    private int executeUpdate(String sql, INbExecutor executor) {
        return executor.doUpdate(new DoUpdateOptions()
                .setDsName(getDsName())
                .setSql(sql)
                .setBatch(batch)
                .setParams(getParams()));
    }


    @Override
    public int saveBatchMap(List<Map<String, Object>> dataList) {
        if (CollUtil.isEmpty(dataList)) {
            return 0;
        }
        interceptorChain.doBeforeSave(this.queryInfo, dataList);
        INbExecutor executor = newExecutor();
        return executor.insertBatch(this, dataList, this.getParams());
    }

    @Override
    public <F> F func(String funcName, ResultType type, boolean limit) {
        String sql = sqlBuilder.buildFuncSql(this, funcName, limit);
        INbExecutor executor = newExecutor();
        return executor.doQuery(getDsName(), sql, getParams(), type);
    }

    @Override
    public int count(boolean page) {
        String countSql = "count(*)";
        if (expression.getGroupBy().isEmpty()) {
            return func(countSql, ResultType.INT, page);
        } else {
            //存在groupBy
            String sql = sqlBuilder.buildQuerySql(this, "1", page);
            String buildSql = StrUtil.format("SELECT {} FROM ({}) tc", countSql, sql);
            INbExecutor executor = newExecutor();
            return executor.doQuery(getDsName(), buildSql, getParams(), ResultType.INT);
        }
    }

    @Override
    public NbQuerier<T> conditionCount(ILaterQuerier laterQuerier, String alias, boolean coalesce) {
        ISqlSegment sqlSegment = BuildInFunc.count(() -> laterQuerier.getSqlSegment(this));
        if (coalesce) {
            sqlSegment = BuildInFunc.coalesce(this, sqlSegment, QueryValParam.intVal(0));
        }
        addConditionAgg(sqlSegment, alias);
        //query 计算
        return this;
    }

    public NbQuerier<T> addConditionAgg(ISqlSegment count, String alias) {
        safeGetConditionAggList().add(new ConditionAgg(count, alias));
        return this;
    }

    @Override
    public double sum(String field) {
        QueryItem queryItem = new QueryItem(field);
        QueryField one = BuildUtils.findOne(getQueryFields(), queryItem, queryInfo.getTableName(), tableAliasMap);
        Assert.notNull(one, "求和字段:[{}]不存在", field);
        return func(StrUtil.format("sum({})", queryItem.getExpr()), ResultType.DOUBLE, false);
    }

    @Override
    public boolean exist() {
        String sql = sqlBuilder.buildQuerySql(this, "1", false);
        sql += " limit 1";
        INbExecutor executor = newExecutor();
        List<?> list = executor.doQuery(getDsName(), sql, getParams(), ResultType.LIST);
        return CollUtil.isNotEmpty(list);
    }

    @Override
    public List<LinkedHashMap<String, Object>> selectMapList() {
        interceptorChain.doQueryBefore(getQueryInfo());
        INbExecutor executor = newExecutor();
        return executor.doQuery(getDsName(), buildQuerySql(), this.getParams(), ResultType.LIST);
    }


    @Override
    public boolean hasWhere() {
        return CollUtil.isNotEmpty(expression.getNormal());
    }

    @Override
    public NbQuerier<T> alias(String alias) {
        tableAliasMap.put(queryInfo.getTableName(), alias);
        queryInfo.setAlias(alias);
        return this;
    }

    /**
     * 构建并连接参数
     *
     * @param querier
     * @return
     */
    @Override
    public String buildConnect(NbQuerier<?> querier) {
        String sql = querier.buildQuerySql();
        return connectParam(querier, sql);
    }

    /**
     * 连接参数
     *
     * @param querier
     * @param sql
     * @return
     */
    public String connectParam(NbQuerier<?> querier, String sql) {
        List<QueryValParam> params = querier.getParams();
        if (CollUtil.isNotEmpty(params)) {
            for (int i = 0; i < params.size(); i++) {
                QueryValParam param = params.get(i);
                String key = String.format(NbQuerierCons.SQL_PARAM_FORMAT, i);
                String replaceTo = formatVariable(param.getType(), param.getValue(), false);
                sql = StrUtil.replace(sql, key, replaceTo);
            }
        }
        return sql;
    }


    @Override
    public NbQuerier<T> withMany(String fieldName, NbQuerier<?> querier) {
        Assert.isTrue(config.getDbType() == DbType.POSTGRE_SQL || config.getDbType() == DbType.PG_NG, "目前只支持postgresql");
        if (querier != null) {
            //querier 添加条件
            //querier.where();
            String subSql = this.buildConnect(querier);
            String sql = StrUtil.format("(select json_agg(tem) from ({}) tem) as {}", subSql, fieldName);
            additionField(sql);
        }
        return this;
    }

    @Override
    public NbQuerier<T> withOne(String fieldName, NbQuerier<?> querier) {
        Assert.isTrue(config.getDbType() == DbType.POSTGRE_SQL
                || config.getDbType() == DbType.PG_NG, "目前只支持postgresql");
        if (querier != null) {
            //querier 添加条件
            //querier.where();
            String subSql = this.buildConnect(querier);
            QueryItem queryItem = new QueryItem(fieldName);
            String sql = StrUtil.format("(select row_to_json(tem) from ({}) tem) as {}", subSql, queryItem.getName());

            additionField(sql);
        }
        return this;
    }

    /**
     * 查询一对多数量
     *
     * @param fieldName
     * @param querier
     * @return
     */
    @Override
    public NbQuerier<T> withManyCount(String fieldName, NbQuerier<?> querier) {
        switch (config.getDbType()) {
            case MYSQL:
            case POSTGRE_SQL:
                if (querier != null) {
                    querier.field("1");
                    String subSql = buildConnect(querier);
                    String sql = StrUtil.format("(select count(1) from ({}) t) as {}", subSql, StrUtil.toUnderlineCase(fieldName));
                    additionField(sql);
                }
                break;
            default:
                throw new NbSQLException("暂不支持withManyCount方法");
        }
        return this;
    }


    @Override
    public NbQuerier<T> group(String... columns) {
        if (ArrayUtil.isEmpty(columns)) {
            return this;
        }
        List<ISqlSegment> segments = new ArrayList<>();
        segments.add(GROUP_BY);
        for (String column : columns) {
            segments.add(getColumnName(column, false));
        }
        ISqlSegment[] iSqlSegments = segments.toArray(new ISqlSegment[0]);
        expression.add(iSqlSegments);
        return this;
    }

    @Override
    public NbQuerier<T> autoOrder() {
        List<OrderInfo> orderInfos = queryInfo.getTable().getOrderInfos();
        if (CollUtil.isNotEmpty(orderInfos)) {
            for (OrderInfo orderInfo : orderInfos) {
                String fieldName = orderInfo.getNbField().getName();
                String tableName = orderInfo.getNbField().getTableName();
                if (StrUtil.isNotBlank(tableName)) {
                    fieldName = SqlUtils.withAlias(tableAliasMap.get(tableName), fieldName);
                }
                this.order(orderInfo.getOrderType(), fieldName);
            }
        } else {
            //查找排序字段
            INbField orderField = CollUtil.findOne(getQueryInfo().getFields(),
                    i -> i.getFieldType() == NbFieldType.ORDER);
            if (orderField != null) {
                this.order(NbOrderType.DESC, orderField.getName());
                return this;
            }
            //默认主键排序
            IdType type = queryInfo.getPk().getType();
            if (type == IdType.AUTO || type == IdType.ASSIGN_ID) {
                desc(SqlUtils.withAlias(queryInfo.getAlias(), getPkName()));
            }
        }
        return this;
    }

    @Override
    public NbQuerier<T> updateExpr(String expr) {
        safeGetUpdateData().addExpr(expr);
        return this;
    }

    @Override
    public NbQuerier<T> updateExpr(String sql, Object... params) {
        QueryValParam[] queryValParam = new QueryValParam[params.length];
        for (int i = 0; i < params.length; i++) {
            Object param = params[i];
            if (param instanceof String) {
                queryValParam[i] = QueryValParam.stringVal((String) param);
            } else if (param instanceof Long) {
                queryValParam[i] = QueryValParam.longVal((Long) param);
            } else {
                throw new NbSQLMessageException("未处理设置该类型");
            }
        }
        return updateExpr(sql, queryValParam);
    }

    public String parseSqlParam(String sql, QueryValParam... params) {
        if (params.length == 0) {
            return sql;
        }
        int length = sql.length();
        final StringBuilder builder = new StringBuilder(sql.length() + params.length * NbQuerierCons.SQL_PARAM_FORMAT.length());
        int preIndex = 0;
        for (QueryValParam param : params) {
            String paramStr = paramBuilder.formatVariable(param.getType(), param.getValue(), true);
            int index = sql.indexOf(StringPool.QUESTION_MARK, preIndex);
            Assert.isTrue(index > 0, "sql:{}占位符不够:{}", params.length);
            builder.append(sql.subSequence(preIndex, index));
            builder.append(paramStr);
            //+ ?.len
            preIndex = index + 1;
        }
        if (preIndex < length) {
            // 结尾部分
            builder.append(sql.subSequence(preIndex, length));
        }
        return builder.toString();
    }

    @Override
    public NbQuerier<T> tableSchema(String schema) {
        this.queryInfo.setSchema(schema);
        return this;
    }

    @Override
    public String getDsName() {
        return getQueryInfo().getDs();
    }

    @Override
    public NbQuerier<T> lock() {
        return last("FOR UPDATE");
    }

    @Override
    public DbType getDbType() {
        return config.getDbType();
    }


    @Override
    public QueryField findField(String fieldName) {
        QueryField one = BuildUtils.findOne(queryFields, new QueryItem(fieldName),
                queryInfo.getTableName(), tableAliasMap);
        Assert.notNull(one, "表:{},不存在字段:{}", getTableName(), fieldName);
        return one;

    }

    /**
     * 查询重定向
     *
     * @param column
     * @param isOrder
     * @return
     */
    private QueryItem getColumnName(String column, boolean isOrder) {
        QueryItem queryItem = new QueryItem(column);
        QueryField queryField = BuildUtils.findOne(queryFields, queryItem, queryInfo.getTableName(), tableAliasMap);
        Assert.notNull(queryField, "表单:{}操作字段:{}不存在", queryInfo.getTableName(), column);
        queryItem.setQueryField(queryField);
        String orderRedirect = queryField.getOrderRedirect();
        if (isOrder && StrUtil.isNotBlank(orderRedirect)) {
            queryItem.setName(orderRedirect);
            return queryItem;
        }

        if (StrUtil.isNotBlank(queryField.getRedirect())) {
            queryItem.setName(queryField.getRedirect());
        }
        String alias = queryField.getTableName();
        if (StrUtil.isNotBlank(alias) && tableAliasMap.containsKey(alias)) {
            queryItem.setTableAlias(tableAliasMap.get(alias));
        }
        return queryItem;
    }

    public NbQuerier<T> additionFields(List<String> fields) {
        List<String> additionFields = getOrCreateAdditionFields();
        additionFields.addAll(fields);
        return this;
    }

    private List<String> getOrCreateAdditionFields() {
        List<String> additionFields = queryInfo.getAdditionFields();
        if (additionFields == null) {
            additionFields = new ArrayList<>();
            queryInfo.setAdditionFields(additionFields);
        }
        return additionFields;
    }

    /**
     * 添加附加字段
     *
     * @param field
     */
    @Override
    public NbQuerier<T> additionField(String field) {
        //todo sql 注入检测
        List<String> additionFields = getOrCreateAdditionFields();
        if (!additionFields.contains(field)) {
            additionFields.add(field);
        }
        return this;
    }

    /**
     * 查找一个对象,自动构建类
     *
     * @return
     */
    public <F> F find(Class<F> clazz) {
        LinkedHashMap<String, Object> map = findMap(true);
        if (map == null) {
            return null;
        }
        Assert.notNull(clazz, "find 方法class不能为空");
        return mapToBean(map, clazz);
    }

    @Override
    public T find() {
        T t = find(resultClass);
        List<Consumer<T>> consumers = resultConsumer.getDoAfterForItemConsumers();
        if (t != null && CollUtil.isNotEmpty(consumers)) {
            for (Consumer<T> consumer : consumers) {
                consumer.accept(t);
            }
        }
        return t;
    }

    @Override
    public LinkedHashMap<String, Object> selectOne(boolean assertOne) {
        List<LinkedHashMap<String, Object>> maps = this.selectMap();
        /*String sql = buildQuerySql();
        List<LinkedHashMap<String, Object>> maps = executor.doQuery(sql, this.getParams(), ResultType.LIST);*/
        if (assertOne) {
            Assert.isTrue(maps.size() <= 1, "查询结果超过一个");
        }
        LinkedHashMap<String, Object> map = null;
        if (CollUtil.isNotEmpty(maps)) {
            resultConsumer.doMapList(maps);
            map = maps.get(0);
            if (map != null) {
                resultConsumer.doMap(map);
            }
        }
        return map;
    }

    @Override
    public NbQuerier<T> doAfterForMapList(Consumer<List<LinkedHashMap<String, Object>>> doAfter) {
        resultConsumer.doAfterForMapList(doAfter);
        return this;
    }

    /**
     * select 之后对item操作
     *
     * @return
     */
    @Override
    public NbQuerier<T> doAfterForItem(Consumer<T> doAfterForItem) {
        resultConsumer.doAfterForItem(doAfterForItem);
        return this;

    }

    @Override
    public NbQuerier<T> doAfterForItemMap(Consumer<LinkedHashMap<String, Object>> doAfterForItemMap) {
        resultConsumer.doAfterForItemMap(doAfterForItemMap);
        return this;
    }

    /**
     * select 执行后都对数据处理
     *
     * @param doAfter
     * @return
     */
    @Override
    public NbQuerier<T> doAfterForList(Consumer<List<T>> doAfter) {
        resultConsumer.doAfterForList(doAfter);
        return this;
    }

    @Override
    public String getSqlSegment() {
        String sql = expression.getSqlSegment();
        if (StrUtil.isNotBlank(lastSql)) {
            sql += lastSql;
        }
        return sql;
    }

    @Override
    public Map<String, Object> beanToMap(T bean) {
        Map<String, Object> map = BeanUtil.beanToMap(bean, true, false);
        if (queryInfo.isClass()) {
            mapFieldAlias(map, true);
            //判断是否有改key
            Set<String> keys = map.keySet();
            List<String> removes = new ArrayList<>();
            for (String key : keys) {
                if (findQueryField(key) == null) {
                    removes.add(key);
                }
            }
            if (CollUtil.isNotEmpty(removes)) {
                removes.forEach(map::remove);
            }

        }
        return map;
    }


}
