package cn.hperfect.nbquerier.core.metedata.custom;

import cn.hperfect.nbquerier.annotation.NbField;
import cn.hperfect.nbquerier.core.json.serializer.CustomEntityJsonSerializer;
import cn.hperfect.nbquerier.toolkit.map.UnderlineCaseMap;
import cn.hutool.core.bean.BeanDesc;
import cn.hutool.core.bean.BeanException;
import cn.hutool.core.bean.BeanUtil;
import cn.hutool.core.bean.PropDesc;
import cn.hutool.core.bean.copier.CopyOptions;
import cn.hutool.core.convert.Convert;
import cn.hutool.core.getter.BasicTypeGetter;
import cn.hutool.core.map.CamelCaseMap;
import cn.hutool.core.util.StrUtil;
import com.fasterxml.jackson.databind.annotation.JsonSerialize;
import lombok.*;

import java.io.Serializable;
import java.lang.reflect.Type;
import java.math.BigDecimal;
import java.math.BigInteger;
import java.util.*;
import java.util.stream.Collectors;


/**
 * 自定义字段实体
 * 多余字段(实体中没有的字段)复制到 _customFieldData (驼峰方式存储)
 */
@Data
@JsonSerialize(using = CustomEntityJsonSerializer.class)
public abstract class BaseCustomEntity implements BasicTypeGetter<String>, Serializable, Map<String, Object> {

    public static final String CUSTOM_FIELD_DATA_KEY = "_customFieldData";

    /**
     * 只处理自定义字段数据
     */
    @NbField(exist = false)
    public transient final Map<String, Object> _customFieldData;
    @NbField(exist = false)
    private transient final BeanDesc _beanDesc;
    @NbField(exist = false)
    @Getter(AccessLevel.NONE)
    private transient Set<Map.Entry<String, Object>> _entrySet;

    /**
     * 移除实体key
     */
    @NbField(exist = false)
    public transient Set<String> _removeKeySet = new HashSet<>();

    @NbField(exist = false)
    private transient int _size;

    protected BaseCustomEntity() {
        this._customFieldData = new HashMap<>();
        this._beanDesc = BeanUtil.getBeanDesc(this.getClass());
        this._size = getProps().size();
    }

    protected BaseCustomEntity(RemoveValueMapProvider provider) {
        this();
        initByProvider(provider);
        //字节码增强
    }

    protected void initByMap(Map<String, Object> map) {
        initByProvider(new RemoveValueMapProvider(new CamelCaseMap<>(map)));
    }

    protected void initByProvider(RemoveValueMapProvider provider) {
        BeanUtil.fillBeanWithMap(provider, this, false, new CopyOptions());
        _size += provider.size();
        //剩余的移到自定义字段数据中
        _customFieldData.putAll(provider);
    }

    /**
     * 有数据重写该方法
     * 传入map 构建实体
     * 1. 转成驼峰
     * 2. 填充到当前实体
     *
     * @param data
     */
    protected BaseCustomEntity(Map<String, Object> data) {
        this(data instanceof RemoveValueMapProvider ? (RemoveValueMapProvider) data : new RemoveValueMapProvider(new CamelCaseMap<>(data)));
    }


    @SneakyThrows
    public final void set(String fieldName, Object value) {
        //todo size
        PropDesc prop = _beanDesc.getProp(StrUtil.toCamelCase(fieldName));
        if (prop != null) {
            //设置值,类型装换
            Type valueType = prop.getFieldType();
            prop.setValue(this, Convert.convertWithCheck(valueType, value, null, false));
        } else {
            _customFieldData.put(fieldName, value);
        }
    }

    public final <T> T get(String fieldName) throws BeanException {
        return get(fieldName, false);
    }

    @SneakyThrows
    @SuppressWarnings("unchecked")
    public final <T> T get(String fieldName, boolean toCamelCase) throws BeanException {
        if (toCamelCase) {
            fieldName = StrUtil.toCamelCase(fieldName);
        }
        final PropDesc prop = BeanUtil.getBeanDesc(this.getClass()).getProp(fieldName);
        if (prop != null) {
            return (T) prop.getGetter().invoke(this);
        }
        return (T) _customFieldData.get(fieldName);
    }

    @SuppressWarnings("unchecked")
    public final Map<String, Object> getDataMap() {
        Map<String, Object> map = BeanUtil.beanToMap(this, false, true);
        Map<String, Object> newMap = (Map<String, Object>) map.remove("_custom_field_data");
        if (newMap != null) {
            map.putAll(newMap);
        }
        return map;
    }


    /**
     * 只读 获取下划线的map
     *
     * @return
     */
    public Map<String, Object> getReadOnlyDataUnderlineCaseMap() {
        return Collections.unmodifiableMap(new UnderlineCaseMap<>(this));
    }


    @Override
    public <E extends Enum<E>> E getEnum(Class<E> clazz, String key) {
        throw new RuntimeException("为实现呢");
    }

    /**
     * @return
     */
    public final Set<String> keySet() {
        //todo 迭代器模式
        Set<String> keySet = _beanDesc.getPropMap(false).keySet();
        keySet.addAll(_customFieldData.keySet());
        return Collections.unmodifiableSet(keySet);
    }


    @Override
    public Object getObj(String key) {
        return get(key);
    }

    @Override
    public String getStr(String key) {
        return Convert.toStr(get(key));
    }

    @Override
    public Integer getInt(String key) {
        return Convert.toInt(get(key));
    }

    @Override
    public Short getShort(String key) {
        return Convert.toShort(get(key));
    }

    @Override
    public Boolean getBool(String key) {
        return Convert.toBool(get(key));
    }

    @Override
    public Long getLong(String key) {
        return Convert.toLong(get(key));
    }

    @Override
    public Character getChar(String key) {
        return Convert.toChar(get(key));
    }

    @Override
    public Float getFloat(String key) {
        return Convert.toFloat(get(key));
    }

    @Override
    public Double getDouble(String key) {
        return Convert.toDouble(get(key));
    }

    @Override
    public Byte getByte(String key) {
        return Convert.toByte(get(key));
    }

    @Override
    public BigDecimal getBigDecimal(String key) {
        return Convert.toBigDecimal(get(key));
    }

    @Override
    public BigInteger getBigInteger(String key) {
        return Convert.toBigInteger(get(key));
    }

    /**
     * 获取map对象
     *
     * @param key
     * @return
     */
    public Map<String, Object> getMap(String key) {
        return Convert.toMap(String.class, Object.class, get(key));
    }

    public <T> T getBean(String name, Class<T> clazz) {
        return BeanUtil.toBean(get(name), clazz);
    }


    @Override
    public Date getDate(String key) {
        return Convert.toDate(get(key));
    }

//----------- map 方法实现

    /**
     * @return
     * @Deprecated size 不正确
     */

    @Override
    @Deprecated
    public int size() {
        return _size;
    }

    @Override
    public boolean isEmpty() {
        return false;
    }

    @Override
    public boolean containsKey(Object key) {
        throw new RuntimeException("为实现containsKey");
    }

    @Override
    public boolean containsValue(Object value) {
        throw new RuntimeException("为实现containsValue");
    }

    @Override
    public Object get(Object key) {
        return get(Convert.toStr(key), true);
    }

    @Override
    public Object put(String key, Object value) {
        set(key, value);
        return value;
    }

    private void resize() {

    }

    @SneakyThrows
    @Override
    public Object remove(Object key) {
        String keyStr = StrUtil.toCamelCase((String) key);
        PropDesc prop = _beanDesc.getProp(keyStr);
        if (prop != null) {
            Object value = prop.getGetter().invoke(this);
            if (!_removeKeySet.contains(keyStr)) {
                _removeKeySet.add(keyStr);
                _size--;
            }
            if (value != null) {
                //设置为空 就是移除
                prop.getSetter().invoke(this, (String) null);
            }
            return value;
        } else {
            return _customFieldData.remove(keyStr);
        }
    }

    @Override
    public void putAll(Map<? extends String, ?> m) {
        m.forEach(this::put);
    }

    @Override
    public void clear() {
        throw new RuntimeException("为实现clear");
    }

    @SneakyThrows
    @Override
    public Collection<Object> values() {
        Set<PropDesc> props = getProps();
        List<Object> values = new ArrayList<>(_customFieldData.values());
        for (PropDesc prop : props) {
            values.add(prop.getGetter().invoke(this));
        }
        return values;
    }

    private Set<PropDesc> getProps() {
        return _beanDesc.getProps()
                .stream()
                .filter(i -> !StrUtil.startWith(i.getFieldName(), "_"))
//                .filter(i -> CollUtil.contains(_keySet, i.getFieldName()))
                .collect(Collectors.toSet());
    }

    @SneakyThrows
    @Override
    public Set<Entry<String, Object>> entrySet() {
//        Set<Map.Entry<String, Object>> es;
//        return (es = _entrySet) == null ? (_entrySet = new EntrySet()) : es;
        HashSet<Entry<String, Object>> es = new HashSet<>(_customFieldData.entrySet());
        for (PropDesc prop : getProps()) {
            Object value = prop.getGetter().invoke(this);
            Node node = new Node(prop.getFieldName(), value, null);
            es.add(node);
        }

        return es;
    }


    final class EntrySet extends AbstractSet<Map.Entry<String, Object>> {


        @Override
        public Iterator<Entry<String, Object>> iterator() {
            return new EntryIterator();
        }


        @Override
        public int size() {
            return BaseCustomEntity.this.size();
        }

    }

    final class EntryIterator extends BaseCustomEntity.HashIterator
            implements Iterator<Map.Entry<String, Object>> {
        public final Map.Entry<String, Object> next() {
            return nextNode();
        }
    }


    /* ------------------------------------------------------------ */
    // iterators

    abstract class HashIterator {
        BaseCustomEntity.Node next;        // next entry to return
        BaseCustomEntity.Node current;     // current entry
        int expectedModCount;  // for fast-fail
        int index;             // current slot
        final int size;

        HashIterator() {
            current = new BaseCustomEntity.Node("535", "1", new Node("1", "4", null));
            next = current.getNext();
            size = BaseCustomEntity.this.size();
        }

        public final boolean hasNext() {
            return index < this.size;
        }

        final BaseCustomEntity.Node nextNode() {
            next = current.getNext();
            return next;
        }

        public final void remove() {
            BaseCustomEntity.Node p = current;
            if (p == null)
                throw new IllegalStateException();
         /*   if (modCount != expectedModCount)
                throw new ConcurrentModificationException();
            current = null;
            K key = p.key;
            removeNode(hash(key), key, null, false, false);
            expectedModCount = modCount;*/
        }
    }


    /**
     * Basic hash bin node, used for most entries.  (See below for
     * TreeNode subclass, and in LinkedHashMap for its Entry subclass.)
     */
    @Data
    static class Node implements Map.Entry<String, Object> {

        final String key;
        Object value;
        BaseCustomEntity.Node next;

        Node(String key, Object value, BaseCustomEntity.Node next) {
            this.key = key;
            this.value = value;
            this.next = next;
        }

        public Object setValue(Object value) {
            this.value = value;
            return value;
        }

        public String toString() {
            return key + "=" + value;
        }
    }


    @Data
    @EqualsAndHashCode
    static class TestEntity extends BaseCustomEntity {
        private String name;
        private String name2;
        private String nameTest;
    }

}
