package cn.hperfect.nbquerier.core.components.type;


import cn.hperfect.nbquerier.core.components.convertor.*;
import cn.hperfect.nbquerier.core.conditions.ISqlSegment;
import cn.hperfect.nbquerier.core.conditions.segments.MergeSegments;
import cn.hperfect.nbquerier.core.querier.NbQuerier;
import cn.hperfect.nbquerier.core.type.JsonNbType;
import cn.hperfect.nbquerier.enums.DbDataType;
import cn.hperfect.nbquerier.enums.INbEnum;
import cn.hperfect.nbquerier.enums.QueryRuleEnum;
import cn.hperfect.nbquerier.exceptions.NbSQLException;
import cn.hperfect.nbquerier.exceptions.NbSQLMessageException;
import cn.hperfect.nbquerier.exceptions.TypeConvertException;
import cn.hutool.core.bean.BeanUtil;
import cn.hutool.core.convert.Convert;
import cn.hutool.core.lang.Assert;
import cn.hutool.core.util.StrUtil;
import cn.hutool.json.JSON;
import cn.hutool.json.JSONArray;
import lombok.Getter;
import lombok.extern.slf4j.Slf4j;

import java.lang.reflect.ParameterizedType;
import java.lang.reflect.Type;
import java.time.LocalDateTime;
import java.util.Collection;
import java.util.List;
import java.util.Map;


/**
 * 数据库查询类型
 *
 * @author huanxi
 * @version 1.0
 * @date 2021/3/1 8:36 下午
 */
@Slf4j
public enum NbQueryType implements INbQueryType {

    /**
     * 字符串
     */
    TEXT(null, TextTypeConvertor.INSTANCE, "text", STRING_RULE),
    /**
     * 字符串数组
     */
    TEXT_LIST(null, TextListTypeConvertor.INSTANCE, "text[]", NbQueryType.LIST_RULE),
    LONG(null, LongTypeConvertor.INSTANCE, "bigint", NUM_RULE),
    LONG_LIST(null, LongListTypeConvertor.INSTANCE, "bigint[]", NbQueryType.LIST_RULE),
    INT(null, IntTypeConvertor.INSTANCE, "integer", NUM_RULE),
    INT_LIST(null, IntListTypeConvertor.INSTANCE, "integer[]", NbQueryType.LIST_RULE),
    DATE("日期", DateTypeConvertor.INSTANCE, "timestamp", NUM_RULE),
    DOUBLE(null, DoubleTypeConvertor.INSTANCE, "double precision", NUM_RULE),
    BOOL(null, BoolTypeConvertor.INSTANCE, "bool", BOOL_RULE),

    ENUM(null, EnumTypeConvertor.INSTANCE, "text", STRING_RULE),
    ENUM_LIST(null, TextListTypeConvertor.INSTANCE, "text[]", LIST_RULE),
    /**
     * I_ENUM
     */
    MAP_ENUM(null, MapEnumConvertor.INSTANCE, "text", STRING_RULE),

    MAP_ENUM_LIST(null, NoTypeConvertor.INSTANCE, "text[]", LIST_RULE),
    /**
     * 浮点自增
     */
    AUTO_INCR_DOUBLE(null, DoubleTypeConvertor.INSTANCE, null, NUM_RULE),
    /**
     * todo 这里默认(10,5)
     */
    DECIMAL("数字", DecimalConvertor.INSTANCE, "decimal(10,5)", NUM_RULE),

    FLOAT(null, FloatTypeConvertor.INSTANCE, null, NUM_RULE),
    UNKNOWN(null, FloatTypeConvertor.INSTANCE, null, null),

    /**
     * 函数字段
     */
    FUNC(null, FloatTypeConvertor.INSTANCE, null, null),
    ;

    @Getter
    private final ITypeConvertor convert;


    @Getter
    private final String dbTypeSql;
    /**
     * 查询规则
     */
    @Getter
    private final List<QueryRuleEnum> queryRules;
    @Getter
    private final String title;


    NbQueryType(String title, ITypeConvertor convert, String createSql, List<QueryRuleEnum> queryRules) {
        this.convert = convert;
        this.dbTypeSql = createSql;
        this.queryRules = queryRules;
        this.title = title;
    }


    /**
     * class 转换成 PowerTableTypeEnum
     *
     * @param fieldClazz
     * @return
     */
    public static INbQueryType convertFromClass(Class<?> fieldClazz, Type fieldType) {
        if (String.class.equals(fieldClazz)) {
            return TEXT;
        } else if (Long.class.equals(fieldClazz)) {
            return LONG;
        } else if (Integer.class.equals(fieldClazz)) {
            return INT;
        } else if (Double.class.equals(fieldClazz)) {
            return DOUBLE;
        } else if (Boolean.class.equals(fieldClazz)) {
            return BOOL;
        } else if (LocalDateTime.class.equals(fieldClazz)) {
            return DATE;
        } else if (Collection.class.isAssignableFrom(fieldClazz)) {
            if (fieldType instanceof ParameterizedType) {
                ParameterizedType pt = (ParameterizedType) fieldType;
                Type[] types = pt.getActualTypeArguments();
                if (types == null) {
                    return new JsonNbType();
                }
                Assert.isTrue(types.length == 1, () -> new NbSQLMessageException("list类型字段泛型只支持一个"));
                Class<?> clazz = (Class<?>) types[0];
                if (String.class.isAssignableFrom(clazz)) {
                    return TEXT_LIST;
                } else if (Long.class.equals(clazz)) {
                    return LONG_LIST;
                } else if (clazz.isEnum()) {
                    return TEXT_LIST;
                }
                if (BeanUtil.isBean(clazz)) {
                    return new JsonNbType(true);
                } else {
                    throw new NbSQLMessageException("list不支持类型:{}", clazz.getSimpleName());
                }
            }
            if (JSONArray.class.isAssignableFrom(fieldClazz)) {
                return new JsonNbType(true);
            }
            throw new NbSQLException("集合类型解析失败");
        } else if (Map.class.isAssignableFrom(fieldClazz)) {
            return new JsonNbType(false);
        } else if (BeanUtil.isBean(fieldClazz)) {
            return new JsonNbType(false);
        } else if (INbEnum.class.isAssignableFrom(fieldClazz)) {
            //枚举
            return MAP_ENUM;
        } else if (fieldClazz.isEnum()) {
            return TEXT;
        } else if (JSON.class.isAssignableFrom(fieldClazz)) {
            return new JsonNbType(null);
        }

        throw new NbSQLException("class类型:{},暂不支持", fieldClazz.getName());
    }

    /**
     * 获取类型的数组类型
     *
     * @return
     */
    @Override
    public NbQueryType getArrayType() {
        switch (this) {
            case ENUM:
            case TEXT:
                return TEXT_LIST;
            case LONG:
                return LONG_LIST;
            case INT:
                return INT_LIST;
            case MAP_ENUM:
                return MAP_ENUM_LIST;
            default:
                throw new NbSQLMessageException("类型:{},暂不支持数组查询", this.name());
        }

    }

    @Override
    public NbQueryType getArraySubType() {
        switch (this) {
            case TEXT_LIST:
                return TEXT;
            case LONG_LIST:
                return LONG;
            case INT_LIST:
                return INT;
            case MAP_ENUM_LIST:
                return MAP_ENUM;
            case ENUM_LIST:
                return ENUM;
            default:
                throw new NbSQLException("类型:{},暂不支持获取数组子类型", this);
        }

    }

    @Override
    public DbDataType getDbDataType() {
        switch (this) {
            case LONG_LIST:
            case TEXT_LIST:
            case INT_LIST:
            case MAP_ENUM_LIST:
            case ENUM_LIST:
                return DbDataType.ARRAY;
            default:
                return DbDataType.GENERAL;
        }
    }


    @Override
    @Deprecated
    public Object convert(Object value) throws TypeConvertException {
        return this.convert.convert(value);
    }


    @Override
    public void whereIn(NbQuerier<?> querier, MergeSegments expression, QueryRuleEnum ruleEnum, String name, Object value) {
        if (!isArray()) {
            defaultWhereIn(querier, expression, ruleEnum, name, value);
            return;
        }
        INbQueryType.super.whereIn(querier, expression, ruleEnum, name, value);
    }

    @Override
    public boolean orderAble() {
        return !isArray();
    }


    public void defaultWhereIn(NbQuerier<?> querier, MergeSegments expression, QueryRuleEnum ruleEnum, String name, Object value) {
        //可以是子查询
        if ((value instanceof ISqlSegment)) {
            expression.add(() -> name, ruleEnum, (ISqlSegment) value);
        } else {
            Assert.isTrue(value instanceof Collection, () -> new NbSQLMessageException("$in查询参数必须为数组"));
            Collection<?> listValue = Convert.toList(value);
            Assert.notEmpty(listValue, () -> new NbSQLMessageException("查询条件数组值不能为空,字段:{}", name));
            String paramKey = querier.formatVariable(this.getArrayType(), value, false);
            //name in (数组)
            expression.add(() -> name, ruleEnum, () -> paramKey);
        }
    }

    @Override
    public void whereLike(NbQuerier<?> querier, MergeSegments expression, String name, Object value) {
        Assert.isTrue(this == NbQueryType.TEXT, "类型:{},不支持模糊查询", this);
        assert value instanceof String : "模糊查询必须传入字符串";
        String valueStr = (String) value;
        valueStr = StrUtil.removePrefix(valueStr, "%");
        valueStr = StrUtil.removeSuffix(valueStr, "%");
        value = StrUtil.concat(true, "%", valueStr, "%");
        String paramKey = querier.formatVariable(this, value, false);
        expression.add(() -> name, QueryRuleEnum.LIKE, () -> paramKey);
    }


}
