package cn.hperfect.nbquerier.core.components.type;

import cn.hperfect.nbquerier.core.conditions.segments.MergeSegments;
import cn.hperfect.nbquerier.core.querier.NbQuerier;
import cn.hperfect.nbquerier.core.type.JsonNbType;
import cn.hperfect.nbquerier.enums.DbDataType;
import cn.hperfect.nbquerier.enums.DbType;
import cn.hperfect.nbquerier.enums.QueryRuleEnum;
import cn.hperfect.nbquerier.enums.SqlKeyword;
import cn.hperfect.nbquerier.exceptions.NbSQLException;
import cn.hperfect.nbquerier.exceptions.NbSQLMessageException;
import cn.hperfect.nbquerier.exceptions.TypeConvertException;
import cn.hperfect.nbquerier.toolkit.StringPool;
import cn.hutool.core.collection.CollUtil;
import cn.hutool.core.collection.ListUtil;
import cn.hutool.core.convert.Convert;
import cn.hutool.core.lang.Assert;
import cn.hutool.core.util.StrUtil;

import java.io.Serializable;
import java.util.Collection;
import java.util.List;

import static cn.hperfect.nbquerier.enums.QueryRuleEnum.*;

/**
 * 一个查询类型
 */
public interface INbQueryType extends Serializable {

    List<QueryRuleEnum> LIST_RULE = ListUtil.toList(NOT_NULL,
            NOT_EMPTY, EMPTY,
            INTERS,
            NULL, IN, NOT_IN, ANY, NOT_ANY);
    List<QueryRuleEnum> NUM_RULE = ListUtil.toList(NOT_NULL, NULL, EQ, NE, BETWEEN, GT, GE, LE, LT, IN, NOT_IN);
    List<QueryRuleEnum> BOOL_RULE = ListUtil.toList(NOT_NULL, NULL, EQ, NE);
    List<QueryRuleEnum> STRING_RULE = ListUtil.toList(NOT_NULL,
            NULL, EQ, NE, LIKE, IN, NOT_IN, NOT_BLANK, BLANK,
            EMPTY, NOT_EMPTY);

    /**
     * 获取类型的数组类型(用于In操作)
     *
     * @return
     */
    default NbQueryType getArrayType() {
        throw new NbSQLException("类型:{},暂不支持数组查询", this);
    }

    /**
     * 查询是 如 where xxx in (?,?,?) 问好中存放的是子类型数据
     *
     * @return
     */
    default NbQueryType getArraySubType() {
        throw new NbSQLException("类型:{},暂不支持获取数组子类型", this);
    }

    /**
     * 获取数据库数据类型
     *
     * @return
     */

    DbDataType getDbDataType();

    /**
     * 是否是数组类型
     *
     * @return
     */
    default boolean isArray() {
        return getDbDataType() == DbDataType.ARRAY;
    }

    default boolean isJson() {
        return getDbDataType() == DbDataType.JSON;
    }

    /**
     * 类型转换
     *
     * @param value
     * @return
     * @throws TypeConvertException
     */
    Object convert(Object value) throws TypeConvertException;

    /**
     * 获取数据库类型名称 创建时,新建数组参数 取值(text,bigint,varchar)
     *
     * @return
     */
    String getDbTypeSql();

    /**
     * 标题
     * @return
     */
    String getTitle();

    /**
     * 处理whereLike
     *
     * @param expression
     * @param value
     * @param value
     */
    void whereLike(NbQuerier<?> querier, MergeSegments expression, String name, Object value);

    default void whereIn(NbQuerier<?> querier, MergeSegments expression, QueryRuleEnum ruleEnum, String name, Object value) {
        throw new NbSQLMessageException("类型:{},不支持in查询", getTitle());
    }

    /**
     * 获取可查询规则
     *
     * @return
     */
    List<QueryRuleEnum> getQueryRules();

    /**
     * 设置where 条件
     *
     * @param querier
     * @param expression
     * @param ruleEnum
     * @param name
     * @param value
     */
    default void setWhereCondition(NbQuerier<?> querier, MergeSegments expression, QueryRuleEnum ruleEnum, String name, Object value) {
        //判断是否支持改查询方式
        List<QueryRuleEnum> queryRules = getQueryRules();
        //为空不校验
        Assert.isTrue(queryRules == null || CollUtil.contains(queryRules, ruleEnum), "类型:{}暂时不支持:{}操作,只支持:{}", getTitle(), ruleEnum, queryRules);
        switch (ruleEnum) {
            /**
             * 可以查数组，也可以子查询
             */
            case IN:
            case NOT_IN:
                whereIn(querier, expression, ruleEnum, name, value);
                break;
            case LIKE:
                if (!StrUtil.isBlankIfStr(value)) {
                    whereLike(querier, expression, name, value);
                }
                break;
            case NOT_EMPTY:
            case EMPTY: {
                if (this.isJson()) {
                    JsonNbType jsonNbType = (JsonNbType) this;
                    if (jsonNbType.isJsonArray()) {
                        String key = StrUtil.format("coalesce(jsonb_array_length({}),0)", name);
                        QueryRuleEnum rule = ruleEnum == EMPTY ? EQ : GT;
                        String paramKey = querier.formatVariable(NbQueryType.INT, 0, false);
                        expression.add(() -> key, rule, () -> paramKey);
                    } else {
                        throw new NbSQLMessageException("暂时未实现json类型空查询");
                    }
                } else if (this.isArray()) {
                    //数组非空
                    String key = StrUtil.format("coalesce(array_length({}, 1),0)", name);
                    QueryRuleEnum rule = ruleEnum == EMPTY ? EQ : GT;
                    String paramKey = querier.formatVariable(NbQueryType.INT, 0, false);
                    expression.add(() -> key, rule, () -> paramKey);
                } else if (this.equals(NbQueryType.TEXT)) {
                    QueryRuleEnum rule = ruleEnum == EMPTY ? BLANK : NOT_BLANK;
                    setWhereCondition(querier, expression, rule, name, value);
                } else {
                    throw new NbSQLMessageException("该类型不支持 EMPTY 操作");
                }
            }
            break;
            case BETWEEN:
                //返回查询,时间,数字
                //between 处理
                List<?> list = Convert.toList(value);
                if (CollUtil.isEmpty(list)) {
                    break;
                }
                //bt 扩展
                Assert.isTrue(list.size() > 1, "$bt取中间必须要有两个值");
                Object[] objects = list.toArray();
                expression.add(() -> name, ruleEnum, () -> querier.formatVariable(this, objects[0], false), SqlKeyword.AND, () -> querier.formatVariable(this, objects[1], false));
                break;
            case NULL:
            case NOT_NULL:
                // null 查询
                expression.add(() -> name, ruleEnum);
                break;
            case EQ:
            case NE:
            case GE:
            case GT:
            case LT:
            case LE:
                if (!StrUtil.isBlankIfStr(value)) {
                    whereOperation(querier, expression, ruleEnum, name, value);
                }
                break;
            case NOT_BLANK:
            case BLANK:
                // = ''
                String key = StrUtil.format("trim(coalesce({}, ''))", name);
                QueryRuleEnum rule = ruleEnum == BLANK ? EQ : NE;
                whereOperation(querier, expression, rule, key, StringPool.BLANK);
                break;
            case INTERS: {
                String paramKey = querier.formatVariable(this, value, true);
                expression.add(() -> name, ruleEnum, () -> paramKey);
            }
            break;
            case ANY:
            case NOT_ANY:
                DbType dbType = querier.getDbType();
                Assert.isTrue(dbType == DbType.POSTGRE_SQL || dbType == DbType.PG_NG, "只有POSTGRESQL支持ANY操作");
                if (ruleEnum == NOT_ANY) {
                    expression.add(SqlKeyword.NOT);
                }
                //如果只是数组
                if (value instanceof Collection) {
                    String paramKey = querier.formatVariable(this, value, false, false);
                    //over && []
                    expression.add(() -> name, SqlKeyword.ARRAY_OVERLAP, () -> paramKey);
                } else {
                    String paramKey = querier.formatVariable(this.getArraySubType(), value, false, false);
                    //not any
                    expression.add(() -> paramKey, (ruleEnum == QueryRuleEnum.ANY ? SqlKeyword.EQ : SqlKeyword.NE), SqlKeyword.ANY, () -> StrUtil.format("({})", name));
                }
                break;
            default:
                throw new NbSQLException("暂不支持该查询方式");
        }
    }

    /**
     * 运算符
     *
     * @param querier
     * @param expression
     * @param ruleEnum
     * @param name
     * @param value
     */
    default void whereOperation(NbQuerier<?> querier, MergeSegments expression, QueryRuleEnum ruleEnum, String name, Object value) {
        String paramKey = querier.formatVariable(this, value, false);
        expression.add(() -> name, ruleEnum, () -> paramKey);
    }

    /**
     * 可排序
     *
     * @return
     */
    boolean orderAble();

    /**
     * 格式化导入数据
     */
    default Object formatImportValue(String val){
        return null;
    }

 /*   @Override
    default void writeExternal(ObjectOutput out) throws IOException {
        out.writeUTF(getName());
    }

    @Override
    default void readExternal(ObjectInput in) throws IOException, ClassNotFoundException {
        String name = in.readUTF();
        System.out.println(name);
    }*/
}
