package cn.hperfect.nbquerier.core.components.builder.impl;

import cn.hperfect.nbquerier.core.components.builder.INbFieldBuilder;
import cn.hperfect.nbquerier.core.components.builder.INbSqlBuilder;
import cn.hperfect.nbquerier.core.components.dialect.IDialectWare;
import cn.hperfect.nbquerier.core.metedata.NbQueryInfo;
import cn.hperfect.nbquerier.core.metedata.PageInfo;
import cn.hperfect.nbquerier.core.metedata.QueryField;
import cn.hperfect.nbquerier.core.metedata.inter.INbField;
import cn.hperfect.nbquerier.core.metedata.querier.UpdateData;
import cn.hperfect.nbquerier.core.querier.NbQuerier;
import cn.hperfect.nbquerier.enums.QueryType;
import cn.hperfect.nbquerier.enums.SqlKeyword;
import cn.hperfect.nbquerier.enums.perm.PermType;
import cn.hperfect.nbquerier.exceptions.NbSQLMessageException;
import cn.hperfect.nbquerier.toolkit.SqlUtils;
import cn.hperfect.nbquerier.toolkit.StringPool;
import cn.hutool.core.collection.CollUtil;
import cn.hutool.core.lang.Assert;
import cn.hutool.core.util.ObjectUtil;
import cn.hutool.core.util.StrUtil;
import lombok.extern.slf4j.Slf4j;

import java.util.*;
import java.util.stream.Collectors;

/**
 * @author huanxi
 * @version 1.0
 * @date 2021/11/24 10:29 下午
 */
@Slf4j
public class DefaultSqlBuilder implements INbSqlBuilder {


    private final INbFieldBuilder fieldBuilder;


    private final IDialectWare dialectWare;

    public DefaultSqlBuilder(IDialectWare dialectWare, INbFieldBuilder fieldBuilder) {
        this.fieldBuilder = fieldBuilder;
        this.dialectWare = dialectWare;
    }

    @Override
    public String buildDeleteSql(NbQuerier<?> querier) {
        Assert.isTrue(querier.hasWhere(), "该删除语句没有条件");
        NbQueryInfo queryInfo = querier.getQueryInfo();
        StringBuilder builder = new StringBuilder();
        builder.append("DELETE");
        builder.append(StringPool.SPACE);
        builder.append("FROM");
        builder.append(StringPool.SPACE);
        builder.append(buildFormSql(queryInfo));
        appendWhere(querier, queryInfo, builder, true);
        return builder.toString();
    }


    @Override
    public <T> String buildQuerySql(NbQuerier<T> querier, String field, boolean page) {
        NbQueryInfo queryInfo = querier.getQueryInfo();
        if (StrUtil.isBlank(field)) {
            field = fieldBuilder.buildFieldSql(querier);
        }
        StringBuilder builder = new StringBuilder();
        builder.append("SELECT");
        builder.append(StringPool.SPACE);
        builder.append(field);
        builder.append(StringPool.SPACE);
        builder.append("FROM");
        builder.append(StringPool.SPACE);
        builder.append(buildFormSql(queryInfo));
        //加上连表条件
        if (CollUtil.isNotEmpty(querier.getJoins())) {
            querier.getJoins().forEach(joinTable -> {
                builder.append(StringPool.SPACE);
                builder.append(joinTable.getSqlSegment());
                builder.append(StringPool.SPACE);
                builder.append(buildFormSql(joinTable.getQueryInfo()));
                builder.append(StringPool.SPACE);
                builder.append(SqlKeyword.ON);
                builder.append(StringPool.SPACE);
                builder.append(joinTable.getOn());
            });
        }
        appendWhere(querier, queryInfo, builder, page);
        if (CollUtil.isNotEmpty(queryInfo.getUnionAllQueries())) {
            for (NbQuerier<?> unionAllQuery : queryInfo.getUnionAllQueries()) {
                builder.append(StringPool.SPACE);
                builder.append(SqlKeyword.UNION_ALL);
                builder.append(StringPool.SPACE);
                builder.append(buildQuerySql(unionAllQuery));
            }
        }
        return builder.toString();
    }

    /**
     * 一个构造器值构建一次
     *
     * @param querier
     * @param data
     * @return
     */
    @Override
    public String buildSaveSql(NbQuerier<?> querier, Map<String, Object> data) {
        List<String> keys = new ArrayList<>();
        List<String> values = new ArrayList<>();
        NbQueryInfo queryInfo = querier.getQueryInfo();
        Assert.isFalse(queryInfo.isSub(), "子查询不支持保存");
        String table = queryInfo.getTableName();
        INbField orderField = queryInfo.getTable().getOrderField();
        //数据权限字段构建
        if (queryInfo.getPermType() == PermType.SELF && !queryInfo.isIgnoreDataScope()) {
            INbField one = queryInfo.getPermField();
            Assert.notNull(one, "私有表:{},没有权限字段", queryInfo.getTableName());
            Object permValue = queryInfo.getPermValue();
            Assert.notNull(permValue, "私有表:{},权限值不能为空");
            Object userPermVal = data.get(one.getName());
            if (userPermVal == null) {
                data.put(one.getName(), permValue);
            } else {
                Assert.isTrue(permValue.equals(userPermVal), "您无权限操作该数据");
            }
        }
        //获取sql语句
        data.forEach((key, value) -> {
            INbField one = CollUtil.findOne(queryInfo.getFields(), i -> key.equals(i.getName()));
            if (one != null && !one.isSave()) {
                return;
            }
            //判断表
            if (one != null && StrUtil.isNotBlank(one.getTableName())
                    && !StrUtil.equals(one.getTableName(), table)) {
                return;
            }

            QueryField queryField = querier.findQueryField(key, table);
            boolean isOrder = queryField != null && (orderField != null && orderField.getName().equals(queryField.getName()));
            boolean insertDefault = ((one != null && one.isInsertDefault()) || isOrder);
            //key->重定向
            String dbName = Optional.ofNullable(queryField).map(i -> {
                if (StrUtil.isNotBlank(i.getRedirect())) {
                    return i.getRedirect();
                }
                return null;
            }).orElse(key);
            // queryField != null && StrUtil.isNotBlank(queryField.getRedirect()) ? queryField.getRedirect() : key;
            if (StrUtil.isBlankIfStr(value) && insertDefault) {
                //为空填充 default
                keys.add(dialectWare.quotSqlKey(dbName));
                values.add("default");
            } else if (value != null && queryField != null) {
                keys.add(dialectWare.quotSqlKey(dbName));
                values.add(querier.formatVariable(queryField.getType(), value, true));
            }
        });
        Assert.notNull(keys, "保存字段为空");
        String keySql = String.join(",", keys);
        String valueSql = String.join(",", values);
        //排序字段
        return String.format("INSERT INTO %s (%s) VALUES (%s)", queryInfo.getSchemaTable(true), keySql, valueSql);
    }

    @Override
    public String buildSaveSql(NbQuerier<?> querier, List<Map<String, Object>> maps) {
        List<String> values = new ArrayList<>();
        NbQueryInfo queryInfo = querier.getQueryInfo();
        Set<String> keys = queryInfo.getFields()
                .stream()
                .filter(INbField::isSave)
                .filter(i -> StrUtil.isBlank(i.getTableName()) || ObjectUtil.equal(i.getTableName(), querier.getTableName()))
                .map(INbField::getRedirectOrName)
                .collect(Collectors.toSet());
        Assert.notEmpty(keys, "没有更新字段");
        String tableName = queryInfo.getTableName();
        INbField orderField = queryInfo.getTable().getOrderField();
        //map数据转换成
        //获取sql语句
        for (Map<String, Object> datum : maps) {
            //一个数据的值
            List<String> oneValue = new ArrayList<>();
            for (String key : keys) {
                QueryField queryField = querier.findQueryField(key, tableName);
                Assert.notNull(queryField, String.format("table:%s,字段%s不存在", tableName, key));
                Object value = datum.get(key);
                if (StrUtil.isBlankIfStr(value) && orderField != null && orderField.getName().equals(queryField.getName())) {
                    oneValue.add("default");
                } else {
                    oneValue.add(querier.formatVariable(queryField.getType(), value, true));
                }

            }
            values.add(String.format("(%s)", String.join(",", oneValue)));
        }

        String keySql = String.join(",", keys);
        String valueSql = String.join(",", values);
        return StrUtil.format("INSERT INTO {} ({}) VALUES {}", queryInfo.getSchemaTable(true), keySql, valueSql);
    }

    @Override
    public String buildUpdateSql(NbQuerier<?> querier, UpdateData updateData) {
        Assert.notNull(updateData, "没有更新数据");
        Assert.isTrue(querier.hasWhere(), "该更新语句没有条件");
        StringBuilder builder = new StringBuilder();
        NbQueryInfo queryInfo = querier.getQueryInfo();
        String schemaTable = queryInfo.getSchemaTable(true);
        if (StrUtil.isNotBlank(queryInfo.getAlias())) {
            schemaTable = schemaTable + " " + queryInfo.getAlias();
        }
        builder.append(String.format("UPDATE %s SET ", schemaTable));
        List<String> setStrings = new ArrayList<>();
        Map<INbField, Object> updateDataMap = updateData.getUpdateDataMap();
        //忽略主键
        updateDataMap.forEach((field, value) -> {
            //忽略主键更新
            if (field.isPk()) {
                return;
            }
            //字段替换
            QueryField queryField = querier.findQueryField(field.getName(), queryInfo.getTableName());
            if (queryField == null) {
                return;
            }
            String key = StrUtil.blankToDefault(queryField.getRedirect(), field.getName());
            //String key = queryField != null && StrUtil.isNotBlank(queryField.getRedirect()) ? queryField.getRedirect() : field.getName();
            //获取类型
            String valueParam = querier.formatVariable(field.getQueryType(), value, true);
            setStrings.add(StrUtil.format("{}={}", dialectWare.quotSqlKey(key), valueParam));
        });
        //添加表达式
        List<String> updateExprList = updateData.getUpdateExprList();
        if (CollUtil.isNotEmpty(updateExprList)) {
            setStrings.addAll(updateExprList);
        }

        Assert.notEmpty(setStrings, "没有需要更新的数据");
        builder.append(String.join(",", setStrings));
        //where
        appendWhere(querier, queryInfo, builder, true);

        return builder.toString();
    }

    @Override
    public String buildFuncSql(NbQuerier<?> querier, String funcName, boolean limit) {
        return buildQuerySql(querier, funcName, limit);
    }

    /**
     * 附加where条件
     *
     * @param querier
     * @param queryInfo
     * @param builder
     * @param page
     */
    private void appendWhere(NbQuerier<?> querier, NbQueryInfo queryInfo, StringBuilder builder, boolean page) {
        Assert.isFalse(!page && querier.getQueryInfo().getPageInfo() != null && querier.getQueryInfo().getPageInfo().isBuild(),
                "不查询page信息,但是已经查询过了page信息，请调整位置");
        //page信息
        if (page && queryInfo.getPageInfo() != null) {
            PageInfo pageInfo = queryInfo.getPageInfo();
            if (!pageInfo.isBuild()) {
                querier.limit((pageInfo.getPageNo() - 1) * pageInfo.getPageSize(), pageInfo.getPageSize());
                pageInfo.setBuild(true);
            }
        }
        //数据权限构建
        buildDataScope(querier, queryInfo);
        //构建软删除
        buildDelete(querier, queryInfo);

        String sqlSegment = querier.getSqlSegment();
        if (querier.hasWhere()) {
            builder.append(StringPool.SPACE);
            builder.append("WHERE");
            builder.append(StringPool.SPACE);
        }
        builder.append(sqlSegment);
    }

    /**
     * 限制数据范围
     *
     * @param querier
     * @param queryInfo
     */
    private void buildDataScope(NbQuerier<?> querier, NbQueryInfo queryInfo) {
        if (queryInfo.isIgnoreDataScope() || queryInfo.isBuildDataScope()) {
            return;
        }
        //检测
        List<Object> dataScope = queryInfo.getDataScope();
        Object permValue = queryInfo.getPermValue();
        PermType permType = querier.getQueryInfo().getPermType();
        if (permType != null && permType != PermType.ALL) {
            switch (permType) {
                case SELF:
                    Assert.notNull(permValue, "私有表:{},权限值不能为空", queryInfo.getTableName());
                    break;
                case AUTHORISED:
                    Assert.isTrue(CollUtil.isNotEmpty(dataScope), "表:{}数据域不能为空", queryInfo.getTableName());
                    break;
            }
        }
        //数据权限是数据
        if (CollUtil.isNotEmpty(dataScope)) {
            INbField permField = queryInfo.getPermField();
            setPermValue(querier, queryInfo, dataScope, permField);
        }

        if (permValue != null && permType != PermType.ALL) {
            //todo
            INbField permField = queryInfo.getPermField();
            setPermValue(querier, queryInfo, permValue, permField);

        }

        queryInfo.setBuildDataScope(true);
    }

    /**
     * 设置权限值
     *
     * @param querier
     * @param queryInfo
     * @param permValue
     * @param permField
     */
    private void setPermValue(NbQuerier<?> querier, NbQueryInfo queryInfo, Object permValue, INbField permField) {
        if (permField == null && queryInfo.getPermType() == PermType.ALL) {
            return;
        }

        Assert.notNull(permField, "权限字段不能为空");
        //获取字段名,都是主表的权限
        String aField = SqlUtils.withAlias(querier.getTableAliasMap().get(queryInfo.getTableName()), permField.getName());
        //判断该权限字段的数据类型
        if (permField.getQueryType().isArray()) {
            //todo  数组权限
            throw new NbSQLMessageException("为实现数组数据权限");
        } else {
            if (permValue instanceof List) {
                List<?> permValueList = (List<?>) permValue;
                if (permValueList.size() == 1) {
                    querier.where(aField, permValueList.get(0));
                } else {
                    querier.whereIn(aField, permValueList);
                }
            } else {
                querier.where(aField, permValue);
            }

        }
    }

    /**
     * 构建软删除
     *
     * @param querier
     * @param queryInfo
     */
    private void buildDelete(NbQuerier<?> querier, NbQueryInfo queryInfo) {
        if (queryInfo.getQueryType() != null && queryInfo.getQueryType() == QueryType.FORM_QUERY) {
            //忽略from 语句的软删除
            return;
        }
        INbField deleteField = queryInfo.getDeleteField();
        if (!queryInfo.isWithDelete() && !queryInfo.isBuildDelete() && deleteField != null) {
            //todo 如果包含or的情况
            //查别名
            querier.whereNull(SqlUtils.withAlias(queryInfo.getAlias(), deleteField.getName()));
            queryInfo.setBuildDelete(true);
        }
    }


    /**
     * 构建from语句
     *
     * @param info
     * @return
     */
    public String buildFormSql(NbQueryInfo info) {
        StringBuilder stringBuilder = new StringBuilder();
        if (info.isSub()) {
            stringBuilder.append(StringPool.LEFT_BRACKET);
            stringBuilder.append(info.getSql());
            stringBuilder.append(StringPool.RIGHT_BRACKET);
        } else if (info.getQueryType() != null && info.getQueryType() == QueryType.FORM_QUERY) {
            stringBuilder.append(StringPool.LEFT_BRACKET);
            stringBuilder.append(buildQuerySql(info.getFromQuerier()));
            stringBuilder.append(StringPool.RIGHT_BRACKET);
        } else {
            stringBuilder.append(SqlUtils.withAlias(info.getSchema(), info.getTableName()));
        }
        String alias = info.getAlias();
        if (StrUtil.isNotBlank(alias)) {
            stringBuilder.append(StringPool.SPACE);
            stringBuilder.append(alias);
        }
        return stringBuilder.toString();
    }

}
