package cn.hperfect.nbquerier.core.components.builder.impl;

import cn.hperfect.nbquerier.annotation.*;
import cn.hperfect.nbquerier.config.NbQuerierConfiguration;
import cn.hperfect.nbquerier.config.properties.FillStrategyProperties;
import cn.hperfect.nbquerier.config.properties.NbQuerierProperties;
import cn.hperfect.nbquerier.cons.ValidatorCons;
import cn.hperfect.nbquerier.core.components.builder.INbQueryBuilder;
import cn.hperfect.nbquerier.core.components.interceptor.components.fill.FieldFillStrategyRegistry;
import cn.hperfect.nbquerier.core.components.interceptor.components.fill.FillStrategy;
import cn.hperfect.nbquerier.core.components.interceptor.components.fill.IFieldApplier;
import cn.hperfect.nbquerier.core.components.route.ISchemaRoute;
import cn.hperfect.nbquerier.core.components.type.INbQueryType;
import cn.hperfect.nbquerier.core.components.type.NbQueryType;
import cn.hperfect.nbquerier.core.metedata.*;
import cn.hperfect.nbquerier.core.metedata.field.ClassNbField;
import cn.hperfect.nbquerier.core.metedata.inter.INbTable;
import cn.hperfect.nbquerier.core.metedata.table.ClassNbTable;
import cn.hperfect.nbquerier.core.metedata.table.ThisTable;
import cn.hperfect.nbquerier.core.querier.DefaultNbQuerier;
import cn.hperfect.nbquerier.core.querier.NbQuerier;
import cn.hperfect.nbquerier.enums.IdType;
import cn.hperfect.nbquerier.enums.NbFieldType;
import cn.hperfect.nbquerier.exceptions.NbSQLException;
import cn.hutool.cache.Cache;
import cn.hutool.cache.CacheUtil;
import cn.hutool.core.bean.BeanDesc;
import cn.hutool.core.bean.BeanUtil;
import cn.hutool.core.bean.PropDesc;
import cn.hutool.core.collection.CollUtil;
import cn.hutool.core.lang.Assert;
import cn.hutool.core.lang.Dict;
import cn.hutool.core.util.StrUtil;
import lombok.extern.slf4j.Slf4j;

import java.lang.reflect.Field;
import java.lang.reflect.Method;
import java.lang.reflect.Modifier;
import java.lang.reflect.Type;
import java.util.*;

/**
 * @author huanxi
 * @version 1.0
 * @date 2021/11/24 9:52 上午
 */
@Slf4j
public class DefaultNbQueryBuilder implements INbQueryBuilder {
    private final NbQuerierProperties config;

    private final NbQuerierConfiguration configuration;

    private final FieldFillStrategyRegistry strategyRegistry;

    public DefaultNbQueryBuilder(NbQuerierConfiguration configuration) {
        this.config = configuration.getConfig();
        this.configuration = configuration;
        this.strategyRegistry = configuration.getFieldFillStrategyRegistry();
    }


    private final static Cache<Class<?>, ClassNbTable> TABLE_INFOS = CacheUtil.newLFUCache(100);

    private ISchemaRoute getSchemaRoute() {
        return configuration.getSchemaRoute();
    }

    @Override
    public <T> NbQuerier<T> build(Class<T> clazz, String alias) {
        DefaultNbQuerier<T> querier = newNbQuerier();
        NbQueryInfo nbQueryInfo = buildQueryInfo(clazz, alias);
        querier.setQueryInfo(nbQueryInfo);
        querier.setResultClass(clazz);
        return querier;
    }

    @Override
    public <T> NbQuerier<T> build(NbQuerier<?> outer, Class<T> clazz) {
        DefaultNbQuerier<T> querier = newNbQuerier();

//        ClassNbTable tableInfo = parseTableInfo(tableClazz);
//        NbQueryInfo queryInfo = buildQueryInfo(tableInfo, alias);

        NbQueryInfo nbQueryInfo = buildQueryInfo(clazz, null);
        querier.setQueryInfo(nbQueryInfo);
        querier.setResultClass(clazz);
        return querier;
    }

    @Override
    public <T> NbQuerier<T> build(INbTable nbTable, Class<T> clazz, String alias) {
        NbQueryInfo queryInfo = buildQueryInfo(nbTable, alias);
        return build(queryInfo, clazz);
    }

    public <T> NbQuerier<T> build(NbQueryInfo queryInfo, Class<T> clazz) {
        DefaultNbQuerier<T> querier = newNbQuerier();
        querier.setQueryInfo(queryInfo);
        querier.setResultClass(clazz);
        return querier;
    }

    @Override
    public <T> NbQuerier<T> build(NbQueryInfo queryInfo) {
        DefaultNbQuerier<T> querier = newNbQuerier();
        querier.setQueryInfo(queryInfo);
        return querier;
    }

    @Override
    public <T> DefaultNbQuerier<T> newNbQuerier() {
        return new DefaultNbQuerier<>(configuration);
    }


    @Override
    public NbQueryInfo buildQueryInfo(INbTable nbTable, String alias) {
        Assert.notNull(nbTable, "table不能为空");
        if (ThisTable.ME.equals(nbTable)) {
            NbQueryInfo queryInfo = new NbQueryInfo();
            queryInfo.setTable(nbTable);
            return queryInfo;
        }

        NbQueryInfo queryInfo = new NbQueryInfo();
        String tableName = nbTable.getTableName();
        Assert.notBlank(tableName, "查询信息表名不能为空");
        queryInfo.setTable(nbTable);
        queryInfo.setAlias(alias);
        queryInfo.setDs(nbTable.getDs());

        //schema 路由器中获取
        if (StrUtil.isBlank(nbTable.getSchema())) {
            ISchemaRoute schemaRoute = getSchemaRoute();
            if (schemaRoute != null) {
                queryInfo.setSchema(schemaRoute.determineSchema(nbTable.getTableName()));
            }
        } else {
            queryInfo.setSchema(nbTable.getSchema());
        }
        return queryInfo;
    }

    @Override
    public NbQueryInfo buildQueryInfo(NbQuerier<?> querier, String alias) {
        NbQueryInfo queryInfo = new NbQueryInfo();
        queryInfo.setAlias(alias);
        return queryInfo;
    }

    @Override
    public NbQueryInfo buildQueryInfo(Class<?> tableClazz, String alias) {
        //构建nbTable
        Assert.notNull(tableClazz, "table不能为空");
        ClassNbTable tableInfo = parseTableInfo(tableClazz);
        NbQueryInfo queryInfo = buildQueryInfo(tableInfo, alias);
        queryInfo.setClass(true);
        return queryInfo;
    }

    /**
     * 解析tableInfo
     *
     * @param tableClazz
     * @return
     */
    @Override
    public ClassNbTable parseTableInfo(Class<?> tableClazz) {
        return parseTableInfo(new ParseContext(), tableClazz);
    }

    public ClassNbTable parseTableInfo(ParseContext context, Class<?> tableClazz) {
        ClassNbTable tableInfo = TABLE_INFOS.get(tableClazz);
        if (tableInfo == null || !config.isClassCache()) {
            //构建tableInfo
            tableInfo = context.getTableInfo(tableClazz);
            if (tableInfo != null) {
                return tableInfo;
            }
            tableInfo = context.createClassNbTable(tableClazz);
            NbTable nbTable = tableClazz.getAnnotation(NbTable.class);
            if (nbTable != null) {
                if (StrUtil.isNotBlank(nbTable.value())) {
                    tableInfo.setTableName(nbTable.value());
                }
                tableInfo.setSchema(nbTable.schema());
                tableInfo.setDs(nbTable.ds());
                tableInfo.setPermType(nbTable.perm());
            }
            if (StrUtil.isBlank(tableInfo.getTableName())) {
                tableInfo.setTableName(tableClazz.getSimpleName());
                if (config.isTableNameToUnderlineCase()) {
                    tableInfo.setTableName(StrUtil.toUnderlineCase(tableInfo.getTableName()));
                }
            }
            BeanDesc beanDesc = BeanUtil.getBeanDesc(tableClazz);
            tableInfo.setFields(parseFields(context, tableInfo, beanDesc));
            //解析排序信息
            setOrderInfo(tableInfo, tableClazz);
            //解析字段
            if (config.isClassCache()) {
                TABLE_INFOS.put(tableClazz, tableInfo);
            }
            //检测软删除
            this.checkSoftDelete(tableInfo, tableClazz);
        }
        return tableInfo;
    }

    private void setOrderInfo(ClassNbTable tableInfo, Class<?> tableClazz) {
        NbTableOrders orders = tableClazz.getAnnotation(NbTableOrders.class);
        if (orders != null) {
            List<OrderInfo> orderInfos = new ArrayList<>();
            NbTableOrder[] value = orders.value();
            for (NbTableOrder nbTableOrder : value) {
                OrderInfo orderInfo = new OrderInfo();
                orderInfo.setOrderType(nbTableOrder.type());
                //查找字段
                String orderName = StrUtil.toUnderlineCase(nbTableOrder.fieldName());
                ClassNbField field = CollUtil.findOne(tableInfo.getFields(), i -> i.getName().equals(orderName));
                Assert.notNull(field, "表单{}排序注解中,排序字段:{}不存在", tableInfo.getTableName(), orderName);
                orderInfo.setNbField(field);
                orderInfos.add(orderInfo);
            }
            tableInfo.setOrderInfos(orderInfos);
        }

    }

    private void checkSoftDelete(ClassNbTable tableInfo, Class<?> tableClazz) {
        if (tableInfo.getDeleteField() == null && config.isSoftDeleteNotice()) {
            IgnoreSoftDelete ignoreSoftDelete = tableClazz.getAnnotation(IgnoreSoftDelete.class);
            if (ignoreSoftDelete == null && tableClazz.getAnnotation(NbView.class) == null) {
                log.error("表单:{}没有软删除字段,如果不需要软删除请使用@IgnoreSoftDelete 注解", tableInfo.getTableName());
            }
        }
    }


    /**
     * 解析实体为NbField 之后无需反射解析
     *
     * @param context
     * @param tableInfo
     * @param beanDesc
     * @return
     */
    public List<ClassNbField> parseFields(ParseContext context, ClassNbTable tableInfo, BeanDesc beanDesc) {
        List<ClassNbField> nbTableFields = new ArrayList<>();
        String softDeleteField = config.getSoftDeleteField();
        //获取字段名称
        for (PropDesc prop : beanDesc.getProps()) {
            Class<?> declaringClass = prop.getField().getDeclaringClass();
            boolean isFromDict = declaringClass.isAssignableFrom(Dict.class); //Map.class.isAssignableFrom(declaringClass);
            if (isFromDict) {
                continue;
            }
            Field field = prop.getField();
            ClassNbField classNbField = new ClassNbField();
            //忽略final,static属性
            int modifiers = field.getModifiers();
            if (Modifier.isStatic(modifiers) || Modifier.isFinal(modifiers)) {
                continue;
            }
            //父类的注解获取不到
            NbField nbField = field.getAnnotation(NbField.class);
            if (nbField != null && parseNbField(context, field, classNbField, nbField)) {
                continue;
            }
            //类中解析字段名
            if (StrUtil.isBlank(classNbField.getName())) {
                String fieldName = field.getName();
                if (config.isFieldNameToUnderlineCase()) {
                    classNbField.setName(StrUtil.toUnderlineCase(fieldName));
                } else {
                    classNbField.setName(fieldName);
                }
            }
            classNbField.setFieldType(NbFieldType.NORMAL);
            TableId tableId = field.getAnnotation(TableId.class);
            if (tableId != null) {
                parseTableId(tableInfo, classNbField, tableId);
            }
            //设置字段名称
            classNbField.setPropName(field.getName());
            if (nbField != null && nbField.type() != NbQueryType.UNKNOWN) {
                //类型
                classNbField.setQueryType(nbField.type());
            } else {
                //解析类型
                Type genericType = field.getGenericType();
                try {
                    INbQueryType nbQueryType = NbQueryType.convertFromClass(field.getType(), genericType);
                    classNbField.setQueryType(nbQueryType);
                    if (nbQueryType == NbQueryType.MAP_ENUM) {
                        //构建map
                        Class<?> type = field.getType();
                        Object[] enumConstants = type.getEnumConstants();
                        Method getNameMethod = type.getMethod("name");
                        Map<String, Object> map = new HashMap<>();
                        for (Object enumConstant : enumConstants) {
                            String name = (String) getNameMethod.invoke(enumConstant);
                            map.put(name, enumConstant);
                        }
                        classNbField.setEnumMap(map);
                    }
                } catch (Exception throwable) {
                    throw new NbSQLException("table:{},字段类型:{},类型解析失败:{}", tableInfo.getTableName(), field.getName(), throwable);
                }

            }
            //软删除
            if (StrUtil.isNotBlank(softDeleteField) && softDeleteField.equals(classNbField.getName())) {
                classNbField.setFieldType(NbFieldType.DELETE);
                tableInfo.setDeleteField(classNbField);
            }
            //外键
            setFk(context, field, classNbField);
            //排序字段
            setOrderField(field, classNbField);
            //约束获取
            setConstraint(field, classNbField);
            //填充策略
            setFillStrategy(classNbField);
            //权限字段


            nbTableFields.add(classNbField);
        }
        return nbTableFields;
    }

    private void setOrderField(Field field, ClassNbField classNbField) {
        NbOrderField annotation = field.getAnnotation(NbOrderField.class);
        if (annotation != null) {
            classNbField.setFieldType(NbFieldType.ORDER);
        }
    }

    private void setConstraint(Field field, ClassNbField classNbField) {
        NbConstraint annotation = field.getAnnotation(NbConstraint.class);
        classNbField.setNotNull(false);
        if (annotation != null) {
            List<ValidateRule> validateRules = new ArrayList<>();
            classNbField.setNotNull(annotation.notNull());
            if (annotation.notNull()) {
                ValidateRule validateRule = new ValidateRule(ValidatorCons.VALIDATOR_NAME_NOT_NULL);
                validateRule.setMsg(annotation.isNullMessage());
                validateRules.add(validateRule);
            }
            if (annotation.unique()) {
                ValidateRule validateRule = new ValidateRule(ValidatorCons.VALIDATOR_NAME_UNIQUE);
                validateRule.setMsg(annotation.notUniqueMessage());
                validateRules.add(validateRule);
            }
            classNbField.setValidateRules(validateRules);
        }
    }

    private boolean parseNbField(ParseContext context, Field field, ClassNbField classNbField, NbField nbField) {
        classNbField.setInsertDefault(nbField.insertDefault());
        classNbField.setDescription(nbField.description());
        //解析别名
        String fieldAlias = nbField.alias();
        if (StrUtil.isBlank(fieldAlias) && nbField.type() == NbQueryType.FUNC) {
            fieldAlias = field.getName();
        }
        if (StrUtil.isNotBlank(fieldAlias)) {
            classNbField.setAlias(StrUtil.toUnderlineCase(fieldAlias));
        }
        if (!Objects.equals(nbField.table(), UnknownClass.class)) {
            classNbField.setTable(parseTableInfo(context, nbField.table()));
            Assert.isTrue(nbField.exist(), "字段:{}不能被忽略", field.getName());
        }
        if (!nbField.exist()) {
            //字段不存在
            return true;
        }
        //设置字段名称
        if (StrUtil.isNotBlank(nbField.value())) {
            classNbField.setName(nbField.value());
        }
        return false;
    }

    /**
     * 解析TableId
     *
     * @param tableInfo
     * @param classNbField
     * @param tableId
     * @return
     */
    private void parseTableId(ClassNbTable tableInfo, ClassNbField classNbField, TableId tableId) {
        IdType idType = tableId.type();
        if (StrUtil.isNotBlank(tableId.value())) {
            classNbField.setName(tableId.value());
        }
        if (idType == IdType.DEFAULT) {
            idType = config.getIdType();
        }
        PrimaryKey primaryKey = new PrimaryKey(classNbField.getName(), idType, tableId.queryType());
        //主键
        tableInfo.setPk(primaryKey);
        classNbField.setFieldType(NbFieldType.PK);
        //设置主键填充策略
        if (idType == IdType.ASSIGN_ID) {
            classNbField.setFillStrategy(strategyRegistry.getIdSnowflakeStrategy());
        } else if (idType == IdType.UUID) {
            throw new IllegalArgumentException("未实现uuid算法生成");
        }
    }

    /**
     * 解析外键
     *
     * @param context
     * @param field
     * @param classNbField
     */
    private void setFk(ParseContext context, Field field, ClassNbField classNbField) {
        TableFk tableFk = field.getAnnotation(TableFk.class);
        if (tableFk != null) {
            //循环依赖
            ClassNbTable classNbTable = parseTableInfo(context, tableFk.value());
            classNbField.setJoinTable(classNbTable);
            classNbField.setFieldType(NbFieldType.FK);
        }
    }

    /**
     * 解析配置填策略
     *
     * @param classNbField
     */
    private void setFillStrategy(ClassNbField classNbField) {
        if (CollUtil.isNotEmpty(config.getFillStrategies())) {
            for (FillStrategyProperties configStrategy : config.getFillStrategies()) {
                if (classNbField.getName().equals(configStrategy.getFieldName())) {
                    Assert.notEmpty(configStrategy.getApplier(), "填充器名称不能为空");
                    //添加策略
                    IFieldApplier fieldApplier = strategyRegistry.getFieldApplier(configStrategy.getApplier());
                    FillStrategy strategy = new FillStrategy(configStrategy.isSaveFill(),
                            configStrategy.isUpdateFill(), configStrategy.getFillType(), fieldApplier);
                    classNbField.setFillStrategy(strategy);
                }
            }
        }
    }


}
