package cn.hperfect.nbquerier.core.components.builder.impl;

import cn.hperfect.nbquerier.core.components.builder.INbFieldBuilder;
import cn.hperfect.nbquerier.core.components.dialect.IDialectWare;
import cn.hperfect.nbquerier.core.components.type.NbQueryType;
import cn.hperfect.nbquerier.core.metedata.*;
import cn.hperfect.nbquerier.core.metedata.inter.INbField;
import cn.hperfect.nbquerier.core.metedata.table.VirtualTable;
import cn.hperfect.nbquerier.core.querier.NbQuerier;
import cn.hperfect.nbquerier.enums.QueryType;
import cn.hperfect.nbquerier.exceptions.NbSQLMessageException;
import cn.hperfect.nbquerier.toolkit.BuildUtils;
import cn.hperfect.nbquerier.toolkit.FieldParseUtils;
import cn.hperfect.nbquerier.toolkit.SqlUtils;
import cn.hperfect.nbquerier.toolkit.StringPool;
import cn.hutool.core.collection.CollUtil;
import cn.hutool.core.collection.ListUtil;
import cn.hutool.core.lang.Assert;
import cn.hutool.core.map.BiMap;
import cn.hutool.core.map.MapUtil;
import cn.hutool.core.util.ObjectUtil;
import cn.hutool.core.util.StrUtil;
import lombok.extern.slf4j.Slf4j;
import net.sf.jsqlparser.JSQLParserException;
import net.sf.jsqlparser.parser.CCJSqlParserUtil;
import net.sf.jsqlparser.statement.select.*;
import net.sf.jsqlparser.statement.values.ValuesStatement;

import java.util.ArrayList;
import java.util.Collection;
import java.util.List;
import java.util.stream.Collectors;

/**
 * 字段构造器
 *
 * @author huanxi
 * @version 1.0
 * @date 2021/11/27 4:07 下午
 */
@Slf4j
public class DefaultFieldBuilder implements INbFieldBuilder {

    final IDialectWare dialectWare;

    /**
     * 忽略构建的字段
     */
    private final List<String> ignoreBuild = ListUtil.toList("*", "1");

    public DefaultFieldBuilder(IDialectWare dialectWare) {
        this.dialectWare = dialectWare;
    }

    @Override
    public <T> String buildFieldSql(NbQuerier<T> querier) {

        BiMap<String, String> tableAliasMap = querier.getTableAliasMap();
        NbQueryInfo queryInfo = querier.getQueryInfo();
        Collection<QueryField> fields = querier.getQueryFields().fields();
        List<JoinTableRule> joins = querier.getJoins();
        List<? extends ConditionAgg> conditionAggList = querier.getConditionAggList();

        String setField = queryInfo.getSetField();
        List<String> selectFields = new ArrayList<>();
        List<QueryItem> queryItemList = new ArrayList<>();
        //是否是排除 ! 开头
        boolean isExclude = true;
        //解析查询元素
        if (StrUtil.isNotBlank(setField)) {
            //todo addition 字段
            if (CollUtil.contains(ignoreBuild, setField)) {
                return setField;
            }
            isExclude = StrUtil.startWith(setField, "!");
            if (isExclude) {
                setField = StrUtil.removePrefix(setField, "!");
            }
            List<QueryItem> items = FieldParseUtils.simpleParse(setField);
            for (QueryItem item : items) {
                if (!item.isFunc()) {
                    item.setName(StrUtil.toUnderlineCase(item.getName()));
                }
            }
            queryItemList.addAll(items);
            //parseField(setField).forEach(i -> queryItemList.add(new QueryItem(i)));
        }
        if (!StringPool.EMPTY.equals(setField)) {
            if (isExclude) {
                //所有字段排除,
                selectFields.addAll(allExclude(tableAliasMap, queryInfo, joins, queryItemList));
            } else {
                //指定字段
                selectFields.addAll(pointField(tableAliasMap, queryInfo, fields, queryItemList));
            }
        }

        //处理附加字段
        List<String> additionFields = queryInfo.getAdditionFields();
        if (CollUtil.isNotEmpty(additionFields)) {
            selectFields.addAll(additionFields);
        }
        if (CollUtil.isNotEmpty(conditionAggList)) {
            selectFields.addAll(buildConditionAggList(conditionAggList));
        }
        String queryField = String.join(",", selectFields);
        //如果不是聚合
        Assert.notBlank(queryField, () -> new NbSQLMessageException("查询字段不能为空"));
        return queryField;
    }

    private List<String> buildConditionAggList(List<? extends ConditionAgg> conditionAggList) {
        return conditionAggList.stream()
                .map(ConditionAgg::getSqlSegment)
                .collect(Collectors.toList());
    }

    /**
     * 全局排除字段
     *
     * @param tableAliasMap
     * @param queryInfo
     * @param joins
     * @param excludeList
     * @return
     */
    private List<String> allExclude(BiMap<String, String> tableAliasMap, NbQueryInfo queryInfo, List<JoinTableRule> joins, List<QueryItem> excludeList) {
        List<String> selectFields = new ArrayList<>();
        for (INbField field : CollUtil.filterNew(queryInfo.getFields(), INbField::isQuery)) {
            String addSelectField = parseAddField(tableAliasMap, queryInfo, joins, excludeList, field);
            if (StrUtil.isNotBlank(addSelectField)) {
                selectFields.add(addSelectField);
            }
        }
        return selectFields;
    }

    /**
     * 解析添加字段
     *
     * @param tableAliasMap
     * @param queryInfo
     * @param joins
     * @param excludeList
     * @param field
     * @return
     */
    private String parseAddField(BiMap<String, String> tableAliasMap, NbQueryInfo queryInfo, List<JoinTableRule> joins, List<QueryItem> excludeList, INbField field) {
        String tableName;

        String fieldTable = field.getTableName();
        //是否排除 ! 字段
        QueryItem exclude = CollUtil.findOne(excludeList, i -> i.getName().equals(field.getName()));
        String addSelectField = null;
        //不是当前表,去连表
        boolean isInThisTable = StrUtil.isNotBlank(fieldTable) && !fieldTable.equals(queryInfo.getTableName());
        if (isInThisTable) {
            //不属于改表的判断是否在连表中
            if (CollUtil.isNotEmpty(joins)) {
                // 如果连表连的是子查询,判断表是否在子查询的表中
                boolean contains = CollUtil.contains(joins, i -> {
                    NbQueryInfo joinQueryInfo = i.getQueryInfo();
                    String queryTable;
                    if (joinQueryInfo.getQueryType() == QueryType.SUB) {
                        VirtualTable virtualTable = joinQueryInfo.getVirtualTable();
                        queryTable = virtualTable.getMasterTable();
                    } else {
                        queryTable = joinQueryInfo.getTableName();
                    }
                    Assert.notBlank(queryTable, () -> new NbSQLMessageException("连表表名不能为空"));
                    return ObjectUtil.equal(queryTable, fieldTable);

                });
                //判断join-> 是否需要加入
                if (contains) {
                    tableName = fieldTable;
                    addSelectField = isAddSelectField(tableAliasMap, tableName, exclude, field);
                }
            }
        } else {
            tableName = queryInfo.getTableName();
            //筛选字段
            addSelectField = isAddSelectField(tableAliasMap, tableName, exclude, field);
        }
        return addSelectField;
    }

    /**
     * 返回null 排除,
     * 否则直接返回查询表达式
     *
     * @param tableName
     * @param exclude
     * @return
     */
    public String isAddSelectField(BiMap<String, String> tableAliasMap, String tableName, QueryItem exclude, INbField field) {
        String fieldName = field.getName();
        String alias = tableAliasMap.get(tableName);
        if (exclude != null && exclude.getName().equals(fieldName)) {
            if (StrUtil.isBlank(exclude.getTableAlias()) || MapUtil.isEmpty(tableAliasMap)) {
                //没有表名
                //所有表的字段都排除
                return null;
            } else if (exclude.getTableAlias().equals(alias)) {
                //通过表别名排除
                return null;
            } else if (exclude.getTableAlias().equals(tableName)) {
                //通过表名排除
                return null;
            }
        }
        String quotSqlKey = dialectWare.quotSqlKey(fieldName);
        //函数字段不用表别名
        String queryName = field.getQueryType() == NbQueryType.FUNC ? quotSqlKey : SqlUtils.withAlias(alias, quotSqlKey);
        String fieldAlias = field.getAlias();
        if (StrUtil.isNotBlank(fieldAlias)) {
            queryName = queryName + " AS " + fieldAlias;
        }
        return queryName;
    }

    /**
     * 设置指定字段
     *
     * @param tableAliasMap
     * @param queryInfo
     * @param fields
     * @param pointList
     * @return
     */
    private List<String> pointField(BiMap<String, String> tableAliasMap, NbQueryInfo queryInfo, Collection<QueryField> fields, List<QueryItem> pointList) {
        List<String> queryItemList = new ArrayList<>();
        for (QueryItem pointField : pointList) {
            //字段的几种形式
            // xxx,a.xxx ,xxx bb, a.xxx as bb
            if (pointField.isFunc()) {
                //函数
                queryItemList.add(pointField.getExpr());
                continue;
            }
            if ("*".equals(pointField.getName())) {
                //*替换
                if (queryInfo.isFieldReplace()) {
                    queryItemList.add(pointField.getExpr());
                    break;
                }
                //加入所有
                String table = BuildUtils.getTableByField(pointField, tableAliasMap);
                Collection<QueryField> queryFields = CollUtil.filterNew(fields, i -> i.getTableName().equals(table));
                for (QueryField queryField : queryFields) {
                    queryItemList.add(SqlUtils.withAlias(pointField.getTableAlias(), queryField.getName()));
                }
                break;
            }
            if (StrUtil.isBlank(pointField.getTableAlias())) {
                QueryField one = BuildUtils.findOne(new QueryFields(fields), pointField, queryInfo.getTableName(), tableAliasMap);
                Assert.notNull(one, "查询字段:{}为空", pointField.getName());
                String tableAlias = tableAliasMap.get(one.getTableName());
                queryItemList.add(pointField.getExpr(tableAlias));
            } else {
                queryItemList.add(pointField.getExpr());
            }
        }
        return queryItemList;
    }

    /**
     * 解析字段
     *
     * @param str
     * @return
     */
    @Deprecated
    public static List<String> parseField(String str) {
        String sql = StrUtil.format("select {} from a", str);
        List<String> fields = new ArrayList<>();
        Select select = null;
        try {
            select = (Select) CCJSqlParserUtil.parse(sql);
        } catch (JSQLParserException e) {
            log.error("字段解析失败:", e);
            throw new NbSQLMessageException("字段解析失败");
        }
        SelectBody selectBody = select.getSelectBody();
        selectBody.accept(new SelectVisitor() {
            @Override
            public void visit(PlainSelect plainSelect) {
                List<SelectItem> selectItems = plainSelect.getSelectItems();
                selectItems.forEach(i -> fields.add(i.toString()));
            }

            @Override
            public void visit(SetOperationList setOperationList) {
            }

            @Override
            public void visit(WithItem withItem) {
            }

            @Override
            public void visit(ValuesStatement valuesStatement) {
            }
        });
        return fields;
    }

}
