package cn.hperfect.nbquerier.binding;

import cn.hperfect.nbquerier.annotation.sql.Delete;
import cn.hperfect.nbquerier.annotation.sql.Insert;
import cn.hperfect.nbquerier.annotation.sql.Select;
import cn.hperfect.nbquerier.annotation.sql.Update;
import cn.hperfect.nbquerier.config.NbQuerierConfiguration;
import cn.hperfect.nbquerier.core.components.builder.impl.ParamBuilder;
import cn.hperfect.nbquerier.core.components.executor.INbExecutor;
import cn.hperfect.nbquerier.core.querier.NbQuerier;
import cn.hperfect.nbquerier.enums.ResultType;
import cn.hperfect.nbquerier.enums.SqlCommandType;
import cn.hperfect.nbquerier.exceptions.NbSQLException;
import cn.hperfect.nbquerier.exceptions.NbSQLMessageException;
import cn.hperfect.nbquerier.parsing.GenericTokenParser;
import cn.hutool.core.collection.ListUtil;
import cn.hutool.core.lang.Assert;
import cn.hutool.core.util.TypeUtil;

import java.lang.annotation.Annotation;
import java.lang.reflect.*;
import java.util.*;
import java.util.stream.Collectors;

/**
 * mapper 的动态代理
 * 代理接口去执行 sql 语句
 *
 * @param <T>
 */
public class SqlMapperProxy<T> implements InvocationHandler {
    private final List<Class<? extends Annotation>> sqlAnnotationTypes =
            ListUtil.toList(Select.class, Insert.class, Update.class, Delete.class);
    private final NbQuerierConfiguration configuration;


    public SqlMapperProxy(NbQuerierConfiguration configuration, Class<T> clazz) {
        this.configuration = configuration;
    }


    /**
     * 执行方法
     *
     * @return
     * @throws Throwable
     */

    @Override
    public Object invoke(Object proxy, Method method, Object[] args) throws Throwable {
        //todo 缓存匹配,xml 匹配绑定
        SqlCommandType sqlCommandType = getSqlCommandType(method);
        if (sqlCommandType == null) {
            throw new NbSQLException("方法:{},无查询注解sql", method.getName());
        }
        boolean isUpdate = sqlCommandType == SqlCommandType.UPDATE || sqlCommandType == SqlCommandType.INSERT;
        ParamBuilder paramBuilder = new ParamBuilder();
        String sql = getSqlStr(sqlCommandType, method);
        //参数替换
        Map<String, ArgVariable> argVariableMap = getArgVariableMap(method, args);
        sql = new GenericTokenParser("#{", "}", new ParameterMappingTokenHandler(paramBuilder, isUpdate, argVariableMap)).parse(sql);
        // ${} 替换
        sql = new GenericTokenParser("${", "}", new ParameterReplaceTokenHandler(argVariableMap)).parse(sql);

        if (isUpdate) {
            return getExecutor().doUpdate(null, sql, paramBuilder.getParams());
        } else {
            Class<?> returnClass = method.getReturnType();
            if (String.class.equals(returnClass)) {
                return getExecutor().doQuery(null, sql, paramBuilder.getParams(), ResultType.STRING);
            } else if (Collection.class.isAssignableFrom(returnClass)) {
                Type beanClass = getBeanType(method);
                List<Map<String, Object>> maps = getExecutor().doQuery(null, sql, paramBuilder.getParams(), ResultType.LIST);
                //ParameterizedTypeImpl
                if (beanClass instanceof ParameterizedType) {
                    Class<?> rawType = (Class<?>) ((ParameterizedType) beanClass).getRawType();
                    //判断类型是否是map
                    if (Map.class.isAssignableFrom(rawType)) {
                        return maps;
                    }
                    throw new NbSQLMessageException("未处理结果集转换类型:{}", rawType);
                }
                NbQuerier<?> querier = NbQuerier.table((Class<?>) beanClass);
                return maps.stream().map(querier::mapToBean).collect(Collectors.toList());
            } else {

                //结果集 处理, class to table info
                throw new NbSQLMessageException("暂未处理该类型返回");
            }
        }
    }

    public INbExecutor getExecutor() {
        return configuration.newNbExecutor();
    }

    @SuppressWarnings("unchecked")
    private static Type getBeanType(Method method) {
        Type t = method.getGenericReturnType();
        ParameterizedType p = TypeUtil.toParameterizedType(t);
        Assert.isTrue(p.getActualTypeArguments().length == 1, "不支持泛型list返回");
        return p.getActualTypeArguments()[0];
    }

    private String getSqlStr(SqlCommandType sqlCommandType, Method method) {
        switch (sqlCommandType) {
            case SELECT:
                return method.getAnnotation(Select.class).value();
            case UPDATE:
                return method.getAnnotation(Update.class).value();
            case DELETE:
                return method.getAnnotation(Delete.class).value();
            case INSERT:
                return method.getAnnotation(Insert.class).value();
            case UNKNOWN:
                break;
        }

        throw new NbSQLException("无法解析sql");
    }

    /**
     * 1. 对象解析
     *
     * @param method
     * @param args
     * @return
     */
    private Map<String, ArgVariable> getArgVariableMap(Method method, Object[] args) {
        Map<String, ArgVariable> variableMap = new HashMap<>();
        Parameter[] parameters = method.getParameters();

        for (int i = 0; i < parameters.length; i++) {
            Parameter parameter = parameters[i];
            Object value = args[i];
            variableMap.put(parameter.getName(), new ArgVariable(parameter, value));
        }
        return variableMap;
    }


    /**
     * 获取查询类型
     *
     * @param method
     * @return
     */
    private SqlCommandType getSqlCommandType(Method method) {
        Class<? extends Annotation> type = getSqlAnnotationType(method);
        if (type == null) {
            return null;
        }
        return SqlCommandType.valueOf(type.getSimpleName().toUpperCase(Locale.ENGLISH));
    }

    private Class<? extends Annotation> getSqlAnnotationType(Method method) {
        return chooseAnnotationType(method, sqlAnnotationTypes);
    }

    private Class<? extends Annotation> chooseAnnotationType(Method method, List<Class<? extends Annotation>> types) {
        for (Class<? extends Annotation> type : types) {
            Annotation annotation = method.getAnnotation(type);
            if (annotation != null) {
                return type;
            }
        }
        return null;
    }


}
