package cn.hiroz.android.core;

import android.os.Handler;
import android.os.HandlerThread;
import android.os.Message;
import cn.hiroz.android.core.configuration.CoredroidConfiguration;

import java.util.*;
import java.util.concurrent.Executor;

/**
 * Created by hiro on 2/4/15.
 */
public class CoredroidEngine {

    private static final int MESSAGE_ENQUEUE = 1;
    private static final int MESSAGE_FINISH = 10;
    private static final int MESSAGE_CANCEL_TAG = 100;
    private static final int MESSAGE_CANCEL_TASK = 101;

    private Executor taskExecutor = null;
    private HandlerThread taskDistributorThread = null;
    private Handler taskDistributorHandler = null;

    private CoredroidConfiguration configuration;

    private Map<String, BaseTask> runningTask = new HashMap<String, BaseTask>();
    private Queue<BaseTask> taskQueue = new LinkedList<BaseTask>();

    public CoredroidEngine(CoredroidConfiguration configuration) {
        this.configuration = configuration;
        this.taskExecutor = DefaultConfigurationFactory.createExecutor(configuration.threadPoolSize);
        start();
    }

    public void enqueue(BaseTask task) {
        Message message = new Message();
        message.what = MESSAGE_ENQUEUE;
        message.obj = task;
        taskDistributorHandler.sendMessage(message);
    }

    public void cancel(String tag) {
        Message message = new Message();
        message.what = MESSAGE_CANCEL_TAG;
        message.obj = tag;
        taskDistributorHandler.sendMessage(message);
    }

    public void cancel(BaseTask task) {
        Message message = new Message();
        message.what = MESSAGE_CANCEL_TASK;
        message.obj = task;
        taskDistributorHandler.sendMessage(message);
    }

    void start() {
        stop();
        taskDistributorThread = new HandlerThread(UUID.randomUUID().toString());
        taskDistributorThread.start();
        taskDistributorHandler = new Handler(taskDistributorThread.getLooper(), new CoredroidEngineDistributorHandler());
    }

    void stop() {
        if (taskDistributorThread != null) {
            taskDistributorThread.quit();
        }
    }

    private void nextTask() {
        synchronized (this) {
            List<BaseTask> willRemove = new ArrayList<BaseTask>();
            for (BaseTask task : taskQueue) {
                if (task.getTag() != null) {
                    if (!runningTask.containsKey(task.getTag())) {
                        willRemove.add(task);
                        runningTask.put(task.getTag(), task);
                        submit(task);
                    }
                } else {
                    willRemove.add(task);
                    submit(task);
                }
            }
            taskQueue.removeAll(willRemove);
        }
    }

    private void submit(BaseTask task) {
        task.engine = this;
        taskExecutor.execute(task);
    }

    void doEnqueue(BaseTask task) {
        synchronized (this) {
            taskQueue.add(task);
        }
        nextTask();
    }

    void doFinish(BaseTask task) {
        synchronized (this) {
            if (task.getTag() != null) {
                runningTask.remove(task.getTag());
            }
        }
        nextTask();
    }

    void doCancel(String tag) {
        synchronized (this) {
            if (runningTask.containsKey(tag)) {
                runningTask.get(tag).stop();
            } else {
                List<BaseTask> willRemove = new ArrayList<BaseTask>();
                for (BaseTask task : taskQueue) {
                    if (tag.equals(task.getTag())) {
                        willRemove.add(task);
                    }
                }
                taskQueue.removeAll(willRemove);
            }
        }
    }

    void doCancel(BaseTask task) {
        synchronized (this) {
            if (runningTask.containsValue(task)) {
                task.stop();
            } else {
                taskQueue.remove(task);
            }
        }
    }

    class CoredroidEngineDistributorHandler implements Handler.Callback {

        @Override
        public boolean handleMessage(Message msg) {
            switch (msg.what) {
                case MESSAGE_ENQUEUE:
                    doEnqueue((BaseTask) msg.obj);
                    return true;
                case MESSAGE_FINISH:
                    doFinish((BaseTask) msg.obj);
                    return true;
                case MESSAGE_CANCEL_TAG:
                    doCancel((String) msg.obj);
                    return true;
                case MESSAGE_CANCEL_TASK:
                    doCancel((BaseTask) msg.obj);
                    return true;
                default:
                    return false;
            }
        }

    }

}
