package cn.hex01.billing.open.sdk.api;

import cn.hex01.billing.open.sdk.api.a.AccountApi;
import cn.hex01.billing.open.sdk.api.lq.LimitPurchaseRecordApi;
import cn.hex01.billing.open.sdk.api.lq.LimitQuotaApi;
import cn.hex01.billing.open.sdk.api.lq.LimitUsageRecordApi;
import cn.hex01.billing.open.sdk.api.q.PurchaseRecordApi;
import cn.hex01.billing.open.sdk.api.q.QuotaApi;
import cn.hex01.billing.open.sdk.api.q.UsageRecordApi;
import cn.hex01.billing.open.sdk.api.vq.ValidityPurchaseRecordApi;
import cn.hex01.billing.open.sdk.api.vq.ValidityQuotaApi;
import cn.hex01.billing.open.sdk.config.ApiConfig;
import cn.hex01.billing.open.sdk.interceptor.CustomRequestInterceptor;
import cn.hex01.billing.open.sdk.interceptor.CustomResponseInterceptor;
import feign.Feign;
import feign.gson.GsonDecoder;
import feign.gson.GsonEncoder;
import feign.okhttp.OkHttpClient;
import lombok.AccessLevel;
import lombok.NoArgsConstructor;

/**
 * @author laoji
 */
@NoArgsConstructor(access = AccessLevel.PRIVATE)
public final class BillingOpenApi {
    public static OpenApiBuilder builder(ApiConfig config) {
        return new OpenApiBuilder(config);
    }

    public static final class OpenApiBuilder {
        private final Feign.Builder feignBuilder;
        private final ApiConfig config;

        private OpenApiBuilder(ApiConfig config) {
            this.feignBuilder = Feign.builder()
                    .encoder(new GsonEncoder())
                    .decoder(new GsonDecoder())
                    .requestInterceptor(new CustomRequestInterceptor(config))
                    .responseInterceptor(new CustomResponseInterceptor())
                    .client(new OkHttpClient());
            this.config = config;
        }

        public AccountApi buildAccount() {
            return this.feignBuilder.target(AccountApi.class, this.config.getBaseUrl());
        }

        public LimitQuota buildLimitQuota() {
            return new LimitQuota(this.feignBuilder, this.config);
        }

        public Quota buildQuota() {
            return new Quota(this.feignBuilder, this.config);
        }

        public ValidityQuota buildValidityQuota() {
            return new ValidityQuota(this.feignBuilder, this.config);
        }
    }

    /**
     * 用量配额（仅额度）
     */
    public static final class LimitQuota {
        private final LimitPurchaseRecordApi limitPurchaseRecordApi;
        private final LimitQuotaApi limitQuotaApi;
        private final LimitUsageRecordApi limitUsageRecordApi;

        private LimitQuota(Feign.Builder feignBuilder, ApiConfig config) {
            this.limitPurchaseRecordApi = feignBuilder
                    .target(LimitPurchaseRecordApi.class, config.getBaseUrl());
            this.limitQuotaApi = feignBuilder
                    .target(LimitQuotaApi.class, config.getBaseUrl());
            this.limitUsageRecordApi = feignBuilder
                    .target(LimitUsageRecordApi.class, config.getBaseUrl());
        }

        /**
         * 购买记录API
         *
         * @return
         */
        public LimitPurchaseRecordApi purchaseRecord() {
            return this.limitPurchaseRecordApi;
        }

        /**
         * 配额API
         *
         * @return
         */
        public LimitQuotaApi quota() {
            return this.limitQuotaApi;
        }

        /**
         * 使用记录API
         *
         * @return
         */
        public LimitUsageRecordApi usageRecord() {
            return this.limitUsageRecordApi;
        }
    }

    /**
     * 配额（有效期+额度）
     */
    public static final class Quota {
        private final PurchaseRecordApi purchaseRecordApi;
        private final QuotaApi quotaApi;
        private final UsageRecordApi usageRecordApi;

        private Quota(Feign.Builder feignBuilder, ApiConfig config) {
            this.purchaseRecordApi = feignBuilder
                    .target(PurchaseRecordApi.class, config.getBaseUrl());
            this.quotaApi = feignBuilder
                    .target(QuotaApi.class, config.getBaseUrl());
            this.usageRecordApi = feignBuilder
                    .target(UsageRecordApi.class, config.getBaseUrl());
        }

        /**
         * 购买记录API
         *
         * @return
         */
        public PurchaseRecordApi purchaseRecord() {
            return this.purchaseRecordApi;
        }

        /**
         * 配额API
         *
         * @return
         */
        public QuotaApi quota() {
            return this.quotaApi;
        }

        /**
         * 使用记录API
         *
         * @return
         */
        public UsageRecordApi usageRecord() {
            return this.usageRecordApi;
        }
    }

    /**
     * 有效配额（仅有效期）
     */
    public static final class ValidityQuota {
        private final ValidityPurchaseRecordApi validityPurchaseRecordApi;
        private final ValidityQuotaApi validityQuotaApi;

        private ValidityQuota(Feign.Builder feignBuilder, ApiConfig config) {
            this.validityPurchaseRecordApi = feignBuilder
                    .target(ValidityPurchaseRecordApi.class, config.getBaseUrl());
            this.validityQuotaApi = feignBuilder
                    .target(ValidityQuotaApi.class, config.getBaseUrl());
        }

        /**
         * 购买记录API
         *
         * @return
         */
        public ValidityPurchaseRecordApi purchaseRecord() {
            return this.validityPurchaseRecordApi;
        }

        /**
         * 配额API
         *
         * @return
         */
        public ValidityQuotaApi quota() {
            return this.validityQuotaApi;
        }
    }
}
