/*
 * DO NOT ALTER OR REMOVE COPYRIGHT NOTICES OR THIS HEADER.
 *
 * Copyright (c) 2020-2030 郑庚伟 ZHENGGENGWEI (码匠君), <herodotus@aliyun.com> Licensed under the AGPL License
 *
 * This file is part of Herodotus Stirrup.
 *
 * Herodotus Stirrup is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published
 * by the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * Herodotus Stirrup is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <https://www.herodotus.vip>.
 */

package cn.herodotus.stirrup.tsdb.influxdb.core;

import cn.herodotus.stirrup.tsdb.influxdb.properties.InfluxdbProperties;
import org.apache.commons.collections4.MapUtils;
import org.apache.commons.lang3.ObjectUtils;
import org.influxdb.InfluxDB;
import org.influxdb.InfluxDBIOException;
import org.influxdb.dto.*;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.util.List;
import java.util.Map;
import java.util.concurrent.TimeUnit;
import java.util.function.BiConsumer;
import java.util.function.Consumer;

/**
 * <p>Description: TODO </p>
 *
 * @author : gengwei.zheng
 * @date : 2023/11/4 17:16
 */
public class InfluxdbTemplate {

    private static final Logger log = LoggerFactory.getLogger(InfluxdbTemplate.class);

    private final InfluxDB influxDB;
    private final InfluxdbProperties influxdbProperties;

    public InfluxdbTemplate(InfluxDB influxDB, InfluxdbProperties influxdbProperties) {
        this.influxDB = influxDB;
        this.influxdbProperties = influxdbProperties;
    }

    /**
     * 测试连接是否正常
     *
     * @return true 连接正常，false 连接失败
     */
    public boolean ping() {
        boolean isConnected = false;
        try {
            Pong pong = influxDB.ping();
            if (ObjectUtils.isNotEmpty(pong)) {
                isConnected = true;
            }
        } catch (InfluxDBIOException e) {
            log.error("[Herodotus] |- Unable to connect to server.", e);
        }
        return isConnected;
    }

    /**
     * 批量插入数据
     *
     * @param batchPoints 数据 {@link BatchPoints}
     */
    public void batchInsert(BatchPoints batchPoints) {
        influxDB.write(batchPoints);
    }

    /**
     * 批量插入数据
     *
     * @param points 数据
     */
    public void batchInsert(List<Point> points) {
        batchInsert(points, null);
    }

    /**
     * 批量插入数据
     *
     * @param points 数据
     * @param tags   标签
     */
    public void batchInsert(List<Point> points, Map<String, String> tags) {
        batchInsert(null, points, tags);
    }

    /**
     * 批量插入数据
     *
     * @param database 数据库。为空则使用外部配置值
     * @param points   数据
     * @param tags     标签
     */
    public void batchInsert(String database, List<Point> points, Map<String, String> tags) {
        batchInsert(database, influxdbProperties.getRetentionPolicy(), influxdbProperties.getConsistency(), influxdbProperties.getFlushDurationTimeUnit(), points, tags);
    }

    /**
     * 批量插入数据
     *
     * @param database        数据库。为空则使用外部配置值
     * @param retentionPolicy 保留策略，为空则使用外部配置值
     * @param consistency     一致性级别，为空则使用外部配置值
     * @param precision       时间单位 ，为空则使用外部配置值
     * @param points          数据
     * @param tags            标签
     */
    public void batchInsert(String database, String retentionPolicy, InfluxDB.ConsistencyLevel consistency, TimeUnit precision, List<Point> points, Map<String, String> tags) {
        BatchPoints.Builder builder = BatchPoints.database(database);
        builder.retentionPolicy(retentionPolicy);
        builder.consistency(consistency);
        if (MapUtils.isNotEmpty(tags)) {
            tags.forEach(builder::tag);
        }
        builder.precision(precision);
        builder.points(points);
        batchInsert(builder.build());
    }

    /**
     * 批量将多个记录写入数据库
     *
     * @param database 数据库
     * @param records  记录
     */
    public void batchInsert(String database, List<String> records) {
        batchInsert(database, influxdbProperties.getRetentionPolicy(), influxdbProperties.getConsistency(), records);
    }

    /**
     * 批量将多个记录写入数据库
     *
     * @param database        数据库
     * @param retentionPolicy 保留策略
     * @param consistency     一致性
     * @param records         记录
     */
    public void batchInsert(String database, String retentionPolicy, InfluxDB.ConsistencyLevel consistency, List<String> records) {
        batchInsert(database, retentionPolicy, consistency, TimeUnit.NANOSECONDS, records);
    }

    /**
     * 批量将多个记录写入数据库
     *
     * @param database        数据库
     * @param retentionPolicy 保留策略
     * @param consistency     一致性
     * @param precision       时间单位
     * @param records         记录
     */
    public void batchInsert(String database, String retentionPolicy, InfluxDB.ConsistencyLevel consistency, TimeUnit precision, List<String> records) {
        influxDB.write(database, retentionPolicy, consistency, precision, records);
    }

    /**
     * 将单个记录写入数据库
     *
     * @param database 数据库
     * @param records  记录
     */
    public void insert(String database, String records) {
        insert(database, influxdbProperties.getRetentionPolicy(), influxdbProperties.getConsistency(), records);
    }

    /**
     * 将单个记录写入数据库
     *
     * @param database        数据库
     * @param retentionPolicy 保留策略
     * @param consistency     一致性
     * @param records         记录
     */
    public void insert(String database, String retentionPolicy, InfluxDB.ConsistencyLevel consistency, String records) {
        insert(database, retentionPolicy, consistency, TimeUnit.NANOSECONDS, records);
    }

    /**
     * 将单个记录写入数据库
     *
     * @param database        数据库
     * @param retentionPolicy 保留策略
     * @param consistency     一致性
     * @param precision       时间单位
     * @param records         记录
     */
    public void insert(String database, String retentionPolicy, InfluxDB.ConsistencyLevel consistency, TimeUnit precision, String records) {
        influxDB.write(database, retentionPolicy, consistency, precision, records);
    }

    /**
     * 将单个点写入默认数据库
     *
     * @param measurement 度量（表）
     * @param fields      字段
     */
    public void insert(String measurement, Map<String, Object> fields) {
        insert(measurement, fields, null);
    }

    /**
     * 将单个点写入默认数据库
     *
     * @param measurement 度量（表）
     * @param tags        标签
     * @param fields      字段
     */
    public void insert(String measurement, Map<String, Object> fields, Map<String, String> tags) {
        insert(measurement, System.currentTimeMillis(), TimeUnit.MILLISECONDS, fields, tags);
    }

    /**
     * 将单个点写入默认数据库
     *
     * @param measurement 度量（表）
     * @param time        时间
     * @param precision   单位
     * @param tags        标签
     * @param fields      字段
     */
    public void insert(String measurement, Long time, TimeUnit precision, Map<String, Object> fields, Map<String, String> tags) {
        write(measurement, time, precision, fields, tags, 0);
    }

    /**
     * 将单个点写入默认数据库
     * <p>
     * influxDB开启UDP功能, 默认端口:8089,默认数据库:udp,没提供代码传数据库功能接口
     * 使用UDP的原因:
     * TCP数据传输慢，UDP数据传输快。网络带宽需求较小，而实时性要求高。
     * InfluxDB和服务器在同机房，发生数据丢包的可能性较小，即使真的发生丢包，对整个请求流量的收集影响也较小。
     *
     * @param measurement 度量（表）
     * @param fields      字段
     */
    public void insertWithUDP(String measurement, Map<String, Object> fields) {
        insertWithUDP(measurement, fields, null);
    }

    /**
     * 将单个点写入默认数据库
     * <p>
     * influxDB开启UDP功能, 默认端口:8089,默认数据库:udp,没提供代码传数据库功能接口
     * 使用UDP的原因:
     * TCP数据传输慢，UDP数据传输快。网络带宽需求较小，而实时性要求高。
     * InfluxDB和服务器在同机房，发生数据丢包的可能性较小，即使真的发生丢包，对整个请求流量的收集影响也较小。
     *
     * @param measurement 度量（表）
     * @param tags        标签
     * @param fields      字段
     */
    public void insertWithUDP(String measurement, Map<String, Object> fields, Map<String, String> tags) {
        insertWithUDP(measurement, System.currentTimeMillis(), TimeUnit.MILLISECONDS, fields, tags);
    }

    /**
     * 将单个点写入默认数据库
     * <p>
     * influxDB开启UDP功能, 默认端口:8089,默认数据库:udp,没提供代码传数据库功能接口
     * 使用UDP的原因:
     * TCP数据传输慢，UDP数据传输快。网络带宽需求较小，而实时性要求高。
     * InfluxDB和服务器在同机房，发生数据丢包的可能性较小，即使真的发生丢包，对整个请求流量的收集影响也较小。
     *
     * @param measurement 度量（表）
     * @param time        时间
     * @param precision   单位
     * @param tags        标签
     * @param fields      字段
     */
    public void insertWithUDP(String measurement, Long time, TimeUnit precision, Map<String, Object> fields, Map<String, String> tags) {
        write(measurement, time, precision, fields, tags, 8089);
    }

    /**
     * 将单个点写入默认数据库
     *
     * @param measurement 度量（表）
     * @param time        时间
     * @param precision   单位
     * @param tags        标签
     * @param fields      字段
     * @param udpPort     udp 端口
     */
    private void write(String measurement, Long time, TimeUnit precision, Map<String, Object> fields, Map<String, String> tags, int udpPort) {
        Point.Builder builder = Point.measurement(measurement);
        builder.fields(fields);
        if (ObjectUtils.isNotEmpty(time) && ObjectUtils.isNotEmpty(precision)) {
            builder.time(time, precision);
        }

        if (MapUtils.isNotEmpty(tags)) {
            builder.tag(tags);
        }

        if (udpPort != 0) {
            influxDB.write(udpPort, builder.build());
        } else {
            influxDB.write(builder.build());
        }
    }

    /**
     * 对数据库执行查询
     *
     * @param command 查询命令
     * @return 查询结果 {@link QueryResult}
     */
    public QueryResult query(String command) {
        return query(command, influxdbProperties.getDatabase());
    }

    /**
     * 对数据库执行查询
     *
     * @param command  查询命令
     * @param database 数据库
     * @return 查询结果 {@link QueryResult}
     */
    public QueryResult query(String command, String database) {
        return query(command, database, false);
    }

    /**
     * 对数据库执行查询
     *
     * @param command      查询命令
     * @param database     数据库
     * @param requiresPost 如果命令需要POST而不是GET到influxdb，则为true
     * @return 查询结果 {@link QueryResult}
     */
    public QueryResult query(String command, String database, boolean requiresPost) {
        return query(new Query(command, database, requiresPost));
    }

    /**
     * 对数据库执行查询
     *
     * @param query 查询参数 {@link Query}
     * @return 查询结果 {@link QueryResult}
     */
    public QueryResult query(Query query) {
        return influxDB.query(query);
    }

    /**
     * 对数据库执行查询
     *
     * @param command  查询命令
     * @param timeUnit 结果的时间单位
     * @return 查询结果 {@link QueryResult}
     */
    public QueryResult query(String command, TimeUnit timeUnit) {
        return query(command, influxdbProperties.getDatabase(), timeUnit);
    }

    /**
     * 对数据库执行查询
     *
     * @param command  查询命令
     * @param database 数据库
     * @param timeUnit 结果的时间单位
     * @return 查询结果 {@link QueryResult}
     */
    public QueryResult query(String command, String database, TimeUnit timeUnit) {
        return query(command, database, false, timeUnit);
    }

    /**
     * 对数据库执行查询
     *
     * @param command      查询命令
     * @param database     数据库
     * @param requiresPost 如果命令需要POST而不是GET到influxdb，则为true
     * @param timeUnit     结果的时间单位
     * @return 查询结果 {@link QueryResult}
     */
    public QueryResult query(String command, String database, boolean requiresPost, TimeUnit timeUnit) {
        return query(new Query(command, database, requiresPost), timeUnit);
    }

    /**
     * 对数据库执行查询
     *
     * @param query    查询参数 {@link Query}
     * @param timeUnit 结果的时间单位
     * @return 查询结果 {@link QueryResult}
     */
    public QueryResult query(Query query, TimeUnit timeUnit) {
        return influxDB.query(query, timeUnit);
    }

    /**
     * 对数据库执行查询
     *
     * @param command   查询命令
     * @param database  数据库
     * @param chunkSize 一个区块中要处理的QueryResults的数量
     * @param onNext    要为每个接收到的QueryResult调用的使用者
     */
    public void query(String command, String database, int chunkSize, Consumer<QueryResult> onNext) {
        query(command, database, false, chunkSize, onNext);
    }

    /**
     * 对数据库执行查询
     *
     * @param command      查询命令
     * @param database     数据库
     * @param requiresPost 如果命令需要POST而不是GET到influxdb，则为true
     * @param chunkSize    一个区块中要处理的QueryResults的数量
     * @param onNext       要为每个接收到的QueryResult调用的使用者
     */
    public void query(String command, String database, boolean requiresPost, int chunkSize, Consumer<QueryResult> onNext) {
        query(new Query(command, database, requiresPost), chunkSize, onNext);
    }

    /**
     * 对数据库执行查询
     *
     * @param query     查询参数 {@link Query}
     * @param chunkSize 一个区块中要处理的QueryResults的数量
     * @param onNext    要为每个接收到的QueryResult调用的使用者
     */
    public void query(Query query, int chunkSize, Consumer<QueryResult> onNext) {
        influxDB.query(query, chunkSize, onNext);
    }

    /**
     * 对数据库执行查询
     *
     * @param command    查询命令
     * @param database   数据库
     * @param chunkSize  一个区块中要处理的QueryResults的数量
     * @param onNext     要为每个接收到的QueryResult调用的使用者
     * @param onComplete 要调用以成功结束流的onComplete
     */
    public void query(String command, String database, int chunkSize, Consumer<QueryResult> onNext, Runnable onComplete) {
        query(command, database, false, chunkSize, onNext, onComplete);
    }

    /**
     * 对数据库执行查询
     *
     * @param command      查询命令
     * @param database     数据库
     * @param requiresPost 如果命令需要POST而不是GET到influxdb，则为true
     * @param chunkSize    一个区块中要处理的QueryResults的数量
     * @param onNext       要为每个接收到的QueryResult调用的使用者
     * @param onComplete   要调用以成功结束流的onComplete
     */
    public void query(String command, String database, boolean requiresPost, int chunkSize, Consumer<QueryResult> onNext, Runnable onComplete) {
        query(new Query(command, database, requiresPost), chunkSize, onNext, onComplete);
    }

    /**
     * 对数据库执行查询
     *
     * @param query      查询参数 {@link Query}
     * @param chunkSize  一个区块中要处理的QueryResults的数量
     * @param onNext     要为每个接收到的QueryResult调用的使用者
     * @param onComplete 要调用以成功结束流的onComplete
     */
    public void query(Query query, int chunkSize, Consumer<QueryResult> onNext, Runnable onComplete) {
        influxDB.query(query, chunkSize, onNext, onComplete);
    }

    /**
     * 对数据库执行查询
     *
     * @param command   查询命令
     * @param database  数据库
     * @param chunkSize 一个区块中要处理的QueryResults的数量
     * @param onNext    消费者为每个接收到的QueryResult调用；具有中断流式查询的功能
     */
    public void query(String command, String database, int chunkSize, BiConsumer<InfluxDB.Cancellable, QueryResult> onNext) {
        query(command, database, false, chunkSize, onNext);
    }

    /**
     * 对数据库执行查询
     *
     * @param command      查询命令
     * @param database     数据库
     * @param requiresPost 如果命令需要POST而不是GET到influxdb，则为true
     * @param chunkSize    一个区块中要处理的QueryResults的数量
     * @param onNext       消费者为每个接收到的QueryResult调用；具有中断流式查询的功能
     */
    public void query(String command, String database, boolean requiresPost, int chunkSize, BiConsumer<InfluxDB.Cancellable, QueryResult> onNext) {
        query(new Query(command, database, requiresPost), chunkSize, onNext);
    }

    /**
     * 对数据库执行查询
     *
     * @param query     查询参数 {@link Query}
     * @param chunkSize 一个区块中要处理的QueryResults的数量
     * @param onNext    消费者为每个接收到的QueryResult调用；具有中断流式查询的功能
     */
    public void query(Query query, int chunkSize, BiConsumer<InfluxDB.Cancellable, QueryResult> onNext) {
        influxDB.query(query, chunkSize, onNext);
    }

    /**
     * 对数据库执行查询
     *
     * @param command    查询命令
     * @param database   数据库
     * @param chunkSize  一个区块中要处理的QueryResults的数量
     * @param onNext     消费者为每个接收到的QueryResult调用；具有中断流式查询的功能
     * @param onComplete 要调用以成功结束流的onComplete
     */
    public void query(String command, String database, int chunkSize, BiConsumer<InfluxDB.Cancellable, QueryResult> onNext, Runnable onComplete) {
        query(command, database, false, chunkSize, onNext, onComplete);
    }

    /**
     * 对数据库执行查询
     *
     * @param command      查询命令
     * @param database     数据库
     * @param requiresPost 如果命令需要POST而不是GET到influxdb，则为true
     * @param chunkSize    一个区块中要处理的QueryResults的数量
     * @param onNext       消费者为每个接收到的QueryResult调用；具有中断流式查询的功能
     * @param onComplete   要调用以成功结束流的onComplete
     */
    public void query(String command, String database, boolean requiresPost, int chunkSize, BiConsumer<InfluxDB.Cancellable, QueryResult> onNext, Runnable onComplete) {
        query(new Query(command, database, requiresPost), chunkSize, onNext, onComplete);
    }

    /**
     * 对数据库执行查询
     *
     * @param query      查询参数 {@link Query}
     * @param chunkSize  一个区块中要处理的QueryResults的数量
     * @param onNext     消费者为每个接收到的QueryResult调用；具有中断流式查询的功能
     * @param onComplete 要调用以成功结束流的onComplete
     */
    public void query(Query query, int chunkSize, BiConsumer<InfluxDB.Cancellable, QueryResult> onNext, Runnable onComplete) {
        influxDB.query(query, chunkSize, onNext, onComplete);
    }

    /**
     * 对数据库执行查询
     *
     * @param command    查询命令
     * @param database   数据库
     * @param chunkSize  一个区块中要处理的QueryResults的数量
     * @param onNext     消费者为每个接收到的QueryResult调用；具有中断流式查询的功能
     * @param onComplete 要调用以成功结束流的onComplete
     * @param onFailure  错误处理的使用者
     */
    public void query(String command, String database, int chunkSize, BiConsumer<InfluxDB.Cancellable, QueryResult> onNext, Runnable onComplete, Consumer<Throwable> onFailure) {
        query(command, database, false, chunkSize, onNext, onComplete, onFailure);
    }

    /**
     * 对数据库执行查询
     *
     * @param command      查询命令
     * @param database     数据库
     * @param requiresPost 如果命令需要POST而不是GET到influxdb，则为true
     * @param chunkSize    一个区块中要处理的QueryResults的数量
     * @param onNext       消费者为每个接收到的QueryResult调用；具有中断流式查询的功能
     * @param onComplete   要调用以成功结束流的onComplete
     * @param onFailure    错误处理的使用者
     */
    public void query(String command, String database, boolean requiresPost, int chunkSize, BiConsumer<InfluxDB.Cancellable, QueryResult> onNext, Runnable onComplete, Consumer<Throwable> onFailure) {
        query(new Query(command, database, requiresPost), chunkSize, onNext, onComplete, onFailure);
    }

    /**
     * 对数据库执行查询
     *
     * @param query      查询参数 {@link Query}
     * @param chunkSize  一个区块中要处理的QueryResults的数量
     * @param onNext     消费者为每个接收到的QueryResult调用；具有中断流式查询的功能
     * @param onComplete 要调用以成功结束流的onComplete
     * @param onFailure  错误处理的使用者
     */
    public void query(Query query, int chunkSize, BiConsumer<InfluxDB.Cancellable, QueryResult> onNext, Runnable onComplete, Consumer<Throwable> onFailure) {
        influxDB.query(query, chunkSize, onNext, onComplete, onFailure);
    }

    /**
     * 对数据库执行查询
     *
     * @param command   查询命令
     * @param database  数据库
     * @param onSuccess 收到结果时要调用的使用者
     * @param onFailure 引发错误时要调用的使用者
     */
    public void query(String command, String database, Consumer<QueryResult> onSuccess, Consumer<Throwable> onFailure) {
        query(command, database, false, onSuccess, onFailure);
    }

    /**
     * 对数据库执行查询
     *
     * @param command      查询命令
     * @param database     数据库
     * @param requiresPost 如果命令需要POST而不是GET到influxdb，则为true
     * @param onSuccess    收到结果时要调用的使用者
     * @param onFailure    引发错误时要调用的使用者
     */
    public void query(String command, String database, boolean requiresPost, Consumer<QueryResult> onSuccess, Consumer<Throwable> onFailure) {
        query(new Query(command, database, requiresPost), onSuccess, onFailure);
    }

    /**
     * 对数据库执行查询
     *
     * @param query     查询参数 {@link Query}
     * @param onSuccess 收到结果时要调用的使用者
     * @param onFailure 引发错误时要调用的使用者
     */
    public void query(Query query, Consumer<QueryResult> onSuccess, Consumer<Throwable> onFailure) {
        influxDB.query(query, onSuccess, onFailure);
    }

    /**
     * 创建数据库
     *
     * @param databaseName 数据库名称
     */
    public void createDatabase(final String databaseName) {
        query("CREATE DATABASE", databaseName, true);
    }

    /**
     * 删除数据库
     *
     * @param databaseName 数据库名称
     */
    public void deleteDatabase(final String databaseName) {
        query("DROP DATABASE", databaseName, true);
    }


}
