/*
 * DO NOT ALTER OR REMOVE COPYRIGHT NOTICES OR THIS HEADER.
 *
 * Copyright (c) 2020-2030 郑庚伟 ZHENGGENGWEI (码匠君), <herodotus@aliyun.com> Licensed under the AGPL License
 *
 * This file is part of Herodotus Stirrup.
 *
 * Herodotus Stirrup is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published
 * by the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * Herodotus Stirrup is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <https://www.herodotus.vip>.
 */

package cn.herodotus.stirrup.tsdb.influxdb.configuration;

import cn.herodotus.stirrup.tsdb.influxdb.annotation.ConditionalOnInfluxdbEnabled;
import cn.herodotus.stirrup.tsdb.influxdb.core.InfluxdbTemplate;
import cn.herodotus.stirrup.tsdb.influxdb.properties.InfluxdbProperties;
import jakarta.annotation.PostConstruct;
import org.apache.commons.lang3.StringUtils;
import org.influxdb.InfluxDB;
import org.influxdb.InfluxDBFactory;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.boot.context.properties.EnableConfigurationProperties;
import org.springframework.context.annotation.Bean;
import org.springframework.context.annotation.Configuration;

/**
 * <p>Description: Influxdb SDK 模块配置 </p>
 *
 * @author : gengwei.zheng
 * @date : 2023/11/4 16:20
 */
@Configuration(proxyBeanMethods = false)
@ConditionalOnInfluxdbEnabled
@EnableConfigurationProperties(InfluxdbProperties.class)
public class TsdbInfluxdbConfiguration {

    private static final Logger log = LoggerFactory.getLogger(TsdbInfluxdbConfiguration.class);

    @PostConstruct
    public void postConstruct() {
        log.debug("[Herodotus] |- Module [Tsdb Influxdb] Auto Configure.");
    }

    @Bean
    public InfluxDB influxdb(InfluxdbProperties influxdbProperties) {

        InfluxDB influxdb;
        if (StringUtils.isNotBlank(influxdbProperties.getUsername()) && StringUtils.isNotBlank(influxdbProperties.getPassword())) {
            influxdb = InfluxDBFactory.connect(influxdbProperties.getUrl(), influxdbProperties.getUsername(), influxdbProperties.getPassword());
        } else {
            influxdb = InfluxDBFactory.connect(influxdbProperties.getUrl());
        }

        try {
            /*
             * 异步插入：enableBatch这里第一个是point的个数，第二个是时间，单位毫秒
             * point的个数和时间是联合使用的，如果满100条或者2000毫秒 满足任何一个条件就会发送一次写的请求。
             */
            influxdb.setDatabase(influxdbProperties.getDatabase())
                    .enableBatch(influxdbProperties.getActions(), influxdbProperties.getFlushDuration(), influxdbProperties.getFlushDurationTimeUnit());
        } catch (IllegalStateException e) {
            log.error("[Herodotus] |- Influxdb set database catch error.", e);
        } finally {
            influxdb.setRetentionPolicy(influxdbProperties.getRetentionPolicy());
        }
        influxdb.setLogLevel(InfluxDB.LogLevel.BASIC);
        return influxdb;
    }

    @Bean
    public InfluxdbTemplate influxdbTemplate(InfluxDB influxDB, InfluxdbProperties influxdbProperties) {
        InfluxdbTemplate influxdbTemplate = new InfluxdbTemplate(influxDB, influxdbProperties);
        log.trace("[Herodotus] |- Bean [Influxdb Template] Auto Configure.");
        return influxdbTemplate;
    }
}
