/*
 * DO NOT ALTER OR REMOVE COPYRIGHT NOTICES OR THIS HEADER.
 *
 * Copyright (c) 2020-2030 郑庚伟 ZHENGGENGWEI (码匠君), <herodotus@aliyun.com> Licensed under the AGPL License
 *
 * This file is part of Herodotus Cloud.
 *
 * Herodotus Cloud is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published
 * by the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * Herodotus Cloud is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <https://www.herodotus.vip>.
 */

package cn.herodotus.stirrup.transform.emqx.utils;

import cn.herodotus.stirrup.transform.emqx.domain.SystemTopic;
import org.apache.commons.lang3.StringUtils;
import org.springframework.messaging.Message;

/**
 * <p>Description: Emqx 消息处理工具 </p>
 *
 * @author : gengwei.zheng
 * @date : 2023/11/22 12:21
 */
public class EmqxMessageUtils {

    /**
     * 获取 Emqx 消息主题
     *
     * @param message Spring Integration 消息对象 {@link Message}
     * @return mqtt_receivedTopic 消息头的值
     */
    public static String getTopic(Message<?> message) {
        Object mqttReceivedTopic = message.getHeaders().get("mqtt_receivedTopic");
        return String.valueOf(mqttReceivedTopic);
    }

    /**
     * 获取 Emqx 消息负载
     *
     * @param message Spring Integration 消息对象 {@link Message}
     * @return byte[] 类型消息内容
     */
    public static byte[] getPayload(Message<?> message) {
        return (byte[]) message.getPayload();
    }

    /**
     * 获取 Emqx 消息时间戳
     *
     * @param message Spring Integration 消息对象 {@link Message}
     * @return 发送消息的时间戳
     */
    public static Long getTimestamp(Message<?> message) {
        return message.getHeaders().getTimestamp();
    }

    /**
     * 当前主题是否是作为标记的主题
     *
     * @param topic Emqx 系统主题
     * @return true 是标记主题，false 不是标记主题
     */
    public static boolean isStampTopic(String topic) {
        return StringUtils.equals(topic, "$SYS/brokers");
    }

    /**
     * 当前主题是否是作为标记的主题
     *
     * @param message Spring Integration 消息对象 {@link Message}
     * @return true 是标记主题，false 不是标记主题
     */
    public static boolean isStampTopic(Message<?> message) {
        String topic = getTopic(message);
        return isStampTopic(topic);
    }

    /**
     * 当前主题名称是否是版本信息
     *
     * @param systemTopic Emqx 系统主题 {@link SystemTopic}
     * @return true 是标记主题，false 不是标记主题
     */
    public static boolean isVersion(SystemTopic systemTopic) {
        return StringUtils.equals(systemTopic.topicName(), "version");
    }
}
