/*
 * DO NOT ALTER OR REMOVE COPYRIGHT NOTICES OR THIS HEADER.
 *
 * Copyright (c) 2020-2030 郑庚伟 ZHENGGENGWEI (码匠君), <herodotus@aliyun.com> Licensed under the AGPL License
 *
 * This file is part of Herodotus Cloud.
 *
 * Herodotus Cloud is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published
 * by the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * Herodotus Cloud is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <https://www.herodotus.vip>.
 */

package cn.herodotus.stirrup.transform.emqx.transformer;

import cn.herodotus.stirrup.core.definition.constants.SymbolConstants;
import cn.herodotus.stirrup.core.foundation.json.jackson2.utils.Jackson2Utils;
import cn.herodotus.stirrup.core.event.domain.emqx.SystemClientConnected;
import cn.herodotus.stirrup.core.event.domain.emqx.SystemClientDisconnected;
import cn.herodotus.stirrup.core.event.domain.emqx.SystemClientSubscribed;
import cn.herodotus.stirrup.core.event.domain.emqx.SystemClientUnsubscribed;
import cn.herodotus.stirrup.core.event.event.emqx.SystemClientConnectedEvent;
import cn.herodotus.stirrup.core.event.event.emqx.SystemClientDisconnectedEvent;
import cn.herodotus.stirrup.core.event.event.emqx.SystemClientSubscribedEvent;
import cn.herodotus.stirrup.core.event.event.emqx.SystemClientUnsubscribedEvent;
import cn.herodotus.stirrup.transform.emqx.utils.EmqxMessageUtils;
import org.apache.commons.lang3.StringUtils;
import org.dromara.hutool.core.text.StrUtil;
import org.springframework.context.ApplicationEvent;
import org.springframework.integration.transformer.AbstractTransformer;
import org.springframework.messaging.Message;

/**
 * <p>Description: Emqx 系统主题消息转换成 Event 转换器 </p>
 *
 * @author : gengwei.zheng
 * @date : 2023/12/9 14:35
 */
public class SystemClientByteArrayToEventTransformer extends AbstractTransformer {

    @Override
    protected Object doTransform(Message<?> message) {
        String topic = parseTopic(message);
        byte[] payload = EmqxMessageUtils.getPayload(message);

        return convert(topic, StrUtil.utf8Str(payload));
    }

    /**
     * 将 String 类型数据转换为指定类型的实体
     *
     * @param data      String 类型数据
     * @param beanClass 实体类
     * @param <T>       实体类型
     * @return 指定的实体
     */
    private <T> T getBean(String data, Class<T> beanClass) {
        return Jackson2Utils.toObject(data, beanClass);
    }

    /**
     * 解析 Emqx Message 中的 Topic 头信息
     *
     * @param message {@link Message}
     * @return emqx 系统主题 topic 名称
     */
    private String parseTopic(Message<?> message) {
        String topic = EmqxMessageUtils.getTopic(message);
        String[] elements = StringUtils.split(topic, SymbolConstants.FORWARD_SLASH);
        return elements[elements.length - 1];
    }

    private ApplicationEvent convert(String type, String data) {
        return switch (type) {
            case "connected" -> new SystemClientConnectedEvent(getBean(data, SystemClientConnected.class));
            case "disconnected" -> new SystemClientDisconnectedEvent(getBean(data, SystemClientDisconnected.class));
            case "subscribed" -> new SystemClientSubscribedEvent(getBean(data, SystemClientSubscribed.class));
            default -> new SystemClientUnsubscribedEvent(getBean(data, SystemClientUnsubscribed.class));
        };
    }
}
