/*
 * DO NOT ALTER OR REMOVE COPYRIGHT NOTICES OR THIS HEADER.
 *
 * Copyright (c) 2020-2030 郑庚伟 ZHENGGENGWEI (码匠君), <herodotus@aliyun.com> Licensed under the AGPL License
 *
 * This file is part of Herodotus Cloud.
 *
 * Herodotus Cloud is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published
 * by the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * Herodotus Cloud is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <https://www.herodotus.vip>.
 */

package cn.herodotus.stirrup.transform.emqx.outbound;

import cn.herodotus.stirrup.transform.emqx.converter.AbstractInfluxdbPointConverter;
import cn.herodotus.stirrup.transform.emqx.converter.Influxdb2Converter;
import cn.herodotus.stirrup.transform.emqx.converter.MessageToRecordConverter;
import cn.herodotus.stirrup.tsdb.influxdb2.pool.Influxdb2ClientObjectPool;
import com.influxdb.client.InfluxDBClient;
import com.influxdb.client.WriteApiBlocking;
import com.influxdb.client.domain.WritePrecision;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.integration.handler.AbstractMessageHandler;
import org.springframework.messaging.Message;

/**
 * <p>Description: Influxdb 数据 Integration Outbound 处理器 </p>
 *
 * @author : gengwei.zheng
 * @date : 2023/11/10 22:01
 */
public class Influxdb2StoringMessageHandler extends AbstractMessageHandler {

    private static final Logger log = LoggerFactory.getLogger(Influxdb2StoringMessageHandler.class);

    private Influxdb2Converter<?> influxdb2Converter;
    private final Influxdb2ClientObjectPool influxdb2ClientObjectPool;

    private enum WriteMode {
        RECORD, POINT, POJO
    }

    public Influxdb2StoringMessageHandler(Influxdb2ClientObjectPool influxdb2ClientObjectPool) {
        this(influxdb2ClientObjectPool, new MessageToRecordConverter());
    }

    public Influxdb2StoringMessageHandler(Influxdb2ClientObjectPool influxdb2ClientObjectPool, Influxdb2Converter<?> influxdb2Converter) {
        this.influxdb2ClientObjectPool = influxdb2ClientObjectPool;
        this.influxdb2Converter = influxdb2Converter;
    }

    public void setInfluxdb2Converter(Influxdb2Converter<?> influxdb2Converter) {
        this.influxdb2Converter = influxdb2Converter;
    }

    @Override
    protected void handleMessageInternal(Message<?> message) {

        Object payload = message.getPayload();

        InfluxDBClient influxDBClient = influxdb2ClientObjectPool.get();
        WriteApiBlocking writeApi = influxDBClient.getWriteApiBlocking();

        switch (getWriteMode()) {
            case POINT -> writeApi.writePoint(((AbstractInfluxdbPointConverter) influxdb2Converter).convert(payload));
            case RECORD ->
                    writeApi.writeRecord(WritePrecision.NS, ((MessageToRecordConverter) influxdb2Converter).convert(payload));
            default -> writeApi.writeMeasurement(WritePrecision.NS, influxdb2Converter.convert(payload));
        }

        log.debug("[Herodotus] |- Iot integration write message {} to influxdb2.", message);

        influxdb2ClientObjectPool.close(influxDBClient);
    }

    private WriteMode getWriteMode() {
        if (influxdb2Converter instanceof MessageToRecordConverter) {
            return WriteMode.RECORD;
        } else if (influxdb2Converter instanceof AbstractInfluxdbPointConverter) {
            return WriteMode.POINT;
        } else {
            return WriteMode.POJO;
        }
    }

}
