/*
 * DO NOT ALTER OR REMOVE COPYRIGHT NOTICES OR THIS HEADER.
 *
 * Copyright (c) 2020-2030 郑庚伟 ZHENGGENGWEI (码匠君), <herodotus@aliyun.com> Licensed under the AGPL License
 *
 * This file is part of Herodotus Cloud.
 *
 * Herodotus Cloud is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published
 * by the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * Herodotus Cloud is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <https://www.herodotus.vip>.
 */

package cn.herodotus.stirrup.transform.emqx.config;

import cn.herodotus.stirrup.message.core.constants.Channels;
import cn.herodotus.stirrup.core.event.event.emqx.*;
import cn.herodotus.stirrup.transform.emqx.transformer.WebhookMapToEventTransformer;
import cn.herodotus.stirrup.transform.emqx.annotation.ConditionalOnEmqxWebhookEvent;
import cn.herodotus.stirrup.transform.emqx.storage.WebhookDemoListener;
import jakarta.annotation.PostConstruct;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.context.annotation.Bean;
import org.springframework.context.annotation.Configuration;
import org.springframework.core.ResolvableType;
import org.springframework.http.HttpMethod;
import org.springframework.http.HttpStatus;
import org.springframework.integration.channel.QueueChannel;
import org.springframework.integration.dsl.IntegrationFlow;
import org.springframework.integration.dsl.MessageChannels;
import org.springframework.integration.event.outbound.ApplicationEventPublishingMessageHandler;
import org.springframework.integration.http.dsl.Http;
import org.springframework.messaging.MessageChannel;

import java.util.LinkedHashMap;
import java.util.Map;

/**
 * <p>Description: Emqx Webhook Client相关事件消息转 ApplicationEvent Flow 配置 </p>
 * <p>
 * 通过 Emqx Webhook 方式获取到的 Emqx 客户端订阅事件消息。
 * $events/message_delivered - 消息投递事件 {@link WebhookMessageDeliveredEvent}
 * $events/message_acked - 消息确认事件 {@link WebhookMessageAckedEvent}
 * $events/message_dropped - 消息在转发的过程中被丢弃事件 {@link WebhookMessageDroppedEvent}
 * $events/delivery_dropped	- 消息在投递的过程中被丢弃事件 {@link WebhookDeliveryDroppedEvent}
 * $events/client_connected	- 客户端连接成功事件 {@link WebhookClientConnectedEvent}
 * $events/client_disconnected- 客户端连接断开事件 {@link WebhookClientDisconnectedEvent}
 * $events/client_connack - 连接确认事件 {@link WebhookClientConnectAckEvent}
 * $events/client_check_authz_complete - 鉴权完成事件 {@link WebhookClientCheckAuthenticationCompleteEvent}
 * $events/session_subscribed - 客户端订阅成功事件 {@link WebhookSessionSubscribedEvent}
 * $events/session_unsubscribed- 客户端取消订阅成功事件 {@link WebhookSessionUnsubscribedEvent}
 * <p>
 * 这些消息内容为 JSON 类型，将其转成对应的实体，然后将这些实体放入到对应的 ApplicationEvent 发送出去。
 * 相关应用代码，进需要监听具体的 ApplicationEvent 即可以完成相关的实现。进一步简化操作。
 *
 * @author : gengwei.zheng
 * @date : 2023/11/28 18:05
 */
@Configuration(proxyBeanMethods = false)
@ConditionalOnEmqxWebhookEvent
public class EmqxWebhookToEventFlowConfiguration {

    private static final Logger log = LoggerFactory.getLogger(EmqxWebhookToEventFlowConfiguration.class);

    @PostConstruct
    public void postConstruct() {
        log.debug("[Herodotus] |- SDK [Emqx Webhook To Event Flow] Auto Configure.");
    }

    /**
     * Emqx Webhook Http 入站通道
     *
     * @return {@link QueueChannel}
     */
    @Bean(Channels.EMQX_DEFAULT_WEBHOOK_HTTP_INBOUND_CHANNEL)
    public MessageChannel emqxWebhookHttpInboundChannel() {
        return MessageChannels.direct().getObject();
    }

    /**
     * Spring Integration ApplicationEvent 出站消息配置
     *
     * @return {@link ApplicationEventPublishingMessageHandler}
     */
    @Bean
    public ApplicationEventPublishingMessageHandler emqxWebhookEventPublishingMessageHandler() {
        ApplicationEventPublishingMessageHandler handler = new ApplicationEventPublishingMessageHandler();
        // 设置该值的作用是将具体的 ApplicationEvent 作为 Payload，而不会将其包装成 MessageEvent
        handler.setPublishPayload(true);
        return handler;
    }

    /**
     * Emqx Webhook 事件转成系统 Event Flow
     *
     * @param emqxWebhookEventPublishingMessageHandler {@link ApplicationEventPublishingMessageHandler}
     * @return {@link IntegrationFlow}
     */
    @Bean
    public IntegrationFlow emqxWebhookToEventFlow(ApplicationEventPublishingMessageHandler emqxWebhookEventPublishingMessageHandler,
                                                  @Qualifier(Channels.EMQX_DEFAULT_WEBHOOK_HTTP_INBOUND_CHANNEL) MessageChannel emqxWebhookHttpInboundChannel) {
        return IntegrationFlow.from(Http.inboundChannelAdapter("/emqx/webhook")
                        .requestMapping(m -> m.methods(HttpMethod.POST))
                        .requestPayloadType(ResolvableType.forClass(Map.class, LinkedHashMap.class))
                        .statusCodeFunction(s -> HttpStatus.OK))
                .channel(emqxWebhookHttpInboundChannel)
                .transform(new WebhookMapToEventTransformer())
                .channel(MessageChannels.direct(Channels.EMQX_DEFAULT_EVENT_OUTBOUND_CHANNEL))
                .handle(emqxWebhookEventPublishingMessageHandler)
                .get();
    }

    @Bean
    public WebhookDemoListener webhookDemoListener() {
        return new WebhookDemoListener();
    }
}
