/*
 * DO NOT ALTER OR REMOVE COPYRIGHT NOTICES OR THIS HEADER.
 *
 * Copyright (c) 2020-2030 郑庚伟 ZHENGGENGWEI (码匠君), <herodotus@aliyun.com> Licensed under the AGPL License
 *
 * This file is part of Herodotus Cloud.
 *
 * Herodotus Cloud is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published
 * by the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * Herodotus Cloud is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <https://www.herodotus.vip>.
 */

package cn.herodotus.stirrup.transform.emqx.config;

import cn.herodotus.stirrup.message.core.constants.Channels;
import cn.herodotus.stirrup.transform.emqx.aggregator.MonitorAggregatingMessageCorrelationIdFactory;
import cn.herodotus.stirrup.transform.emqx.outbound.MonitorAggregatingMessageHandler;
import cn.herodotus.stirrup.transform.emqx.outbound.Influxdb2StoringMessageHandler;
import cn.herodotus.stirrup.message.mqtt.annotation.ConditionalOnMqttEnabled;
import cn.herodotus.stirrup.message.mqtt.config.MessageMqttConfiguration;
import cn.herodotus.stirrup.transform.emqx.converter.EmqxStatsIndicatorConverter;
import cn.herodotus.stirrup.tsdb.influxdb2.annotation.ConditionalOnInfluxdb2Enabled;
import cn.herodotus.stirrup.tsdb.influxdb2.configuration.TsdbInfluxdb2Configuration;
import cn.herodotus.stirrup.tsdb.influxdb2.pool.Influxdb2ClientObjectPool;
import jakarta.annotation.PostConstruct;
import org.eclipse.paho.mqttv5.client.IMqttAsyncClient;
import org.eclipse.paho.mqttv5.client.MqttConnectionOptions;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.context.annotation.Bean;
import org.springframework.context.annotation.Configuration;
import org.springframework.context.annotation.Import;
import org.springframework.integration.dsl.IntegrationFlow;
import org.springframework.integration.dsl.MessageChannels;
import org.springframework.integration.mqtt.core.ClientManager;
import org.springframework.integration.mqtt.inbound.Mqttv5PahoMessageDrivenChannelAdapter;
import org.springframework.messaging.MessageChannel;

/**
 * <p>Description: Emqx 系统主题监控数据转储至 Influxdb2配置。 </p>
 * <p>
 * 原计划是采用 Spring Integration 消息聚合实现对 Emqx 系统主题的监控的转储以及后续的展现。
 * 后面发现使用 Prometheus 搭建监控体系，实现 Emqx 的监控以及监控数据的存储方案更优、更便捷。
 * 目前，还是保留这种基于 Spring Integration 消息聚合实现方式，当做一个 Spring Integration 消息聚合的一个案例。代码不在实际使用。
 *
 * @author : gengwei.zheng
 * @date : 2023/11/22 13:02
 */
@Configuration(proxyBeanMethods = false)
@ConditionalOnInfluxdb2Enabled
@ConditionalOnMqttEnabled
@Import({
        MessageMqttConfiguration.class,
        TsdbInfluxdb2Configuration.class
})
public class EmqxSystemTopicToInfluxdb2FlowConfiguration {

    private static final Logger log = LoggerFactory.getLogger(EmqxSystemTopicToInfluxdb2FlowConfiguration.class);

    private static final String[] EMQX_MONITOR_TOPICS = new String[]{"$SYS/#"};

    @PostConstruct
    public void postConstruct() {
        log.debug("[Herodotus] |- SDK [Emqx System Topic To Influxdb2 Flow] Auto Configure.");
    }

    @Bean(Channels.EMQX_DEFAULT_MONITOR_MQTT_INBOUND_CHANNEL)
    public MessageChannel emqxMonitorInboundChannel() {
        return MessageChannels.publishSubscribe().getObject();
    }

    @Bean
    public MonitorAggregatingMessageCorrelationIdFactory monitorAggregatingMessageCorrelationIdFactory() {
        MonitorAggregatingMessageCorrelationIdFactory factory = new MonitorAggregatingMessageCorrelationIdFactory();
        log.trace("[Herodotus] |- Bean [Monitor Aggregating Message Correlation Id Factory] Auto Configure.");
        return factory;
    }

    @Bean
    public IntegrationFlow emqxSysTopicToInfluxdbFlow(ClientManager<IMqttAsyncClient, MqttConnectionOptions> clientManager,
                                                      @Qualifier(Channels.EMQX_DEFAULT_MONITOR_MQTT_INBOUND_CHANNEL) MessageChannel messageChannel,
                                                      MonitorAggregatingMessageCorrelationIdFactory monitorAggregatingMessageCorrelationIdFactory,
                                                      Influxdb2ClientObjectPool influxdb2ClientObjectPool) {
        Mqttv5PahoMessageDrivenChannelAdapter messageProducer =
                new Mqttv5PahoMessageDrivenChannelAdapter(clientManager, EMQX_MONITOR_TOPICS);
        messageProducer.setPayloadType(String.class);
        messageProducer.setManualAcks(false);
        messageProducer.setOutputChannel(messageChannel);

        return IntegrationFlow.from(messageProducer)
                .handle(new MonitorAggregatingMessageHandler(monitorAggregatingMessageCorrelationIdFactory))
                .handle(new Influxdb2StoringMessageHandler(influxdb2ClientObjectPool, new EmqxStatsIndicatorConverter()))
                .get();
    }
}
