/*
 * DO NOT ALTER OR REMOVE COPYRIGHT NOTICES OR THIS HEADER.
 *
 * Copyright (c) 2020-2030 郑庚伟 ZHENGGENGWEI (码匠君), <herodotus@aliyun.com> Licensed under the AGPL License
 *
 * This file is part of Herodotus Cloud.
 *
 * Herodotus Cloud is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published
 * by the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * Herodotus Cloud is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <https://www.herodotus.vip>.
 */

package cn.herodotus.stirrup.transform.emqx.config;

import cn.herodotus.stirrup.message.core.constants.Channels;
import cn.herodotus.stirrup.core.event.event.emqx.SystemClientConnectedEvent;
import cn.herodotus.stirrup.core.event.event.emqx.SystemClientDisconnectedEvent;
import cn.herodotus.stirrup.core.event.event.emqx.SystemClientSubscribedEvent;
import cn.herodotus.stirrup.core.event.event.emqx.SystemClientUnsubscribedEvent;
import cn.herodotus.stirrup.transform.emqx.transformer.SystemClientByteArrayToEventTransformer;
import cn.herodotus.stirrup.message.mqtt.annotation.ConditionalOnMqttEnabled;
import cn.herodotus.stirrup.message.mqtt.config.MessageMqttConfiguration;
import cn.herodotus.stirrup.transform.emqx.annotation.ConditionalOnEmqxSystemTopicEvent;
import jakarta.annotation.PostConstruct;
import org.eclipse.paho.mqttv5.client.IMqttAsyncClient;
import org.eclipse.paho.mqttv5.client.MqttConnectionOptions;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.context.annotation.Bean;
import org.springframework.context.annotation.Configuration;
import org.springframework.context.annotation.Import;
import org.springframework.integration.dsl.IntegrationFlow;
import org.springframework.integration.dsl.MessageChannels;
import org.springframework.integration.event.outbound.ApplicationEventPublishingMessageHandler;
import org.springframework.integration.mqtt.core.ClientManager;
import org.springframework.integration.mqtt.inbound.Mqttv5PahoMessageDrivenChannelAdapter;
import org.springframework.messaging.MessageChannel;

/**
 * <p>Description: Emqx 系统主题中 Client 相关监控内容转 ApplicationEvent 配置 </p>
 * <p>
 * EMQX 周期性发布自身运行状态、消息统计、客户端上下线事件到以 $SYS/ 开头系统主题。
 * <p>
 * SystemClient 主要对应 Emqx 客户端相关事件的系统主题。
 * · $SYS/brokers/${node}/clients/connected - 上线事件。当任意客户端上线时，EMQX 就会发布该主题的消息 {@link SystemClientConnectedEvent}
 * · $SYS/brokers/${node}/clients/disconnected - 上下线事件。当任意客户端下线时，EMQX 就会发布该主题的消息 {@link SystemClientDisconnectedEvent}
 * · $SYS/brokers/${node}/clients/subscribed - 订阅事件。当任意客户端订阅主题时，EMQX 就会发布该主题的消息 {@link SystemClientSubscribedEvent}
 * · $SYS/brokers/${node}/clients/unsubscribed - 取消订阅事件。当任意客户端取消订阅主题时，EMQX 就会发布该主题的消息 {@link SystemClientUnsubscribedEvent}
 *
 * @author : gengwei.zheng
 * @date : 2023/12/5 16:02
 */
@Configuration(proxyBeanMethods = false)
@ConditionalOnMqttEnabled
@ConditionalOnEmqxSystemTopicEvent
@Import({
        MessageMqttConfiguration.class,
})
public class EmqxSystemClientToEventFlowConfiguration {

    private static final Logger log = LoggerFactory.getLogger(EmqxSystemClientToEventFlowConfiguration.class);

    private static final String[] EMQX_MONITOR_TOPICS = new String[]{"$SYS/brokers/+/clients/#"};

    @PostConstruct
    public void postConstruct() {
        log.debug("[Herodotus] |- SDK [Emqx $sys/client To Event Flow] Auto Configure.");
    }

    @Bean(Channels.EMQX_DEFAULT_MONITOR_MQTT_INBOUND_CHANNEL)
    public MessageChannel emqxMonitorInboundChannel() {
        return MessageChannels.publishSubscribe().getObject();
    }

    @Bean
    public ApplicationEventPublishingMessageHandler emqxSystemClientEventPublishingMessageHandler() {
        ApplicationEventPublishingMessageHandler handler = new ApplicationEventPublishingMessageHandler();
        // 设置该值的作用是将具体的 ApplicationEvent 作为 Payload，而不会将其包装成 MessageEvent
        handler.setPublishPayload(true);
        return handler;
    }

    @Bean
    public IntegrationFlow emqxSystemClientToEventFlow(ClientManager<IMqttAsyncClient, MqttConnectionOptions> clientManager,
                                                       ApplicationEventPublishingMessageHandler emqxSystemClientEventPublishingMessageHandler,
                                                       @Qualifier(Channels.EMQX_DEFAULT_MONITOR_MQTT_INBOUND_CHANNEL) MessageChannel emqxMonitorInboundChannel) {
        Mqttv5PahoMessageDrivenChannelAdapter messageProducer =
                new Mqttv5PahoMessageDrivenChannelAdapter(clientManager, EMQX_MONITOR_TOPICS);
        messageProducer.setPayloadType(String.class);
        messageProducer.setManualAcks(false);
        messageProducer.setOutputChannel(emqxMonitorInboundChannel);

        return IntegrationFlow.from(messageProducer)
                .transform(new SystemClientByteArrayToEventTransformer())
                .channel(MessageChannels.direct(Channels.EMQX_DEFAULT_EVENT_OUTBOUND_CHANNEL))
                .handle(emqxSystemClientEventPublishingMessageHandler)
                .get();
    }
}
