/*
 * DO NOT ALTER OR REMOVE COPYRIGHT NOTICES OR THIS HEADER.
 *
 * Copyright (c) 2020-2030 郑庚伟 ZHENGGENGWEI (码匠君), <herodotus@aliyun.com> Licensed under the AGPL License
 *
 * This file is part of Herodotus Cloud.
 *
 * Herodotus Cloud is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published
 * by the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * Herodotus Cloud is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <https://www.herodotus.vip>.
 */

package cn.herodotus.stirrup.transform.emqx.aggregator;

import cn.herodotus.stirrup.core.definition.constants.SymbolConstants;
import cn.herodotus.stirrup.transform.emqx.domain.StatsIndicator;
import cn.herodotus.stirrup.transform.emqx.domain.SystemTopic;
import cn.herodotus.stirrup.transform.emqx.enums.SystemTopicCategory;
import cn.herodotus.stirrup.transform.emqx.utils.EmqxMessageUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.math.NumberUtils;
import org.dromara.hutool.core.text.StrUtil;
import org.springframework.integration.aggregator.MessageGroupProcessor;
import org.springframework.integration.store.MessageGroup;
import org.springframework.messaging.Message;

import java.time.Instant;
import java.time.LocalDateTime;
import java.util.Collection;

/**
 * <p>Description: Emqx 监控消息聚合分组处理器 </p>
 *
 * @author : gengwei.zheng
 * @date : 2023/11/13 12:26
 */
public class MonitorAggregatingMessageGroupProcessor implements MessageGroupProcessor {

    private String node;

    private String version;

    private Instant uptime;

    private LocalDateTime datetime;

    private StatsIndicator stats;

    @Override
    public Object processMessageGroup(MessageGroup group) {

        stats = new StatsIndicator();

        analysis(group.getMessages());

        return stats;
    }

    private void setNode(String node) {
        stats.setNode(node);
    }

    private void setVersion(String version) {
        stats.setVersion(version);
    }

    private void analysis(Collection<Message<?>> messages) {
        messages.forEach(this::analysis);
    }

    private void analysis(Message<?> message) {
        String topic = EmqxMessageUtils.getTopic(message);
        byte[] payload = EmqxMessageUtils.getPayload(message);

        SystemTopic systemTopic = resolvingTopic(topic);

        switch (systemTopic.systemTopicCategory()) {
            case INFO:
                if (EmqxMessageUtils.isVersion(systemTopic)) {
                    setVersion(StrUtil.utf8Str(payload));
                }
                stats.setTime(Instant.ofEpochMilli(EmqxMessageUtils.getTimestamp(message)));
                break;
            case STATS:
                processStats(systemTopic.topicName(), NumberUtils.toInt(StrUtil.utf8Str(payload)));
                break;
            default:
                break;
        }
    }

    private SystemTopic resolvingTopic(String topic) {
        int count = StringUtils.countMatches(String.valueOf(topic), SymbolConstants.FORWARD_SLASH);

        return switch (count) {
            case 1 -> new SystemTopic(SystemTopicCategory.CLUSTER, "", topic);
            case 3 -> {
                String[] elements = StringUtils.split(topic, SymbolConstants.FORWARD_SLASH);
                setNode(elements[2]);
                yield new SystemTopic(SystemTopicCategory.INFO, elements[3], topic);
            }
            default -> {
                String[] elements = StringUtils.split(topic, SymbolConstants.FORWARD_SLASH, 5);
                yield new SystemTopic(SystemTopicCategory.valueOf(StringUtils.upperCase(elements[3])), elements[4], topic);
            }
        };
    }

    private void processStats(String category, Integer payload) {
        switch (category) {
            case "connections/count":
                stats.setConnectionsCount(payload);
                break;
            case "connections/max":
                stats.setConnectionsMax(payload);
                break;
            case "suboptions/count":
                stats.setSubOptionsCount(payload);
                break;
            case "suboptions/max":
                stats.setSubOptionsMax(payload);
                break;
            case "subscribers/count":
                stats.setSubscribersCount(payload);
                break;
            case "subscribers/max":
                stats.setSubscribersMax(payload);
                break;
            case "subscriptions/count":
                stats.setSubscriptionsCount(payload);
                break;
            case "subscriptions/max":
                stats.setSubscriptionsMax(payload);
                break;
            case "subscriptions/shared/count":
                stats.setSubscriptionsSharedCount(payload);
                break;
            case "subscriptions/shared/max":
                stats.setSubscriptionsSharedMax(payload);
                break;
            case "topics/count":
                stats.setTopicsCount(payload);
                break;
            case "topics/max":
                stats.setTopicsMax(payload);
                break;
            case "routes/count":
                stats.setRoutesCount(payload);
                break;
            case "routes/max":
                stats.setRoutesMax(payload);
                break;
            default:
                break;
        }
    }
}
