/*
 * DO NOT ALTER OR REMOVE COPYRIGHT NOTICES OR THIS HEADER.
 *
 * Copyright (c) 2020-2030 郑庚伟 ZHENGGENGWEI (码匠君), <herodotus@aliyun.com> Licensed under the AGPL License
 *
 * This file is part of Herodotus Cloud.
 *
 * Herodotus Cloud is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published
 * by the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * Herodotus Cloud is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <https://www.herodotus.vip>.
 */

package cn.herodotus.stirrup.transform.emqx.aggregator;

import cn.herodotus.stirrup.transform.emqx.utils.EmqxMessageUtils;
import org.springframework.integration.aggregator.CorrelationStrategy;
import org.springframework.messaging.Message;

/**
 * <p>Description: Emqx 监控消息数据集合相关性策略 </p>
 * <p>
 * Emqx 监控各类消息中，没有 CORRELATION_ID, SEQUENCE_NUMBER, SEQUENCE_SIZE 等头信息，以及可以作为聚合的标志性信息。需要自己指定一定的策略，保证一系列信息的相关性。
 * Emqx 监控各类监控信息，可以通过订阅系统主题获取。但是订阅到的各类指标信息时间戳是不同的，所以无法使用时间戳作为关联性信息。
 * 通过观察，Emqx 监控数据在每个发布周期内，最后一个消息总是来源于主题 "$SYS/brokers"。所以当前就以主题 "$SYS/brokers" 为标志，当接收到主题 "$SYS/brokers" 的消息，就把之前的所有的监控消息聚合。
 * <p>
 * 注意：第一次连接 Emqx 会接收到部分系统主题信息，但是不完整，所以这部分信息应该被排除。
 *
 * @author : gengwei.zheng
 * @date : 2023/11/11 16:24
 */
public class MonitorAggregatingMessageCorrelationStrategy implements CorrelationStrategy {

    private final MonitorAggregatingMessageCorrelationIdFactory correlationIdFactory;

    public MonitorAggregatingMessageCorrelationStrategy(MonitorAggregatingMessageCorrelationIdFactory correlationIdFactory) {
        this.correlationIdFactory = correlationIdFactory;
    }

    @Override
    public Object getCorrelationKey(Message<?> message) {

        String correlationId = this.correlationIdFactory.get();
        // 如果发现某条消息来自于 $SYS/brokers 主题，则认为当前监控周期结束，那么就清除当前 correlationId
        if (EmqxMessageUtils.isStampTopic(message)) {
            this.correlationIdFactory.erase();
        }

        return correlationId;
    }
}
