/*
 * DO NOT ALTER OR REMOVE COPYRIGHT NOTICES OR THIS HEADER.
 *
 * Copyright (c) 2020-2030 郑庚伟 ZHENGGENGWEI (码匠君), <herodotus@aliyun.com> Licensed under the AGPL License
 *
 * This file is part of Herodotus Cloud.
 *
 * Herodotus Cloud is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published
 * by the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * Herodotus Cloud is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <https://www.herodotus.vip>.
 */

package cn.herodotus.stirrup.transform.emqx.aggregator;

import cn.herodotus.stirrup.cache.caffeine.enhance.CaffeineNeverExpire;
import com.github.benmanes.caffeine.cache.Cache;
import com.github.benmanes.caffeine.cache.Caffeine;
import org.apache.commons.lang3.StringUtils;

import java.time.Instant;

/**
 * <p>Description: Emqx 监控消息聚合相关性标记 </p>
 * <p>
 * 利用 Caffeine 本地缓存，缓存一个 Emqx 监控周期内的 CorrelationId。没有就创建个新的，有就直接从缓存中读取
 *
 * @author : gengwei.zheng
 * @date : 2023/11/11 13:39
 */
public class MonitorAggregatingMessageCorrelationIdFactory {

    private static final String AGGREGATING_MESSAGE_CORRELATION_ID = "emqx_monitor_aggregating_correlation_id";

    private final Cache<String, String> cache;

    public MonitorAggregatingMessageCorrelationIdFactory() {
        this.cache = Caffeine.newBuilder().expireAfter(new CaffeineNeverExpire<>()).build();
    }

    private String readFromCache() {
        return cache.getIfPresent(AGGREGATING_MESSAGE_CORRELATION_ID);
    }

    private void writeToCache(String value) {
        cache.put(AGGREGATING_MESSAGE_CORRELATION_ID, value);
    }

    private String create() {
        String correlationId = "STAMP-" + Instant.now().toEpochMilli();
        writeToCache(correlationId);
        return correlationId;
    }

    public boolean absent() {
        String correlationId = readFromCache();
        return StringUtils.isBlank(correlationId);
    }

    public String get() {
        String correlationId = readFromCache();
        if (StringUtils.isEmpty(correlationId)) {
            correlationId = create();
        }
        return correlationId;
    }

    public void erase() {
        cache.invalidate(AGGREGATING_MESSAGE_CORRELATION_ID);
    }
}
