/*
 * DO NOT ALTER OR REMOVE COPYRIGHT NOTICES OR THIS HEADER.
 *
 * Copyright 2020-2030 ZHENGGENGWEI<码匠君>. All rights reserved.
 *
 *    Author: ZHENGGENGWEI<码匠君>
 *    Contact: <herodotus@aliyun.com>
 *    Blog and source code availability: <https://gitee.com/herodotus/herodotus-cloud>
 */

package cn.herodotus.stirrup.rest.servlet.device.controller;

import cn.herodotus.stirrup.logic.iot.device.entity.IotDeviceDetails;
import cn.herodotus.stirrup.logic.iot.device.service.IotDeviceDetailsService;
import cn.herodotus.stirrup.core.definition.domain.Result;
import cn.herodotus.stirrup.data.crud.service.JpaWriteableService;
import cn.herodotus.stirrup.web.api.servlet.AbstractJpaWriteableController;
import cn.herodotus.stirrup.web.core.annotation.AccessLimited;
import io.swagger.v3.oas.annotations.Operation;
import io.swagger.v3.oas.annotations.Parameter;
import io.swagger.v3.oas.annotations.Parameters;
import io.swagger.v3.oas.annotations.enums.ParameterIn;
import io.swagger.v3.oas.annotations.media.Content;
import io.swagger.v3.oas.annotations.media.Schema;
import io.swagger.v3.oas.annotations.responses.ApiResponse;
import io.swagger.v3.oas.annotations.tags.Tag;
import io.swagger.v3.oas.annotations.tags.Tags;
import jakarta.validation.constraints.NotNull;
import org.springframework.data.domain.Page;
import org.springframework.web.bind.annotation.GetMapping;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RequestParam;
import org.springframework.web.bind.annotation.RestController;

import java.util.Map;

/**
 * <p>Description: 物联网设备管理接口 </p>
 *
 * @author : gengwei.zheng
 * @date : 2023/10/6 16:27
 */
@RestController
@RequestMapping("/iot/device/details")
@Tags({
        @Tag(name = "物联网设备管理接口"),
        @Tag(name = "物联网设备详情管理接口"),
})
public class IotDeviceDetailsController extends AbstractJpaWriteableController<IotDeviceDetails, String> {

    private final IotDeviceDetailsService iotDeviceDetailsService;

    public IotDeviceDetailsController(IotDeviceDetailsService iotDeviceDetailsService) {
        this.iotDeviceDetailsService = iotDeviceDetailsService;
    }

    @Override
    public JpaWriteableService<IotDeviceDetails, String> getWriteableService() {
        return this.iotDeviceDetailsService;
    }

    @AccessLimited
    @Operation(summary = "条件分页查询设备详情", description = "根据动态输入的字段查询设备详情分页信息",
            responses = {@ApiResponse(description = "设备详情列表", content = @Content(mediaType = "application/json", schema = @Schema(implementation = Map.class)))})
    @Parameters({
            @Parameter(name = "pageNumber", required = true, in = ParameterIn.QUERY, description = "当前页码"),
            @Parameter(name = "pageSize", required = true, in = ParameterIn.QUERY, description = "每页显示数量"),
            @Parameter(name = "productKey", in = ParameterIn.QUERY, description = "每个产品颁发全局唯一的ProductKey"),
    })
    @GetMapping("/list")
    public Result<Map<String, Object>> findByCondition(@NotNull @RequestParam("pageNumber") Integer pageNumber,
                                                       @NotNull @RequestParam("pageSize") Integer pageSize,
                                                       @RequestParam(value = "productKey", required = false) String productKey) {
        Page<IotDeviceDetails> pages = iotDeviceDetailsService.findByCondition(pageNumber, pageSize, productKey);
        return result(pages);
    }

    @AccessLimited
    @Operation(summary = "条件分页查询设备详情", description = "根据动态输入的字段查询设备详情分页信息",
            responses = {@ApiResponse(description = "设备详情列表", content = @Content(mediaType = "application/json", schema = @Schema(implementation = IotDeviceDetails.class)))})
    @Parameters({
            @Parameter(name = "productKey", required = true, in = ParameterIn.QUERY, description = "每个产品颁发全局唯一的ProductKey"),
            @Parameter(name = "deviceName", required = true, in = ParameterIn.QUERY, description = "产品内唯一的证书 DeviceName"),
    })
    @GetMapping("/one")
    public Result<IotDeviceDetails> findOne(@NotNull @RequestParam("productKey") String productKey,
                                            @NotNull @RequestParam("deviceName") String deviceName) {
        IotDeviceDetails domain = iotDeviceDetailsService.findByProductKeyAndDeviceName(productKey, deviceName);
        return result(domain);
    }
}
