/*
 * DO NOT ALTER OR REMOVE COPYRIGHT NOTICES OR THIS HEADER.
 *
 * Copyright (c) 2020-2030 郑庚伟 ZHENGGENGWEI (码匠君), <herodotus@aliyun.com> Licensed under the AGPL License
 *
 * This file is part of Herodotus Cloud.
 *
 * Herodotus Cloud is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published
 * by the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * Herodotus Cloud is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <https://www.herodotus.vip>.
 */

package cn.herodotus.stirrup.rest.reactive.message.controller;

import cn.herodotus.stirrup.core.definition.domain.Result;
import cn.herodotus.stirrup.data.crud.service.JpaWriteableService;
import cn.herodotus.stirrup.logic.message.entity.Notification;
import cn.herodotus.stirrup.logic.message.enums.NotificationCategory;
import cn.herodotus.stirrup.logic.message.service.NotificationService;
import cn.herodotus.stirrup.web.api.reactive.AbstractJpaWriteableController;
import io.swagger.v3.oas.annotations.Operation;
import io.swagger.v3.oas.annotations.Parameter;
import io.swagger.v3.oas.annotations.Parameters;
import io.swagger.v3.oas.annotations.media.Content;
import io.swagger.v3.oas.annotations.media.Schema;
import io.swagger.v3.oas.annotations.responses.ApiResponse;
import io.swagger.v3.oas.annotations.tags.Tag;
import io.swagger.v3.oas.annotations.tags.Tags;
import jakarta.validation.constraints.NotNull;
import org.apache.commons.lang3.ObjectUtils;
import org.springframework.data.domain.Page;
import org.springframework.web.bind.annotation.*;
import reactor.core.publisher.Mono;

import java.util.Map;

/**
 * <p>Description: 系统通知 </p>
 *
 * @author : gengwei.zheng
 * @date : 2024/3/27 15:38
 */
@RestController
@RequestMapping("/message/notification")
@Tags({
        @Tag(name = "消息管理接口"),
        @Tag(name = "通知管理接口")
})
public class NotificationController extends AbstractJpaWriteableController<Notification, String> {

    private final NotificationService notificationService;

    public NotificationController(NotificationService notificationService) {
        this.notificationService = notificationService;
    }

    @Override
    public JpaWriteableService<Notification, String> getWriteableService() {
        return notificationService;
    }

    @Operation(summary = "条件查询通知信息分页数据", description = "根据输入的字段条件查询通知信息",
            responses = {@ApiResponse(description = "详情列表", content = @Content(mediaType = "application/json", schema = @Schema(implementation = Map.class)))})
    @Parameters({
            @Parameter(name = "pageNumber", required = true, description = "当前页码", schema = @Schema(type = "integer")),
            @Parameter(name = "pageSize", required = true, description = "每页显示数量", schema = @Schema(type = "integer")),
            @Parameter(name = "userId", required = true, description = "用户ID"),
            @Parameter(name = "category", description = "通知类别", schema = @Schema(type = "integer")),
            @Parameter(name = "read", description = "是否已读", schema = @Schema(type = "boolean")),
    })
    @GetMapping("/condition")
    public Mono<Result<Map<String, Object>>> findByCondition(@NotNull @RequestParam(value = "pageNumber") Integer pageNumber,
                                                             @NotNull @RequestParam(value = "pageSize") Integer pageSize,
                                                             @NotNull @RequestParam(value = "userId") String userId,
                                                             @RequestParam(value = "category", required = false) Integer category,
                                                             @RequestParam(value = "read", required = false) Boolean read) {
        NotificationCategory notificationCategory = NotificationCategory.get(category);
        if (ObjectUtils.isNotEmpty(notificationCategory)) {
            if (notificationCategory == NotificationCategory.ANNOUNCEMENT) {
                notificationService.pullAnnouncements(userId);
            }
        }

        Page<Notification> pages = notificationService.findByCondition(pageNumber, pageSize, userId, notificationCategory, read);
        return result(pages);
    }

    @Operation(summary = "全部通知已读", description = "根据用户ID设置该用户的全部通知为已读",
            responses = {@ApiResponse(description = "影响数据条目数", content = @Content(mediaType = "application/json"))})
    @Parameters({
            @Parameter(name = "userId", description = "用户ID")
    })
    @PutMapping("/all-read")
    public Mono<Result<Integer>> setAllRead(@RequestParam("userId") String userId) {
        Integer result = notificationService.setAllRead(userId);
        return Mono.just(Result.success("操作成功", result));
    }
}
