/*
 * DO NOT ALTER OR REMOVE COPYRIGHT NOTICES OR THIS HEADER.
 *
 * Copyright (c) 2020-2030 郑庚伟 ZHENGGENGWEI (码匠君), <herodotus@aliyun.com> Licensed under the AGPL License
 *
 * This file is part of Herodotus Cloud.
 *
 * Herodotus Cloud is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published
 * by the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * Herodotus Cloud is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <https://www.herodotus.vip>.
 */

package cn.herodotus.stirrup.openapi.emqx.core;

import cn.herodotus.stirrup.core.definition.constants.SymbolConstants;
import cn.herodotus.stirrup.openapi.core.definition.AbstractWebClientTemplate;
import cn.herodotus.stirrup.openapi.emqx.domain.response.EmqxResponse;
import cn.herodotus.stirrup.openapi.emqx.properties.EmqxProperties;
import okhttp3.Credentials;
import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.http.HttpHeaders;
import org.springframework.http.HttpMethod;
import org.springframework.http.HttpRequest;
import org.springframework.http.MediaType;
import org.springframework.web.reactive.function.client.ClientRequest;
import org.springframework.web.reactive.function.client.WebClient;
import org.springframework.web.reactive.function.client.WebClientResponseException;
import reactor.core.publisher.Mono;

import java.util.function.Function;

/**
 * <p>Description: Emqx Rest API 封装基础服务 </p>
 *
 * @author : gengwei.zheng
 * @date : 2023/9/20 17:36
 */
public class EmqxTemplate extends AbstractWebClientTemplate {

    private static final Logger log = LoggerFactory.getLogger(EmqxTemplate.class);

    private final EmqxProperties emqxProperties;

    public EmqxTemplate(WebClient webClient, EmqxProperties emqxProperties) {
        super(webClient);
        this.emqxProperties = emqxProperties;
    }

    @Override
    protected Function<ClientRequest, Mono<ClientRequest>> authentication() {
        return clientRequest -> {
            ClientRequest request = ClientRequest.from(clientRequest)
                    .header(HttpHeaders.CONTENT_TYPE, MediaType.APPLICATION_JSON_VALUE)
                    .header(HttpHeaders.AUTHORIZATION, Credentials.basic(emqxProperties.getApiKey(), emqxProperties.getSecretKey()))
                    .build();
            return Mono.just(request);
        };
    }

    @Override
    protected String getBaseUrl() {
        String server = emqxProperties.getUrl();
        if (!StringUtils.endsWith(server, SymbolConstants.FORWARD_SLASH)) {
            server = server + SymbolConstants.FORWARD_SLASH;
        }

        String url = server + "api/" + emqxProperties.getVersion() + "/nodes";
        log.debug("[Herodotus] |- Emqx open api address is [{}]!", url);
        return url;
    }

    @Override
    public <O> Function<WebClientResponseException, Mono<O>> onErrorResume() {
        return exception -> {
            HttpRequest request = exception.getRequest();
            String path = request.getURI().getPath();
            HttpMethod method = request.getMethod();

            EmqxResponse<?> response = exception.getResponseBodyAs(EmqxResponse.class);

            log.error("[Herodotus] |- Invoke nacos open api [{} - {}] failure. Reason is [{}]", method.name(), path, response.getMessage());
            O result = (O) response;
            return Mono.just(result);
        };
    }
}
