/*
 * DO NOT ALTER OR REMOVE COPYRIGHT NOTICES OR THIS HEADER.
 *
 * Copyright (c) 2020-2030 郑庚伟 ZHENGGENGWEI (码匠君), <herodotus@aliyun.com> Licensed under the AGPL License
 *
 * This file is part of Herodotus Stirrup.
 *
 * Herodotus Stirrup is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published
 * by the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * Herodotus Stirrup is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <https://www.herodotus.vip>.
 */

package cn.herodotus.stirrup.openapi.aliyun.audit.service;

import cn.herodotus.stirrup.openapi.aliyun.audit.constants.AliyunAuditConstants;
import cn.herodotus.stirrup.openapi.aliyun.audit.definition.domain.Response;
import cn.herodotus.stirrup.openapi.aliyun.audit.definition.service.AbstractScanService;
import cn.herodotus.stirrup.openapi.aliyun.audit.domain.text.TextResult;
import cn.herodotus.stirrup.openapi.aliyun.audit.domain.text.TextSyncRequest;
import cn.herodotus.stirrup.openapi.aliyun.audit.domain.text.TextSyncResponse;
import cn.herodotus.stirrup.openapi.aliyun.audit.domain.text.TextTask;
import cn.herodotus.stirrup.core.foundation.json.jackson2.utils.Jackson2Utils;
import com.aliyuncs.IAcsClient;
import com.aliyuncs.green.model.v20180509.TextFeedbackRequest;
import com.aliyuncs.green.model.v20180509.TextScanRequest;
import com.google.common.collect.ImmutableList;
import org.apache.commons.lang3.ObjectUtils;
import org.apache.http.HttpStatus;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.stereotype.Service;

import java.util.List;
import java.util.stream.Collectors;

/**
 * <p>Description: 阿里文本审核服务类 </p>
 *
 * @author : gengwei.zheng
 * @date : 2021/4/13 15:38
 */
@Service
public class TextScanService extends AbstractScanService {

    private static final Logger log = LoggerFactory.getLogger(TextScanService.class);

    private final TextScanRequest textScanRequest;
    private final TextFeedbackRequest textFeedbackRequest;

    public TextScanService(IAcsClient iAcsClient, TextScanRequest textScanRequest, TextFeedbackRequest textFeedbackRequest) {
        super(iAcsClient);
        this.textScanRequest = textScanRequest;
        this.textFeedbackRequest = textFeedbackRequest;
    }

    public Response<List<TextSyncResponse>> scan(TextSyncRequest textSyncRequest) {
        String jsonString = this.scan(textSyncRequest, textScanRequest);
        Response<List<TextSyncResponse>> entity = this.parseListResult(jsonString, TextSyncResponse.class);
        log.debug("[Herodotus] |- Aliyun Text Scan result is: {}", entity.toString());
        return entity;
    }

    public Response<String> feedback(List<TextFeedbackRequest> textFeedbackRequests) {
        String data = Jackson2Utils.toJson(textFeedbackRequests);
        String response = this.scan(data, this.textFeedbackRequest);
        Response<String> feedback = this.parseResult(response, String.class);
        log.debug("[Herodotus] |- Aliyun Text Feedback result is: {}", feedback.toString());
        return feedback;
    }

    public TextSyncRequest buildSyncRequest(List<TextTask> tasks, List<String> scenes, String bizType) {
        TextSyncRequest textSyncRequest = new TextSyncRequest();
        textSyncRequest.setBizType(bizType);
        textSyncRequest.setScenes(scenes);
        textSyncRequest.setTasks(tasks);
        return textSyncRequest;
    }

    public TextSyncRequest buildDefaultAsyncRequest(List<String> contents) {
        List<TextTask> tasks = contents.stream().map(content -> {
            TextTask textTask = new TextTask();
            textTask.setContent(content);
            return textTask;
        }).collect(Collectors.toList());

        List<String> scenes = ImmutableList.of(AliyunAuditConstants.SCENE_ANTISPAM);

        return buildSyncRequest(tasks, scenes, AliyunAuditConstants.BIZ_TYPE);
    }

    public boolean asyncAnalyse(List<TextSyncResponse> responses) {
        for (TextSyncResponse textSyncResponse : responses) {
            List<TextResult> textResults = textSyncResponse.getResults();
            for (TextResult textResult : textResults) {
                if (textResult.getSuggestion().equals("block")) {
                    log.warn("[Herodotus] |- Catch the block content: {}", textResult);
                    return false;
                }
            }
        }

        log.debug("[Herodotus] |- Content check result is ok!");
        return true;
    }

    public boolean executeScan(List<String> contents) {
        TextSyncRequest textSyncRequest = this.buildDefaultAsyncRequest(contents);
        Response<List<TextSyncResponse>> response = this.scan(textSyncRequest);
        if (ObjectUtils.isNotEmpty(response) && response.getCode().equals(HttpStatus.SC_OK)) {
            return this.asyncAnalyse(response.getData());
        } else {
            log.error("[Herodotus] |- Aliyun Text Scan catch error! result: {}", response);
            return false;
        }
    }
}
