/*
 * DO NOT ALTER OR REMOVE COPYRIGHT NOTICES OR THIS HEADER.
 *
 * Copyright (c) 2020-2030 郑庚伟 ZHENGGENGWEI (码匠君), <herodotus@aliyun.com> Licensed under the AGPL License
 *
 * This file is part of Herodotus Stirrup.
 *
 * Herodotus Stirrup is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published
 * by the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * Herodotus Stirrup is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <https://www.herodotus.vip>.
 */

package cn.herodotus.stirrup.openapi.aliyun.audit.definition.service;

import cn.herodotus.stirrup.openapi.aliyun.audit.definition.domain.CoreRequest;
import cn.herodotus.stirrup.openapi.aliyun.audit.definition.domain.Response;
import cn.herodotus.stirrup.core.foundation.json.jackson2.utils.Jackson2Utils;
import com.aliyuncs.AcsResponse;
import com.aliyuncs.IAcsClient;
import com.aliyuncs.RoaAcsRequest;
import com.aliyuncs.exceptions.ClientException;
import com.aliyuncs.http.FormatType;
import com.aliyuncs.http.HttpResponse;
import com.fasterxml.jackson.databind.JavaType;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.fasterxml.jackson.databind.type.TypeFactory;
import org.dromara.hutool.core.text.StrUtil;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.nio.charset.StandardCharsets;
import java.util.ArrayList;
import java.util.List;

/**
 * <p>Description: 阿里内容审核基础类 </p>
 *
 * @author : gengwei.zheng
 * @date : 2021/4/13 15:16
 */
public abstract class AbstractScanService {

    private static final Logger log = LoggerFactory.getLogger(AbstractScanService.class);

    private final IAcsClient iAcsClient;

    protected AbstractScanService(IAcsClient iAcsClient) {
        this.iAcsClient = iAcsClient;
    }

    protected <R extends AcsResponse> String scan(CoreRequest coreRequest, RoaAcsRequest<R> roaAcsRequest) {
        String data = Jackson2Utils.toJson(coreRequest);
        return this.scan(data, roaAcsRequest);
    }

    protected <R extends AcsResponse> String query(List<String> taskIds, RoaAcsRequest<R> roaAcsRequest) {
        String data = Jackson2Utils.toJson(taskIds);
        return this.scan(data, roaAcsRequest);
    }

    protected <R extends AcsResponse> String scan(String data, RoaAcsRequest<R> roaAcsRequest) {
        try {
            roaAcsRequest.setHttpContent(data.getBytes(StandardCharsets.UTF_8), StandardCharsets.UTF_8.name(), FormatType.JSON);
            HttpResponse httpResponse = iAcsClient.doAction(roaAcsRequest);
            if (httpResponse.isSuccess()) {
                String response = StrUtil.str(httpResponse.getHttpContent(), StandardCharsets.UTF_8);
                log.debug("[Herodotus] |- Aliyun scan response is: {}", response);
                return response;
            } else {
                log.error("[Herodotus] |- Aliyun scan text cannot get response");
                return null;
            }
        } catch (ClientException e) {
            log.error("[Herodotus] |- Aliyun scan create client got an error!");
            return null;
        }
    }

    protected <P> Response<P> parseResult(String json, Class<P> clazz) {
        TypeFactory typeFactory = Jackson2Utils.getTypeFactory();
        JavaType javaType = typeFactory.constructParametricType(Response.class, clazz);
        return Jackson2Utils.toObject(json, javaType);
    }

    protected <P> Response<List<P>> parseListResult(String json, Class<P> clazz) {
        TypeFactory typeFactory = Jackson2Utils.getTypeFactory();
        JavaType list = typeFactory.constructParametricType(ArrayList.class, clazz);
        JavaType object = typeFactory.constructParametricType(Response.class, list);
        return Jackson2Utils.toObject(json, object);
    }

    protected <P> List<P> parseList(String json, Class<P> clazz) {
        ObjectMapper objectMapper = Jackson2Utils.getObjectMapper();
        JavaType javaType = objectMapper.getTypeFactory().constructParametricType(ArrayList.class, clazz);
        return Jackson2Utils.toObject(json, javaType);
    }
}
