/*
 * DO NOT ALTER OR REMOVE COPYRIGHT NOTICES OR THIS HEADER.
 *
 * Copyright (c) 2020-2030 郑庚伟 ZHENGGENGWEI (码匠君), <herodotus@aliyun.com> Licensed under the AGPL License
 *
 * This file is part of Herodotus Cloud.
 *
 * Herodotus Cloud is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published
 * by the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * Herodotus Cloud is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <https://www.herodotus.vip>.
 */

package cn.herodotus.stirrup.openapi.aliyun.audit.config;

import cn.herodotus.stirrup.openapi.aliyun.audit.annotation.ConditionalOnAliyunAuditConfigured;
import cn.herodotus.stirrup.openapi.aliyun.audit.properties.AliyunProperties;
import com.aliyuncs.DefaultAcsClient;
import com.aliyuncs.IAcsClient;
import com.aliyuncs.green.model.v20180509.*;
import com.aliyuncs.http.FormatType;
import com.aliyuncs.http.MethodType;
import com.aliyuncs.http.ProtocolType;
import com.aliyuncs.profile.DefaultProfile;
import com.aliyuncs.profile.IClientProfile;
import jakarta.annotation.PostConstruct;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.boot.autoconfigure.condition.ConditionalOnMissingBean;
import org.springframework.boot.context.properties.EnableConfigurationProperties;
import org.springframework.context.annotation.Bean;
import org.springframework.context.annotation.ComponentScan;
import org.springframework.context.annotation.Configuration;

import java.nio.charset.StandardCharsets;

/**
 * <p>Description: 环信启动配置类 </p>
 *
 * @author : gengwei.zheng
 * @date : 2021/4/6 13:13
 */
@Configuration(proxyBeanMethods = false)
@ConditionalOnAliyunAuditConfigured
@EnableConfigurationProperties(AliyunProperties.class)
@ComponentScan(basePackages = {
        "cn.herodotus.stirrup.openapi.aliyun.audit.service",
        "cn.herodotus.stirrup.openapi.aliyun.audit.processor"
})
public class AliyunAuditConfiguration {

    private static final Logger log = LoggerFactory.getLogger(AliyunAuditConfiguration.class);

    @PostConstruct
    public void postConstruct() {
        log.debug("[Herodotus] |- Module [Aliyun Audit REST API] Auto Configure.");
    }

    @Bean
    @ConditionalOnMissingBean(IAcsClient.class)
    public IAcsClient iAcsClient(AliyunProperties aliyunProperties) {
        IClientProfile iClientProfile = DefaultProfile.getProfile(aliyunProperties.getRegionId(), aliyunProperties.getAccessKeyId(), aliyunProperties.getAccessKeySecret());
        IAcsClient iAcsClient = new DefaultAcsClient(iClientProfile);
        log.trace("[Herodotus] |- Bean [IAcs Client] in AliyunScanConfiguration Auto Configure.");
        return iAcsClient;
    }

    /**
     * 文本识别请求
     *
     * @return {@link TextScanRequest}
     */
    @Bean
    public TextScanRequest textScanRequest(AliyunProperties aliyunProperties) {
        TextScanRequest textScanRequest = new TextScanRequest();
        //指定api返回格式
        textScanRequest.setSysAcceptFormat(FormatType.JSON);
        //设定请求内容格式
        textScanRequest.setHttpContentType(FormatType.JSON);
        //指定请求方法
        textScanRequest.setSysMethod(MethodType.POST);
        textScanRequest.setSysEncoding(StandardCharsets.UTF_8.name());
        textScanRequest.setSysRegionId(aliyunProperties.getRegionId());
        // 请务必设置超时时间
        textScanRequest.setSysConnectTimeout(aliyunProperties.getAudit().getText().getConnectTimeout());
        textScanRequest.setSysReadTimeout(aliyunProperties.getAudit().getText().getReadTimeout());

        log.trace("[Herodotus] |- Bean [Text Scan Request] Auto Configure.");
        return textScanRequest;
    }

    /**
     * 文本反垃圾结果反馈
     * <p>
     * 如果您认为我们的文本检测结果与您的期望不符，您可以通过文本垃圾结果反馈接口纠正算法检测结果。
     *
     * @return {@link TextScanRequest}
     */
    @Bean
    public TextFeedbackRequest textFeedbackRequest(AliyunProperties aliyunProperties) {
        TextFeedbackRequest textFeedbackRequest = new TextFeedbackRequest();
        //指定api返回格式
        textFeedbackRequest.setSysAcceptFormat(FormatType.JSON);
        //设定请求内容格式
        textFeedbackRequest.setHttpContentType(FormatType.JSON);
        //指定请求方法
        textFeedbackRequest.setSysMethod(MethodType.POST);
        textFeedbackRequest.setSysEncoding(StandardCharsets.UTF_8.name());
        textFeedbackRequest.setSysRegionId(aliyunProperties.getRegionId());
        // 请务必设置超时时间
        textFeedbackRequest.setSysConnectTimeout(aliyunProperties.getAudit().getText().getConnectTimeout());
        textFeedbackRequest.setSysReadTimeout(aliyunProperties.getAudit().getText().getReadTimeout());

        log.trace("[Herodotus] |- Bean [Text Feedback Request] Auto Configure.");
        return textFeedbackRequest;
    }

    /**
     * 图片同步请求（推荐使用同步）
     *
     * @return
     */
    @Bean
    public ImageSyncScanRequest imageSyncScanRequest(AliyunProperties aliyunProperties) {
        ImageSyncScanRequest imageSyncScanRequest = new ImageSyncScanRequest();
        // 指定api返回格式
        imageSyncScanRequest.setSysAcceptFormat(FormatType.JSON);
        // 指定请求方法
        imageSyncScanRequest.setSysMethod(MethodType.POST);
        imageSyncScanRequest.setSysEncoding(StandardCharsets.UTF_8.name());
        //支持http和https
        imageSyncScanRequest.setSysProtocol(ProtocolType.HTTP);

        /**
         * 请设置超时时间, 服务端全链路处理超时时间为10秒，请做相应设置
         * 如果您设置的ReadTimeout小于服务端处理的时间，程序中会获得一个read timeout异常
         */
        imageSyncScanRequest.setSysConnectTimeout(aliyunProperties.getAudit().getImage().getConnectTimeout());
        imageSyncScanRequest.setSysReadTimeout(aliyunProperties.getAudit().getImage().getReadTimeout());

        log.trace("[Herodotus] |- Bean [Image Sync Scan Request] Auto Configure.");
        return imageSyncScanRequest;
    }


    /**
     * 图片异步检测请求
     *
     * @return
     */
    @Bean
    public ImageAsyncScanRequest imageAsyncScanRequest(AliyunProperties aliyunProperties) {
        ImageAsyncScanRequest imageAsyncScanRequest = new ImageAsyncScanRequest();
        // 指定api返回格式
        imageAsyncScanRequest.setSysAcceptFormat(FormatType.JSON);
        // 指定请求方法
        imageAsyncScanRequest.setSysMethod(MethodType.POST);
        imageAsyncScanRequest.setSysEncoding(StandardCharsets.UTF_8.name());
        //支持http和https
        imageAsyncScanRequest.setSysProtocol(ProtocolType.HTTP);
        /**
         * 请设置超时时间, 服务端全链路处理超时时间为10秒，请做相应设置
         * 如果您设置的ReadTimeout小于服务端处理的时间，程序中会获得一个read timeout异常
         */
        imageAsyncScanRequest.setSysConnectTimeout(aliyunProperties.getAudit().getImage().getConnectTimeout());
        imageAsyncScanRequest.setSysReadTimeout(aliyunProperties.getAudit().getImage().getReadTimeout());

        log.trace("[Herodotus] |- Bean [Image Async Scan Request] Auto Configure.");
        return imageAsyncScanRequest;
    }


    /**
     * 图片异步请求结果查询
     *
     * @return
     */
    @Bean
    public ImageAsyncScanResultsRequest imageAsyncSacnResultsRequest(AliyunProperties aliyunProperties) {
        ImageAsyncScanResultsRequest imageAsyncScanResultsRequest = new ImageAsyncScanResultsRequest();
        // 指定api返回格式
        imageAsyncScanResultsRequest.setSysAcceptFormat(FormatType.JSON);
        // 指定请求方法
        imageAsyncScanResultsRequest.setSysMethod(MethodType.POST);
        imageAsyncScanResultsRequest.setSysEncoding(StandardCharsets.UTF_8.name());
        //支持http和https
        imageAsyncScanResultsRequest.setSysProtocol(ProtocolType.HTTP);

        /**
         * 请设置超时时间, 服务端全链路处理超时时间为10秒，请做相应设置
         * 如果您设置的ReadTimeout小于服务端处理的时间，程序中会获得一个read timeout异常
         */
        imageAsyncScanResultsRequest.setSysConnectTimeout(aliyunProperties.getAudit().getImage().getConnectTimeout());
        imageAsyncScanResultsRequest.setSysReadTimeout(aliyunProperties.getAudit().getImage().getReadTimeout());

        log.trace("[Herodotus] |- Bean [Image Async Scan Results Request] Auto Configure.");
        return imageAsyncScanResultsRequest;
    }


    /**
     * 视频同步请求
     *
     * @return
     */
    @Bean
    public VideoSyncScanRequest videoSyncScanRequest(AliyunProperties aliyunProperties) {
        VideoSyncScanRequest videoSyncScanRequest = new VideoSyncScanRequest();
        //指定api返回格式
        videoSyncScanRequest.setSysAcceptFormat(FormatType.JSON);
        //指定请求方法
        videoSyncScanRequest.setSysMethod(MethodType.POST);
        /**
         * 请务必设置超时时间
         */
        videoSyncScanRequest.setSysConnectTimeout(aliyunProperties.getAudit().getVideo().getConnectTimeout());
        videoSyncScanRequest.setSysReadTimeout(aliyunProperties.getAudit().getVideo().getReadTimeout());

        log.trace("[Herodotus] |- Bean [Video Sync Scan Request] Auto Configure.");
        return videoSyncScanRequest;
    }


    /**
     * 视频异步请求（推荐使用异步）
     *
     * @return
     */
    @Bean
    public VideoAsyncScanRequest videoAsyncScanRequest(AliyunProperties aliyunProperties) {
        VideoAsyncScanRequest videoAsyncScanRequest = new VideoAsyncScanRequest();
        //指定api返回格式
        videoAsyncScanRequest.setSysAcceptFormat(FormatType.JSON);
        //指定请求方法
        videoAsyncScanRequest.setSysMethod(MethodType.POST);
        /**
         * 请务必设置超时时间
         */
        videoAsyncScanRequest.setSysConnectTimeout(aliyunProperties.getAudit().getVideo().getConnectTimeout());
        videoAsyncScanRequest.setSysReadTimeout(aliyunProperties.getAudit().getVideo().getReadTimeout());

        log.trace("[Herodotus] |- Bean [Video Async Scan Request] Auto Configure.");
        return videoAsyncScanRequest;
    }

    /**
     * 视频异步请求结果查询请求
     *
     * @return
     */
    @Bean
    public VideoAsyncScanResultsRequest videoAsyncResultRequest(AliyunProperties aliyunProperties) {
        VideoAsyncScanResultsRequest videoAsyncResultRequest = new VideoAsyncScanResultsRequest();
        // 指定api返回格式
        videoAsyncResultRequest.setSysAcceptFormat(FormatType.JSON);
        /**
         * 请务必设置超时时间
         */
        videoAsyncResultRequest.setSysConnectTimeout(aliyunProperties.getAudit().getVideo().getConnectTimeout());
        videoAsyncResultRequest.setSysReadTimeout(aliyunProperties.getAudit().getVideo().getReadTimeout());

        log.trace("[Herodotus] |- Bean [Video Async Scan Results Request] Auto Configure.");
        return videoAsyncResultRequest;
    }

    /**
     * 语音检测请求(语音识别仅支持异步)
     *
     * @return
     */
    @Bean
    public VoiceSyncScanRequest voiceSyncScanRequest(AliyunProperties aliyunProperties) {
        VoiceSyncScanRequest voiceSyncScanRequest = new VoiceSyncScanRequest();
        //指定api返回格式
        voiceSyncScanRequest.setSysAcceptFormat(FormatType.JSON);
        //指定请求方法
        voiceSyncScanRequest.setSysMethod(MethodType.POST);
        voiceSyncScanRequest.setSysRegionId(aliyunProperties.getRegionId());
        voiceSyncScanRequest.setSysConnectTimeout(aliyunProperties.getAudit().getVoice().getConnectTimeout());
        voiceSyncScanRequest.setSysReadTimeout(aliyunProperties.getAudit().getVoice().getReadTimeout());

        log.trace("[Herodotus] |- Bean [Voice Sync Scan Request] Auto Configure.");
        return voiceSyncScanRequest;
    }

    /**
     * 语音检测请求(语音识别仅支持异步)
     *
     * @return
     */
    @Bean
    public VoiceAsyncScanRequest voiceAsyncScanRequest(AliyunProperties aliyunProperties) {
        VoiceAsyncScanRequest voiceAsyncScanRequest = new VoiceAsyncScanRequest();
        //指定api返回格式
        voiceAsyncScanRequest.setSysAcceptFormat(FormatType.JSON);
        //指定请求方法
        voiceAsyncScanRequest.setSysMethod(MethodType.POST);
        voiceAsyncScanRequest.setSysRegionId(aliyunProperties.getRegionId());
        voiceAsyncScanRequest.setSysConnectTimeout(aliyunProperties.getAudit().getVoice().getConnectTimeout());
        voiceAsyncScanRequest.setSysReadTimeout(aliyunProperties.getAudit().getVoice().getReadTimeout());

        log.trace("[Herodotus] |- Bean [Voice Async Scan Request] Auto Configure.");
        return voiceAsyncScanRequest;
    }


    /**
     * 语音异步检测结果查询
     *
     * @return
     */
    @Bean
    public VoiceAsyncScanResultsRequest voiceAsyncResultsRequest(AliyunProperties aliyunProperties) {
        VoiceAsyncScanResultsRequest voiceAsyncScanResultsRequest = new VoiceAsyncScanResultsRequest();
        // 指定api返回格式
        voiceAsyncScanResultsRequest.setSysAcceptFormat(FormatType.JSON);
        // 指定请求方法
        voiceAsyncScanResultsRequest.setSysMethod(MethodType.POST);
        voiceAsyncScanResultsRequest.setSysEncoding(StandardCharsets.UTF_8.name());
        voiceAsyncScanResultsRequest.setSysRegionId(aliyunProperties.getRegionId());
        /**
         * 请务必设置超时时间
         */
        voiceAsyncScanResultsRequest.setSysConnectTimeout(aliyunProperties.getAudit().getVoice().getConnectTimeout());
        voiceAsyncScanResultsRequest.setSysReadTimeout(aliyunProperties.getAudit().getVoice().getReadTimeout());

        log.trace("[Herodotus] |- Bean [Voice Async Scan Results Request] Auto Configure.");
        return voiceAsyncScanResultsRequest;
    }
}
