/*
 * DO NOT ALTER OR REMOVE COPYRIGHT NOTICES OR THIS HEADER.
 *
 * Copyright (c) 2020-2030 郑庚伟 ZHENGGENGWEI (码匠君), <herodotus@aliyun.com> Licensed under the AGPL License
 *
 * This file is part of Herodotus Stirrup.
 *
 * Herodotus Stirrup is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published
 * by the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * Herodotus Stirrup is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <https://www.herodotus.vip>.
 */

package cn.herodotus.stirrup.openapi.core.definition;

import cn.herodotus.stirrup.core.definition.domain.base.Response;
import cn.herodotus.stirrup.core.foundation.exception.openapi.OpenApiInvokingFailedException;
import org.apache.commons.collections4.MapUtils;
import org.apache.commons.lang3.ObjectUtils;
import org.springframework.core.ParameterizedTypeReference;
import org.springframework.core.convert.converter.Converter;
import org.springframework.http.HttpMethod;
import org.springframework.http.MediaType;
import org.springframework.util.MultiValueMap;
import org.springframework.web.reactive.function.BodyInserters;
import org.springframework.web.reactive.function.client.WebClient;
import org.springframework.web.reactive.function.client.WebClientResponseException;
import reactor.core.publisher.Mono;

import java.util.Map;
import java.util.Objects;
import java.util.function.Function;

/**
 * <p>Description: WebClient 请求发送通用方法 </p>
 *
 * @author : gengwei.zheng
 * @date : 2024/5/4 21:50
 */
public interface WebClientTemplate {

    /**
     * WebClient Map 参数转换器
     *
     * @return 转换器 {@link Converter}
     */
    Converter<Map<String, Object>, MultiValueMap<String, String>> getMultiValueMapConverter();

    /**
     * 获取到 WebClient
     *
     * @return {@link WebClient}
     */
    WebClient getWebClient();

    /**
     * WebClient 错误处理。主要解决 {@link WebClientResponseException} 转换成统一的响应实体
     *
     * @param <O> 响应数据类型
     * @return {@link Function}
     */
    <O> Function<WebClientResponseException, Mono<O>> onErrorResume();

    /**
     * WebClient 请求通用操作
     *
     * @param httpMethod    请求类型{@link HttpMethod}
     * @param endpoint      请求 API
     * @param body          请求体数据对象，任意类型对象，用于 ContentType 为 "application/json" 类型请求
     * @param formVariables 请求体数据 {@link Map}，用于 ContentType 为 "application/x-www-form-urlencoded" 请求
     * @param pathVariables pathVariables 类型请求路径参数 例如：/xxx/{id};
     * @param queryParams   queryParams 类型请求路径上的参数 例如：/xxx?id=xxx
     * @param onErrorResume WebClient 调用 API 出错错误处理
     * @param <I>           请求数据类型，用于 ContentType 为 "application/json" 类型请求
     * @param <O>           响应数据类型
     * @return 自定义响应结果。如果出错，则结果内容为错误信息
     */
    default <I, O extends Response> Mono<O> request(HttpMethod httpMethod, String endpoint, I body, Map<String, Object> formVariables, Map<String, Object> pathVariables, Map<String, Object> queryParams, Function<WebClientResponseException, Mono<O>> onErrorResume) {
        WebClient.RequestBodySpec requestBodySpec = getWebClient().method(httpMethod)
                .uri(uriBuilder -> {
                    uriBuilder.path(endpoint);

                    if (MapUtils.isNotEmpty(queryParams)) {
                        uriBuilder.queryParams(Objects.requireNonNull(getMultiValueMapConverter().convert(queryParams)));
                    }

                    if (MapUtils.isNotEmpty(pathVariables)) {
                        return uriBuilder.build(pathVariables);
                    }

                    return uriBuilder.build();
                })
                .contentType(MapUtils.isNotEmpty(formVariables) ? MediaType.APPLICATION_FORM_URLENCODED : MediaType.APPLICATION_JSON);

        if (MapUtils.isNotEmpty(formVariables)) {
            requestBodySpec.body(BodyInserters.fromFormData(Objects.requireNonNull(getMultiValueMapConverter().convert(formVariables))));
        }

        if (ObjectUtils.isNotEmpty(body)) {
            requestBodySpec.bodyValue(body);
        }

        return requestBodySpec
                .retrieve()
                .bodyToMono(new ParameterizedTypeReference<O>() {
                })
                .onErrorResume(throwable -> {
                    if (throwable instanceof WebClientResponseException exception) {
                        return onErrorResume.apply(exception);
                    }
                    return Mono.error(new OpenApiInvokingFailedException(throwable));
                });
    }

    /**
     * GET 和 DELETE 请求，支持同时使用 pathVariables 和 queryParams 两种参数
     *
     * @param httpMethod    请求类型{@link HttpMethod}
     * @param endpoint      请求 API
     * @param pathVariables pathVariables 类型请求路径参数 例如：/xxx/{id};
     * @param queryParams   queryParams 类型请求路径上的参数 例如：/xxx?id=xxx
     * @param <O>           响应数据类型
     * @return 自定义响应结果。如果出错，则结果内容为错误信息
     */
    default <O extends Response> Mono<O> base(HttpMethod httpMethod, String endpoint, Map<String, Object> pathVariables, Map<String, Object> queryParams) {
        return request(httpMethod, endpoint, null, null, pathVariables, queryParams, onErrorResume());
    }

    /**
     * ContentType 为 "application/json" 类型的 POST 和 PUT 请求封装，
     *
     * @param httpMethod    请求类型{@link HttpMethod}
     * @param endpoint      请求 API
     * @param body          请求体数据对象
     * @param pathVariables pathVariables 类型请求路径参数
     * @param <I>           请求数据类型，用于 ContentType 为 "application/json" 类型请求
     * @param <O>           响应数据类型
     * @return 自定义响应结果。如果出错，则结果内容为错误信息
     */
    default <I, O extends Response> Mono<O> json(HttpMethod httpMethod, String endpoint, I body, Map<String, Object> pathVariables) {
        return request(httpMethod, endpoint, body, null, pathVariables, null, onErrorResume());
    }

    /**
     * ContentType 为 "application/x-www-form-urlencoded" 类型的 POST 和 PUT 请求封装，
     *
     * @param httpMethod    请求类型{@link HttpMethod}
     * @param endpoint      请求 API
     * @param formVariables 请求体数据 {@link Map}
     * @param pathVariables pathVariables 类型请求路径参数
     * @param <O>           响应数据类型
     * @return 自定义响应结果。如果出错，则结果内容为错误信息
     */
    default <O extends Response> Mono<O> formUrlEncoded(HttpMethod httpMethod, String endpoint, Map<String, Object> formVariables, Map<String, Object> pathVariables) {
        return request(httpMethod, endpoint, null, formVariables, pathVariables, null, onErrorResume());
    }

    /**
     * GET 请求，支持同时使用 pathVariables 和 pathVariables 两种参数
     *
     * @param endpoint      请求 API
     * @param pathVariables pathVariables 类型请求路径参数 例如：/xxx/{id};
     * @param queryParams   queryParams 类型请求路径上的参数 例如：/xxx?id=xxx
     * @param <O>           响应数据类型
     * @return 自定义响应结果。如果出错，则结果内容为错误信息
     */
    default <O extends Response> Mono<O> get(String endpoint, Map<String, Object> pathVariables, Map<String, Object> queryParams) {
        return base(HttpMethod.GET, endpoint, pathVariables, queryParams);
    }

    /**
     * QueryParams 类型参数的 GET 请求
     *
     * @param endpoint    请求 API
     * @param queryParams queryParams 类型请求路径上的参数 例如：/xxx?id=xxx
     * @param <O>         响应数据类型
     * @return 自定义响应结果。如果出错，则结果内容为错误信息
     */
    default <O extends Response> Mono<O> get(String endpoint, Map<String, Object> queryParams) {
        return get(endpoint, null, queryParams);
    }

    /**
     * QueryParams 类型参数的 GET 请求
     *
     * @param endpoint 请求 API
     * @param key      请求参数名
     * @param value    请求参数值
     * @param <O>      响应数据类型
     * @return 自定义响应结果。如果出错，则结果内容为错误信息
     */
    default <O extends Response> Mono<O> get(String endpoint, String key, String value) {
        return get(endpoint, Map.of(key, value));
    }

    /**
     * 无参数的 GET 请求
     *
     * @param endpoint 请求 API
     * @param <O>      响应数据类型
     * @return 自定义响应结果。如果出错，则结果内容为错误信息
     */
    default <O extends Response> Mono<O> get(String endpoint) {
        return get(endpoint, null);
    }

    /**
     * PathVariables 类型参数的 GET 请求
     *
     * @param endpoint      请求 API
     * @param pathVariables pathVariables 类型请求路径参数 例如：/xxx/{id};
     * @param <O>           响应数据类型
     * @return 自定义响应结果。如果出错，则结果内容为错误信息
     */
    default <O extends Response> Mono<O> getWithPathVariables(String endpoint, Map<String, Object> pathVariables) {
        return get(endpoint, pathVariables, null);
    }

    /**
     * DELETE 请求，支持同时使用 pathVariables 和 pathVariables 两种参数
     *
     * @param endpoint      请求 API
     * @param pathVariables pathVariables 类型请求路径参数 例如：/xxx/{id};
     * @param queryParams   queryParams 类型请求路径上的参数 例如：/xxx?id=xxx
     * @param <O>           响应数据类型
     * @return 自定义响应结果。如果出错，则结果内容为错误信息
     */
    default <O extends Response> Mono<O> delete(String endpoint, Map<String, Object> pathVariables, Map<String, Object> queryParams) {
        return base(HttpMethod.DELETE, endpoint, pathVariables, queryParams);
    }

    /**
     * QueryParams 类型参数的 DELETE 请求
     *
     * @param endpoint    请求 API
     * @param queryParams queryParams 类型请求路径上的参数 例如：/xxx?id=xxx
     * @param <O>         响应数据类型
     * @return 自定义响应结果。如果出错，则结果内容为错误信息
     */
    default <O extends Response> Mono<O> delete(String endpoint, Map<String, Object> queryParams) {
        return delete(endpoint, null, queryParams);
    }

    /**
     * 无参数的 DELETE 请求
     *
     * @param endpoint 请求 API
     * @param <O>      响应数据类型
     * @return 自定义响应结果。如果出错，则结果内容为错误信息
     */
    default <O extends Response> Mono<O> delete(String endpoint) {
        return delete(endpoint, null);
    }

    /**
     * PathVariables 类型参数的 DELETE 请求
     *
     * @param endpoint      请求 API
     * @param pathVariables pathVariables 类型请求路径参数 例如：/xxx/{id};
     * @param <O>           响应数据类型
     * @return 自定义响应结果。如果出错，则结果内容为错误信息
     */
    default <O extends Response> Mono<O> deleteWithPathVariables(String endpoint, Map<String, Object> pathVariables) {
        return delete(endpoint, pathVariables, null);
    }

    /**
     * PathVariables 类型参数的 DELETE 请求
     *
     * @param endpoint 请求 API
     * @param key      请求参数名
     * @param value    请求参数值
     * @param <O>      响应数据类型
     * @return 自定义响应结果。如果出错，则结果内容为错误信息
     */
    default <O extends Response> Mono<O> deleteWithPathVariables(String endpoint, String key, String value) {
        return deleteWithPathVariables(endpoint, Map.of(key, value));
    }

    /**
     * ContentType 为 "application/x-www-form-urlencoded" 类型的 DELETE 请求封装，
     *
     * @param endpoint      请求 API
     * @param formVariables 请求体数据 {@link Map}
     * @param pathVariables pathVariables 类型请求路径参数
     * @param <O>           响应数据类型
     * @return 自定义响应结果。如果出错，则结果内容为错误信息
     */
    default <O extends Response> Mono<O> deleteWithFormUrlEncoded(String endpoint, Map<String, Object> formVariables, Map<String, Object> pathVariables) {
        return formUrlEncoded(HttpMethod.DELETE, endpoint, formVariables, pathVariables);
    }

    /**
     * ContentType 为 "application/x-www-form-urlencoded" 类型的 DELETE 请求封装，
     *
     * @param endpoint      请求 API
     * @param formVariables 请求体数据 {@link Map}
     * @param <O>           响应数据类型
     * @return 自定义响应结果。如果出错，则结果内容为错误信息
     */
    default <O extends Response> Mono<O> deleteWithFormUrlEncoded(String endpoint, Map<String, Object> formVariables) {
        return deleteWithFormUrlEncoded(endpoint, formVariables, null);
    }

    /**
     * ContentType 为 "application/x-www-form-urlencoded" 类型的 DELETE 请求封装，
     *
     * @param endpoint 请求 API
     * @param key      请求参数名
     * @param value    请求参数值
     * @param <O>      响应数据类型
     * @return 自定义响应结果。如果出错，则结果内容为错误信息
     */
    default <O extends Response> Mono<O> deleteWithFormUrlEncoded(String endpoint, String key, String value) {
        return deleteWithFormUrlEncoded(endpoint, Map.of(key, value));
    }

    /**
     * ContentType 为 "application/json" 类型的 POST 请求封装，
     *
     * @param endpoint      请求 API
     * @param body          请求体数据对象
     * @param pathVariables pathVariables 类型请求路径参数
     * @param <I>           请求数据类型，用于 ContentType 为 "application/json" 类型请求
     * @param <O>           响应数据类型
     * @return 自定义响应结果。如果出错，则结果内容为错误信息
     */
    default <I, O extends Response> Mono<O> post(String endpoint, I body, Map<String, Object> pathVariables) {
        return json(HttpMethod.POST, endpoint, body, pathVariables);
    }

    /**
     * ContentType 为 "application/json" 类型的 POST 请求封装，
     *
     * @param endpoint 请求 API
     * @param body     请求体数据对象
     * @param <I>      请求数据类型，用于 ContentType 为 "application/json" 类型请求
     * @param <O>      响应数据类型
     * @return 自定义响应结果。如果出错，则结果内容为错误信息
     */
    default <I, O extends Response> Mono<O> post(String endpoint, I body) {
        return post(endpoint, body, null);
    }

    /**
     * ContentType 为 "application/x-www-form-urlencoded" 类型的 POST 请求封装，
     *
     * @param endpoint      请求 API
     * @param formVariables 请求体数据 {@link Map}
     * @param pathVariables pathVariables 类型请求路径参数
     * @param <O>           响应数据类型
     * @return 自定义响应结果。如果出错，则结果内容为错误信息
     */
    default <O extends Response> Mono<O> postWithFormUrlEncoded(String endpoint, Map<String, Object> formVariables, Map<String, Object> pathVariables) {
        return formUrlEncoded(HttpMethod.POST, endpoint, formVariables, pathVariables);
    }

    /**
     * ContentType 为 "application/x-www-form-urlencoded" 类型的 POST 请求封装，
     *
     * @param endpoint      请求 API
     * @param formVariables 请求体数据 {@link Map}
     * @param <O>           响应数据类型
     * @return 自定义响应结果。如果出错，则结果内容为错误信息
     */
    default <O extends Response> Mono<O> postWithFormUrlEncoded(String endpoint, Map<String, Object> formVariables) {
        return postWithFormUrlEncoded(endpoint, formVariables, null);
    }

    /**
     * ContentType 为 "application/json" 类型的 PUT 请求封装，
     *
     * @param endpoint      请求 API
     * @param body          请求体数据对象
     * @param pathVariables pathVariables 类型请求路径参数
     * @param <I>           请求数据类型，用于 ContentType 为 "application/json" 类型请求
     * @param <O>           响应数据类型
     * @return 自定义响应结果。如果出错，则结果内容为错误信息
     */
    default <I, O extends Response> Mono<O> put(String endpoint, I body, Map<String, Object> pathVariables) {
        return json(HttpMethod.PUT, endpoint, body, pathVariables);
    }

    /**
     * ContentType 为 "application/json" 类型的 PUT 请求封装，
     *
     * @param endpoint 请求 API
     * @param body     请求体数据对象
     * @param <I>      请求数据类型，用于 ContentType 为 "application/json" 类型请求
     * @param <O>      响应数据类型
     * @return 自定义响应结果。如果出错，则结果内容为错误信息
     */
    default <I, O extends Response> Mono<O> put(String endpoint, I body) {
        return put(endpoint, body, null);
    }

    /**
     * ContentType 为 "application/x-www-form-urlencoded" 类型的 PUT 请求封装，
     *
     * @param endpoint      请求 API
     * @param formVariables 请求体数据 {@link Map}
     * @param pathVariables pathVariables 类型请求路径参数
     * @param <O>           响应数据类型
     * @return 自定义响应结果。如果出错，则结果内容为错误信息
     */
    default <O extends Response> Mono<O> putWithFormUrlEncoded(String endpoint, Map<String, Object> formVariables, Map<String, Object> pathVariables) {
        return formUrlEncoded(HttpMethod.PUT, endpoint, formVariables, pathVariables);
    }

    /**
     * ContentType 为 "application/x-www-form-urlencoded" 类型的 PUT 请求封装，
     *
     * @param endpoint      请求 API
     * @param formVariables 请求体数据 {@link Map}
     * @param <O>           响应数据类型
     * @return 自定义响应结果。如果出错，则结果内容为错误信息
     */
    default <O extends Response> Mono<O> putWithFormUrlEncoded(String endpoint, Map<String, Object> formVariables) {
        return putWithFormUrlEncoded(endpoint, formVariables, null);
    }
}
