/*
 * DO NOT ALTER OR REMOVE COPYRIGHT NOTICES OR THIS HEADER.
 *
 * Copyright (c) 2020-2030 郑庚伟 ZHENGGENGWEI (码匠君), <herodotus@aliyun.com> Licensed under the AGPL License
 *
 * This file is part of Herodotus Stirrup.
 *
 * Herodotus Stirrup is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published
 * by the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * Herodotus Stirrup is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <https://www.herodotus.vip>.
 */

package cn.herodotus.stirrup.openapi.core.definition;

import cn.herodotus.stirrup.core.definition.domain.base.Response;
import cn.herodotus.stirrup.openapi.core.definition.converter.MapToMultiValueMapConverter;
import org.springframework.core.convert.converter.Converter;
import org.springframework.util.MultiValueMap;
import org.springframework.web.reactive.function.client.ClientRequest;
import org.springframework.web.reactive.function.client.ExchangeFilterFunction;
import org.springframework.web.reactive.function.client.WebClient;
import reactor.core.publisher.Mono;

import java.util.Map;
import java.util.function.Function;

/**
 * <p>Description: WebClient 请求操作抽象封装 </p>
 *
 * @author : gengwei.zheng
 * @date : 2024/2/27 21:46
 */
public abstract class AbstractWebClientTemplate implements WebClientTemplate {

    private final Converter<Map<String, Object>, MultiValueMap<String, String>> toMultiValueMap;
    private final WebClient webClient;

    public AbstractWebClientTemplate(WebClient webClient) {
        this.webClient = webClient;
        this.toMultiValueMap = new MapToMultiValueMapConverter();
    }

    @Override
    public Converter<Map<String, Object>, MultiValueMap<String, String>> getMultiValueMapConverter() {
        return this.toMultiValueMap;
    }

    @Override
    public WebClient getWebClient() {
        return this.webClient.mutate()
                .baseUrl(getBaseUrl())
                .filter(ExchangeFilterFunction.ofRequestProcessor(authentication()))
                .build();
    }

    protected abstract Function<ClientRequest, Mono<ClientRequest>> authentication();

    protected abstract String getBaseUrl();

    /**
     * QueryParams 类型参数的 GET 请求
     *
     * @param endpoint 请求 API
     * @param request  继承 {@link AbstractRequest} 并且设置了 Map 的参数对象
     * @param <I>      请求数据类型，只能为 {@link AbstractRequest} 子类
     * @param <O>      响应数据类型
     * @return 自定义响应结果。如果出错，则结果内容为错误信息
     */
    public <I extends AbstractRequest, O extends Response> Mono<O> get(String endpoint, I request) {
        return get(endpoint, request.getParams());
    }

    /**
     * QueryParams 类型参数的 DELETE 请求
     *
     * @param endpoint 请求 API
     * @param request  继承 {@link AbstractRequest} 并且设置了 Map 的参数对象
     * @param <I>      请求数据类型，只能为 {@link AbstractRequest} 子类
     * @param <O>      响应数据类型
     * @return 自定义响应结果。如果出错，则结果内容为错误信息
     */
    public <I extends AbstractRequest, O extends Response> Mono<O> delete(String endpoint, I request) {
        return delete(endpoint, request.getParams());
    }

    /**
     * ContentType 为 "application/x-www-form-urlencoded" 类型的 DELETE 请求封装，
     *
     * @param endpoint 请求 API
     * @param request  继承 {@link AbstractRequest} 并且设置了 Map 的参数对象
     * @param <I>      请求数据类型，只能为 {@link AbstractRequest} 子类
     * @param <O>      响应数据类型
     * @return 自定义响应结果。如果出错，则结果内容为错误信息
     */
    public <I extends AbstractRequest, O extends Response> Mono<O> deleteWithFormUrlEncoded(String endpoint, I request) {
        return deleteWithFormUrlEncoded(endpoint, request.getParams());
    }

    /**
     * ContentType 为 "application/x-www-form-urlencoded" 类型的 POST 请求封装，
     *
     * @param endpoint 请求 API
     * @param request  继承 {@link AbstractRequest} 并且设置了 Map 的参数对象
     * @param <I>      请求数据类型，只能为 {@link AbstractRequest} 子类
     * @param <O>      响应数据类型
     * @return 自定义响应结果。如果出错，则结果内容为错误信息
     */
    public <I extends AbstractRequest, O extends Response> Mono<O> postWithFormUrlEncoded(String endpoint, I request) {
        return postWithFormUrlEncoded(endpoint, request.getParams());
    }

    /**
     * ContentType 为 "application/x-www-form-urlencoded" 类型的 PUT 请求封装，
     *
     * @param endpoint 请求 API
     * @param request  继承 {@link AbstractRequest} 并且设置了 Map 的参数对象
     * @param <I>      请求数据类型，只能为 {@link AbstractRequest} 子类
     * @param <O>      响应数据类型
     * @return 自定义响应结果。如果出错，则结果内容为错误信息
     */
    public <I extends AbstractRequest, O extends Response> Mono<O> putWithFormUrlEncoded(String endpoint, I request) {
        return putWithFormUrlEncoded(endpoint, request.getParams());
    }
}
