/*
 * Copyright (c) 2020-2030 郑庚伟 ZHENGGENGWEI (码匠君) (herodotus@aliyun.com & www.herodotus.cn)
 *
 * Dante Engine licensed under the GNU LESSER GENERAL PUBLIC LICENSE 3.0;
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * <http://www.gnu.org/licenses/lgpl-3.0.html>
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package cn.herodotus.stirrup.oauth2.authorization.reactive;

import cn.herodotus.stirrup.oauth2.authorization.definition.HerodotusRequest;
import cn.herodotus.stirrup.oauth2.authorization.definition.HerodotusSecurityAttribute;
import cn.herodotus.stirrup.oauth2.authorization.processor.SecurityAttributeStorage;
import cn.herodotus.stirrup.web.reactive.utils.ExchangeUtils;
import cn.herodotus.stirrup.web.reactive.utils.HeaderUtils;
import org.apache.commons.collections4.CollectionUtils;
import org.apache.commons.collections4.MapUtils;
import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.http.HttpMethod;
import org.springframework.http.server.reactive.ServerHttpRequest;
import org.springframework.security.authorization.AuthenticatedReactiveAuthorizationManager;
import org.springframework.security.authorization.AuthorizationDecision;
import org.springframework.security.authorization.ReactiveAuthorizationManager;
import org.springframework.security.core.Authentication;
import org.springframework.security.web.server.authorization.AuthorizationContext;
import org.springframework.security.web.server.util.matcher.PathPatternParserServerWebExchangeMatcher;
import org.springframework.security.web.server.util.matcher.ServerWebExchangeMatcher;
import org.springframework.web.server.ServerWebExchange;
import reactor.core.publisher.Flux;
import reactor.core.publisher.Mono;

import java.util.List;
import java.util.Map;
import java.util.Optional;

/**
 * <p>Description: 响应式 Spring Security 6 授权管理器 </p>
 *
 * @author : gengwei.zheng
 * @date : 2024/1/26 21:35
 */
public class ReactiveSecurityAuthorizationManager implements ReactiveAuthorizationManager<AuthorizationContext> {

    private static final Logger log = LoggerFactory.getLogger(ReactiveSecurityAuthorizationManager.class);

    private final SecurityAttributeStorage securityAttributeStorage;
    private final ReactiveSecurityMatcherConfigurer reactiveSecurityMatcherConfigurer;

    public ReactiveSecurityAuthorizationManager(SecurityAttributeStorage securityAttributeStorage, ReactiveSecurityMatcherConfigurer reactiveSecurityMatcherConfigurer) {
        this.securityAttributeStorage = securityAttributeStorage;
        this.reactiveSecurityMatcherConfigurer = reactiveSecurityMatcherConfigurer;
    }

    @Override
    public Mono<AuthorizationDecision> check(Mono<Authentication> authentication, AuthorizationContext object) {

        return authentication.flatMap(auth -> {

            ServerWebExchange exchange = object.getExchange();

            String uri = ExchangeUtils.getRequestUri(exchange);
            HttpMethod httpMethod = ExchangeUtils.getHttpMethod(exchange);

            if (reactiveSecurityMatcherConfigurer.isPermitAllRequest(uri)) {
                log.trace("[Herodotus] |- Is white list resource : [{}], Passed!", uri);
                return isGranted(true);
            }

            String feignInnerFlag = HeaderUtils.getHerodotusFromIn(exchange);
            if (StringUtils.isNotBlank(feignInnerFlag)) {
                log.trace("[Herodotus] |- Is feign inner invoke : [{}], Passed!", uri);
                return isGranted(true);
            }

            if (reactiveSecurityMatcherConfigurer.isHasAuthenticatedRequest(uri)) {
                log.trace("[Herodotus] |- Is has authenticated resource : [{}]", uri);
                return isAuthenticated(Mono.just(auth), object);
            }

            List<HerodotusSecurityAttribute> configAttributes = findConfigAttribute(uri, httpMethod, exchange);

            if (CollectionUtils.isEmpty(configAttributes)) {
                log.warn("[Herodotus] |- NO PRIVILEGES : [{}].", uri);
                if (!reactiveSecurityMatcherConfigurer.isStrictMode()) {
                    return isAuthenticated(Mono.just(auth), object);
                }
                return isGranted(false);
            } else {
                return Flux.fromIterable(configAttributes)
                        .flatMap(item -> ReactiveExpressionAuthorizationManager.expression().check(Mono.just(auth), item))
                        .filter(AuthorizationDecision::isGranted)
                        .hasElements()
                        .flatMap(this::isGranted);
            }
        });
    }

    private Mono<AuthorizationDecision> isGranted(boolean granted) {
        return Mono.just(new AuthorizationDecision(granted));
    }

    private Mono<AuthorizationDecision> isAuthenticated(Mono<Authentication> authentication, AuthorizationContext object) {
        return AuthenticatedReactiveAuthorizationManager.authenticated().check(authentication, object);
    }

    private List<HerodotusSecurityAttribute> findConfigAttribute(String url, HttpMethod method, ServerWebExchange serverWebExchange) {

        log.debug("[Herodotus] |- Current Request is : [{}] - [{}]", url, method);

        List<HerodotusSecurityAttribute> configAttributes = this.securityAttributeStorage.getConfigAttribute(url, method.name());
        if (CollectionUtils.isNotEmpty(configAttributes)) {
            log.debug("[Herodotus] |- Get configAttributes from local storage for : [{}] - [{}]", url, method.name());
            return configAttributes;
        } else {
            Map<HerodotusRequest, List<HerodotusSecurityAttribute>> compatible = this.securityAttributeStorage.getCompatible();
            if (MapUtils.isNotEmpty(compatible)) {
                // 支持含有**通配符的路径搜索
                for (Map.Entry<HerodotusRequest, List<HerodotusSecurityAttribute>> entry : compatible.entrySet()) {
                    PathPatternParserServerWebExchangeMatcher matcher = toMatcher(entry.getKey());
                    Mono<ServerWebExchangeMatcher.MatchResult> result = matcher.matches(serverWebExchange);
                    Optional<ServerWebExchangeMatcher.MatchResult> optional = result.blockOptional();
                    if (optional.isPresent() && optional.get().isMatch()) {
                        log.debug("[Herodotus] |- Request match the wildcard [{}] - [{}]", entry.getKey(), entry.getValue());
                        return entry.getValue();
                    }
                }
            }
        }

        return null;
    }

    private PathPatternParserServerWebExchangeMatcher toMatcher(HerodotusRequest herodotusRequest) {
        HttpMethod httpMethod = null;
        if (StringUtils.isNotBlank(herodotusRequest.getHttpMethod())) {
            httpMethod = HttpMethod.valueOf(herodotusRequest.getHttpMethod());
        }

        return new PathPatternParserServerWebExchangeMatcher(herodotusRequest.getPattern(), httpMethod);
    }
}
