/*
 * Copyright (c) 2020-2030 郑庚伟 ZHENGGENGWEI (码匠君) (herodotus@aliyun.com & www.herodotus.cn)
 *
 * Dante Engine licensed under the GNU LESSER GENERAL PUBLIC LICENSE 3.0;
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * <http://www.gnu.org/licenses/lgpl-3.0.html>
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package cn.herodotus.stirrup.oauth2.authorization.reactive;

import cn.herodotus.stirrup.oauth2.authorization.definition.HerodotusSecurityAttribute;
import org.springframework.expression.EvaluationContext;
import org.springframework.expression.EvaluationException;
import org.springframework.expression.Expression;
import org.springframework.security.access.expression.method.DefaultMethodSecurityExpressionHandler;
import org.springframework.security.access.expression.method.MethodSecurityExpressionHandler;
import org.springframework.security.authorization.AuthorizationDecision;
import org.springframework.security.authorization.ReactiveAuthorizationManager;
import org.springframework.security.core.Authentication;
import reactor.core.publisher.Mono;

/**
 * <p>Description: 自定义 Reactive 表达式授权管理 </p>
 *
 * @author : gengwei.zheng
 * @date : 2024/2/2 21:09
 */
public class ReactiveExpressionAuthorizationManager implements ReactiveAuthorizationManager<HerodotusSecurityAttribute> {

    private final MethodSecurityExpressionHandler expressionHandler;

    public ReactiveExpressionAuthorizationManager() {
        this(new DefaultMethodSecurityExpressionHandler());
    }

    public ReactiveExpressionAuthorizationManager(MethodSecurityExpressionHandler expressionHandler) {
        this.expressionHandler = expressionHandler;
    }

    @Override
    public Mono<AuthorizationDecision> check(Mono<Authentication> authentication, HerodotusSecurityAttribute attribute) {
        Expression expression = expressionHandler.getExpressionParser().parseExpression(attribute.getExpression());
        return authentication
                .map((auth) -> expressionHandler.createEvaluationContext(auth, HerodotusSecurityAttribute.createMethodInvocation(attribute)))
                .flatMap((ctx) -> evaluateAsBoolean(expression, ctx))
                .map(AuthorizationDecision::new);
    }

    public static ReactiveExpressionAuthorizationManager expression() {
        return new ReactiveExpressionAuthorizationManager();
    }

    private static Mono<Boolean> evaluateAsBoolean(Expression expr, EvaluationContext ctx) {
        return Mono.defer(() -> {
            Object value;
            try {
                value = expr.getValue(ctx);
            } catch (EvaluationException ex) {
                return Mono.error(() -> new IllegalArgumentException(
                        "Failed to evaluate expression '" + expr.getExpressionString() + "'", ex));
            }
            if (value instanceof Boolean) {
                return Mono.just((Boolean) value);
            }
            if (value instanceof Mono<?> monoValue) {
                // @formatter:off
                return monoValue
                        .filter(Boolean.class::isInstance)
                        .map(Boolean.class::cast)
                        .switchIfEmpty(createInvalidReturnTypeMono(expr));
                // @formatter:on
            }
            return createInvalidReturnTypeMono(expr);
        });
    }

    private static Mono<Boolean> createInvalidReturnTypeMono(Expression expr) {
        return Mono.error(() -> new IllegalStateException(
                "Expression: '" + expr.getExpressionString() + "' must return boolean or Mono<Boolean>"));
    }
}
