/*
 * Copyright (c) 2020-2030 郑庚伟 ZHENGGENGWEI (码匠君) (herodotus@aliyun.com & www.herodotus.cn)
 *
 * Dante Engine licensed under the GNU LESSER GENERAL PUBLIC LICENSE 3.0;
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * <http://www.gnu.org/licenses/lgpl-3.0.html>
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package cn.herodotus.stirrup.oauth2.authorization.reactive;

import cn.herodotus.stirrup.core.definition.constants.BaseConstants;
import cn.herodotus.stirrup.core.foundation.context.ServiceContextHolder;
import cn.herodotus.stirrup.oauth2.core.definition.domain.HerodotusGrantedAuthority;
import org.apache.commons.lang3.StringUtils;
import org.springframework.boot.autoconfigure.security.oauth2.resource.OAuth2ResourceServerProperties;
import org.springframework.core.ParameterizedTypeReference;
import org.springframework.core.io.buffer.DataBuffer;
import org.springframework.core.io.buffer.DataBufferUtils;
import org.springframework.http.HttpHeaders;
import org.springframework.http.HttpStatus;
import org.springframework.http.MediaType;
import org.springframework.security.core.GrantedAuthority;
import org.springframework.security.oauth2.core.OAuth2AuthenticatedPrincipal;
import org.springframework.security.oauth2.core.OAuth2TokenIntrospectionClaimNames;
import org.springframework.security.oauth2.server.resource.introspection.BadOpaqueTokenException;
import org.springframework.security.oauth2.server.resource.introspection.OAuth2IntrospectionAuthenticatedPrincipal;
import org.springframework.security.oauth2.server.resource.introspection.OAuth2IntrospectionException;
import org.springframework.security.oauth2.server.resource.introspection.ReactiveOpaqueTokenIntrospector;
import org.springframework.util.Assert;
import org.springframework.web.reactive.function.BodyInserters;
import org.springframework.web.reactive.function.client.ClientResponse;
import org.springframework.web.reactive.function.client.WebClient;
import reactor.core.publisher.Mono;

import java.net.URI;
import java.time.Instant;
import java.util.*;

/**
 * <p>Description: 响应式的用于内省和验证OAuth 2.0令牌的合约 </p>
 * <p>
 * 基于 {@link org.springframework.security.oauth2.server.resource.introspection.SpringReactiveOpaqueTokenIntrospector} 改造而来。
 * 主要解决：
 * 1. 系统自动配置 introspectionUri，无需手动在配置文件中添加
 * 2. 权限数据的解析。Herodotus 是以 AUTHORITIES 作为权限，而不是 Scope
 *
 * @author : gengwei.zheng
 * @date : 2024/1/28 23:11
 */
public class HerodotusReactiveOpaqueTokenIntrospector implements ReactiveOpaqueTokenIntrospector {

    private static final String AUTHORITY_PREFIX = "SCOPE_";

    private static final ParameterizedTypeReference<Map<String, Object>> STRING_OBJECT_MAP = new ParameterizedTypeReference<Map<String, Object>>() {
    };

    private final URI introspectionUri;

    private final WebClient webClient;

    public HerodotusReactiveOpaqueTokenIntrospector(OAuth2ResourceServerProperties resourceServerProperties) {
        this(getIntrospectionUri(resourceServerProperties),
                resourceServerProperties.getOpaquetoken().getClientId(),
                resourceServerProperties.getOpaquetoken().getClientSecret());
    }

    /**
     * Creates a {@code OpaqueTokenReactiveAuthenticationManager} with the provided
     * parameters
     *
     * @param introspectionUri The introspection endpoint uri
     * @param clientId         The client id authorized to introspect
     * @param clientSecret     The client secret for the authorized client
     */
    public HerodotusReactiveOpaqueTokenIntrospector(String introspectionUri, String clientId, String clientSecret) {
        Assert.hasText(introspectionUri, "introspectionUri cannot be empty");
        Assert.hasText(clientId, "clientId cannot be empty");
        Assert.notNull(clientSecret, "clientSecret cannot be null");
        this.introspectionUri = URI.create(introspectionUri);
        this.webClient = WebClient.builder().defaultHeaders((h) -> h.setBasicAuth(clientId, clientSecret)).build();
    }

    /**
     * Creates a {@code OpaqueTokenReactiveAuthenticationManager} with the provided
     * parameters
     *
     * @param introspectionUri The introspection endpoint uri
     * @param webClient        The client for performing the introspection request
     */
    public HerodotusReactiveOpaqueTokenIntrospector(String introspectionUri, WebClient webClient) {
        Assert.hasText(introspectionUri, "introspectionUri cannot be null");
        Assert.notNull(webClient, "webClient cannot be null");
        this.introspectionUri = URI.create(introspectionUri);
        this.webClient = webClient;
    }

    private static String getIntrospectionUri(OAuth2ResourceServerProperties resourceServerProperties) {
        String introspectionUri = ServiceContextHolder.getInstance().getTokenIntrospectionUri();
        String configIntrospectionUri = resourceServerProperties.getOpaquetoken().getIntrospectionUri();
        if (StringUtils.isNotBlank(configIntrospectionUri)) {
            introspectionUri = configIntrospectionUri;
        }
        return introspectionUri;
    }

    @Override
    public Mono<OAuth2AuthenticatedPrincipal> introspect(String token) {
        return Mono.just(token)
                .flatMap(this::makeRequest)
                .flatMap(this::adaptToNimbusResponse)
                .map(this::convertClaimsSet)
                .onErrorMap((e) -> !(e instanceof OAuth2IntrospectionException), this::onError);
    }

    private Mono<ClientResponse> makeRequest(String token) {
        // @formatter:off
        // TODO: 过时方法采用最新方式替换
        return this.webClient.post()
                .uri(this.introspectionUri)
                .header(HttpHeaders.ACCEPT, MediaType.APPLICATION_JSON_VALUE)
                .body(BodyInserters.fromFormData("token", token))
                .exchange();
        // @formatter:on
    }

    private Mono<Map<String, Object>> adaptToNimbusResponse(ClientResponse responseEntity) {
        if (responseEntity.statusCode() != HttpStatus.OK) {
            // @formatter:off
            return responseEntity.bodyToFlux(DataBuffer.class)
                    .map(DataBufferUtils::release)
                    .then(Mono.error(new OAuth2IntrospectionException(
                            "Introspection endpoint responded with " + responseEntity.statusCode()))
                    );
            // @formatter:on
        }
        // relying solely on the authorization server to validate this token (not checking
        // 'exp', for example)
        return responseEntity.bodyToMono(STRING_OBJECT_MAP)
                .filter((body) -> (boolean) body.compute(OAuth2TokenIntrospectionClaimNames.ACTIVE, (k, v) -> {
                    if (v instanceof String) {
                        return Boolean.parseBoolean((String) v);
                    }
                    if (v instanceof Boolean) {
                        return v;
                    }
                    return false;
                }))
                .switchIfEmpty(Mono.error(() -> new BadOpaqueTokenException("Provided token isn't active")));
    }

    private OAuth2AuthenticatedPrincipal convertClaimsSet(Map<String, Object> claims) {
        claims.computeIfPresent(OAuth2TokenIntrospectionClaimNames.AUD, (k, v) -> {
            if (v instanceof String) {
                return Collections.singletonList(v);
            }
            return v;
        });
        claims.computeIfPresent(OAuth2TokenIntrospectionClaimNames.CLIENT_ID, (k, v) -> v.toString());
        claims.computeIfPresent(OAuth2TokenIntrospectionClaimNames.EXP,
                (k, v) -> Instant.ofEpochSecond(((Number) v).longValue()));
        claims.computeIfPresent(OAuth2TokenIntrospectionClaimNames.IAT,
                (k, v) -> Instant.ofEpochSecond(((Number) v).longValue()));
        // RFC-7662 page 7 directs users to RFC-7519 for defining the values of these
        // issuer fields.
        // https://datatracker.ietf.org/doc/html/rfc7662#page-7
        //
        // RFC-7519 page 9 defines issuer fields as being 'case-sensitive' strings
        // containing
        // a 'StringOrURI', which is defined on page 5 as being any string, but strings
        // containing ':'
        // should be treated as valid URIs.
        // https://datatracker.ietf.org/doc/html/rfc7519#section-2
        //
        // It is not defined however as to whether-or-not normalized URIs should be
        // treated as the same literal
        // value. It only defines validation itself, so to avoid potential ambiguity or
        // unwanted side effects that
        // may be awkward to debug, we do not want to manipulate this value. Previous
        // versions of Spring Security
        // would *only* allow valid URLs, which is not what we wish to achieve here.
        claims.computeIfPresent(OAuth2TokenIntrospectionClaimNames.ISS, (k, v) -> v.toString());
        claims.computeIfPresent(OAuth2TokenIntrospectionClaimNames.NBF,
                (k, v) -> Instant.ofEpochSecond(((Number) v).longValue()));
        Collection<GrantedAuthority> authorities = new ArrayList<>();
        claims.computeIfPresent(OAuth2TokenIntrospectionClaimNames.SCOPE, (k, v) -> v.toString());
        claims.computeIfPresent(BaseConstants.AUTHORITIES, (k, v) -> {
            if (v instanceof ArrayList) {
                List<String> values = (List<String>) v;
                values.forEach(value -> authorities.add(new HerodotusGrantedAuthority(value)));
            }
            return v;
        });
        return new OAuth2IntrospectionAuthenticatedPrincipal(claims, authorities);
    }

    private OAuth2IntrospectionException onError(Throwable ex) {
        return new OAuth2IntrospectionException(ex.getMessage(), ex);
    }
}
