/*
 * Copyright (c) 2020-2030 郑庚伟 ZHENGGENGWEI (码匠君) (herodotus@aliyun.com & www.herodotus.cn)
 *
 * Dante Engine licensed under the GNU LESSER GENERAL PUBLIC LICENSE 3.0;
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * <http://www.gnu.org/licenses/lgpl-3.0.html>
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package cn.herodotus.stirrup.oauth2.authorization.configuration;

import cn.herodotus.stirrup.oauth2.authorization.processor.SecurityAttributeAnalyzer;
import cn.herodotus.stirrup.oauth2.authorization.processor.SecurityAttributeStorage;
import cn.herodotus.stirrup.oauth2.authorization.properties.OAuth2AuthorizationProperties;
import cn.herodotus.stirrup.oauth2.authorization.reactive.*;
import jakarta.annotation.PostConstruct;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.boot.autoconfigure.condition.ConditionalOnMissingBean;
import org.springframework.boot.autoconfigure.condition.ConditionalOnWebApplication;
import org.springframework.boot.autoconfigure.security.oauth2.resource.OAuth2ResourceServerProperties;
import org.springframework.boot.context.properties.EnableConfigurationProperties;
import org.springframework.context.annotation.Bean;
import org.springframework.context.annotation.Configuration;
import org.springframework.context.annotation.Import;
import org.springframework.security.config.annotation.method.configuration.EnableReactiveMethodSecurity;
import org.springframework.web.reactive.resource.ResourceUrlProvider;

/**
 * <p>Description: OAuth2 响应式资源服务器配置 </p>
 * <p>
 * 1. Spring Security WebFlux 核心认证逻辑 {@link org.springframework.security.web.server.authentication.AuthenticationWebFilter}
 * 2. Spring Security WebFlux 核心授权逻辑 {@link org.springframework.security.web.server.authorization.AuthorizationWebFilter}
 *
 * @author : gengwei.zheng
 * @date : 2024/1/29 9:48
 */
@Configuration(proxyBeanMethods = false)
@ConditionalOnWebApplication(type = ConditionalOnWebApplication.Type.REACTIVE)
@EnableConfigurationProperties({OAuth2AuthorizationProperties.class})
@EnableReactiveMethodSecurity
@Import({
        ReactiveRestControllerAdvice.class
})
public class OAuth2ReactiveAuthorizationConfiguration {

    private static final Logger log = LoggerFactory.getLogger(OAuth2ReactiveAuthorizationConfiguration.class);

    @PostConstruct
    public void postConstruct() {
        log.debug("[Herodotus] |- Module [OAuth2 Reactive Authorization] Configure.");
    }

    @Bean
    @ConditionalOnMissingBean
    public SecurityAttributeStorage securityAttributeStorage() {
        SecurityAttributeStorage securityAttributeStorage = new SecurityAttributeStorage();
        log.trace("[Herodotus] |- Bean [Security Attribute Storage] Configure.");
        return securityAttributeStorage;
    }

    @Bean
    @ConditionalOnMissingBean
    public ReactiveSecurityMatcherConfigurer reactiveSecurityMatcherConfigurer(OAuth2AuthorizationProperties authorizationProperties, ResourceUrlProvider resourceUrlProvider) {
        ReactiveSecurityMatcherConfigurer configurer = new ReactiveSecurityMatcherConfigurer(authorizationProperties, resourceUrlProvider);
        log.trace("[Herodotus] |- Bean [Reactive Security Matcher Configurer] Configure.");
        return configurer;
    }

    @Bean
    @ConditionalOnMissingBean
    public ReactiveSecurityAuthorizationManager reactiveSecurityAuthorizationManager(SecurityAttributeStorage securityAttributeStorage, ReactiveSecurityMatcherConfigurer reactiveSecurityMatcherConfigurer) {
        ReactiveSecurityAuthorizationManager manager = new ReactiveSecurityAuthorizationManager(securityAttributeStorage, reactiveSecurityMatcherConfigurer);
        log.trace("[Herodotus] |- Bean [Reactive Security Authorization Manager] Configure.");
        return manager;
    }

    @Bean
    @ConditionalOnMissingBean
    public OAuth2AuthorizeExchangeSpecCustomizer oauth2AuthorizeExchangeSpecCustomizer(ReactiveSecurityMatcherConfigurer reactiveSecurityMatcherConfigurer, ReactiveSecurityAuthorizationManager reactiveSecurityAuthorizationManager) {
        OAuth2AuthorizeExchangeSpecCustomizer customizer = new OAuth2AuthorizeExchangeSpecCustomizer(reactiveSecurityMatcherConfigurer, reactiveSecurityAuthorizationManager);
        log.trace("[Herodotus] |- Bean [OAuth2 Authorize Exchange Spec Customizer] Configure.");
        return customizer;
    }

    @Bean
    @ConditionalOnMissingBean
    public OAuth2ResourceServerSpecCustomizer oauth2ResourceServerSpecCustomizer(OAuth2AuthorizationProperties authorizationProperties, OAuth2ResourceServerProperties resourceServerProperties) {
        OAuth2ResourceServerSpecCustomizer customizer = new OAuth2ResourceServerSpecCustomizer(authorizationProperties, resourceServerProperties);
        log.trace("[Herodotus] |- Bean [OAuth2 Resource Server Spec Customizer] Configure.");
        return customizer;
    }

    @Bean
    @ConditionalOnMissingBean
    public SecurityAttributeAnalyzer securityAttributeAnalyzer(SecurityAttributeStorage securityAttributeStorage, ReactiveSecurityMatcherConfigurer reactiveSecurityMatcherConfigurer) {
        SecurityAttributeAnalyzer securityAttributeAnalyzer = new SecurityAttributeAnalyzer(securityAttributeStorage, reactiveSecurityMatcherConfigurer.getPermitAllAttributes());
        log.trace("[Herodotus] |- Bean [Security Attribute Analyzer] Configure.");
        return securityAttributeAnalyzer;
    }
}
