/*
 * DO NOT ALTER OR REMOVE COPYRIGHT NOTICES OR THIS HEADER.
 *
 * Copyright (c) 2020-2030 郑庚伟 ZHENGGENGWEI (码匠君), <herodotus@aliyun.com> Licensed under the AGPL License
 *
 * This file is part of Herodotus Stirrup.
 *
 * Herodotus Stirrup is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published
 * by the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * Herodotus Stirrup is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <https://www.herodotus.vip>.
 */

package cn.herodotus.stirrup.logic.client.jpa.storage;

import cn.herodotus.stirrup.logic.client.jpa.converter.HerodotusToOAuth2AuthorizedClientConverter;
import cn.herodotus.stirrup.logic.client.jpa.converter.OAuth2ToHerodotusAuthorizedClientConverter;
import cn.herodotus.stirrup.logic.client.jpa.entity.HerodotusAuthorizedClient;
import cn.herodotus.stirrup.logic.client.jpa.service.HerodotusAuthorizedClientService;
import org.apache.commons.lang3.ObjectUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.core.convert.converter.Converter;
import org.springframework.security.core.Authentication;
import org.springframework.security.oauth2.client.OAuth2AuthorizedClient;
import org.springframework.security.oauth2.client.OAuth2AuthorizedClientService;
import org.springframework.security.oauth2.client.registration.ClientRegistrationRepository;
import org.springframework.util.Assert;

/**
 * <p>Description: 基于 JPA 扩展的 OAuth2AuthorizedClientService</p>
 *
 * @author : gengwei.zheng
 * @date : 2023/5/7 0:59
 */
public class JpaOAuth2AuthorizedClientService implements OAuth2AuthorizedClientService {

    private static final Logger log = LoggerFactory.getLogger(JpaOAuth2AuthorizedClientService.class);

    private final HerodotusAuthorizedClientService herodotusAuthorizedClientService;
    private final Converter<HerodotusAuthorizedClient, OAuth2AuthorizedClient> herodotusToOAuth2AuthorizedClientConverter;

    public JpaOAuth2AuthorizedClientService(HerodotusAuthorizedClientService herodotusAuthorizedClientService, ClientRegistrationRepository clientRegistrationRepository) {
        this.herodotusAuthorizedClientService = herodotusAuthorizedClientService;
        this.herodotusToOAuth2AuthorizedClientConverter = new HerodotusToOAuth2AuthorizedClientConverter(clientRegistrationRepository);
    }

    @SuppressWarnings("unchecked")
    @Override
    public <T extends OAuth2AuthorizedClient> T loadAuthorizedClient(String clientRegistrationId, String principalName) {
        Assert.hasText(clientRegistrationId, "clientRegistrationId cannot be empty");
        Assert.hasText(principalName, "principalName cannot be empty");

        OAuth2AuthorizedClient result = herodotusAuthorizedClientService.findByRegisteredClientIdAndPrincipalName(clientRegistrationId, principalName)
                .map(herodotusToOAuth2AuthorizedClientConverter::convert).orElse(null);
        return ObjectUtils.isNotEmpty(result) ? (T) result : null;
    }

    @Override
    public void saveAuthorizedClient(OAuth2AuthorizedClient authorizedClient, Authentication principal) {
        Assert.notNull(authorizedClient, "authorizedClient cannot be null");
        Assert.notNull(principal, "principal cannot be null");
        Converter<OAuth2AuthorizedClient, HerodotusAuthorizedClient> converter =
                new OAuth2ToHerodotusAuthorizedClientConverter(principal);
        HerodotusAuthorizedClient herodotusAuthorizedClient = converter.convert(authorizedClient);
        herodotusAuthorizedClientService.saveAndFlush(herodotusAuthorizedClient);
        log.debug("[Herodotus] |- Jpa OAuth2 Authorized Client Service save entity.");
    }

    @Override
    public void removeAuthorizedClient(String clientRegistrationId, String principalName) {
        Assert.hasText(clientRegistrationId, "clientRegistrationId cannot be empty");
        Assert.hasText(principalName, "principalName cannot be empty");

        herodotusAuthorizedClientService.deleteByClientRegistrationIdAndPrincipalName(clientRegistrationId, principalName);
        log.debug("[Herodotus] |- Jpa OAuth2 Authorized Client Service remove entity.");
    }
}
