/*
 * DO NOT ALTER OR REMOVE COPYRIGHT NOTICES OR THIS HEADER.
 *
 * Copyright (c) 2020-2030 郑庚伟 ZHENGGENGWEI (码匠君), <herodotus@aliyun.com> Licensed under the AGPL License
 *
 * This file is part of Herodotus Stirrup.
 *
 * Herodotus Stirrup is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published
 * by the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * Herodotus Stirrup is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <https://www.herodotus.vip>.
 */

package cn.herodotus.stirrup.logic.client.jpa.converter;

import cn.herodotus.stirrup.logic.client.jpa.entity.HerodotusAuthorizedClient;
import org.apache.commons.collections4.CollectionUtils;
import org.apache.commons.lang3.ObjectUtils;
import org.dromara.hutool.core.date.DateUtil;
import org.springframework.core.convert.converter.Converter;
import org.springframework.security.core.Authentication;
import org.springframework.security.oauth2.client.OAuth2AuthorizedClient;
import org.springframework.security.oauth2.client.registration.ClientRegistration;
import org.springframework.security.oauth2.core.OAuth2AccessToken;
import org.springframework.security.oauth2.core.OAuth2RefreshToken;
import org.springframework.util.StringUtils;

/**
 * <p>Description: OAuth2AuthorizedClient 转 HerodotusAuthorizedClient 转换器 </p>
 *
 * @author : gengwei.zheng
 * @date : 2023/5/22 18:24
 */
public class OAuth2ToHerodotusAuthorizedClientConverter implements Converter<OAuth2AuthorizedClient, HerodotusAuthorizedClient> {

    private final Authentication principal;

    public OAuth2ToHerodotusAuthorizedClientConverter(Authentication principal) {
        this.principal = principal;
    }

    @Override
    public HerodotusAuthorizedClient convert(OAuth2AuthorizedClient authorizedClient) {

        ClientRegistration clientRegistration = authorizedClient.getClientRegistration();
        OAuth2AccessToken accessToken = authorizedClient.getAccessToken();
        OAuth2RefreshToken refreshToken = authorizedClient.getRefreshToken();

        HerodotusAuthorizedClient herodotusAuthorizedClient = new HerodotusAuthorizedClient();
        herodotusAuthorizedClient.setClientRegistrationId(clientRegistration.getRegistrationId());
        herodotusAuthorizedClient.setPrincipalName(principal.getName());
        herodotusAuthorizedClient.setAccessTokenType(accessToken.getTokenType().getValue());
        herodotusAuthorizedClient.setAccessTokenValue(accessToken.getTokenValue());
        herodotusAuthorizedClient.setAccessTokenIssuedAt(DateUtil.toLocalDateTime(accessToken.getIssuedAt()));
        herodotusAuthorizedClient.setAccessTokenExpiresAt(DateUtil.toLocalDateTime(accessToken.getExpiresAt()));

        if (CollectionUtils.isNotEmpty(accessToken.getScopes())) {
            herodotusAuthorizedClient.setAccessTokenScopes(StringUtils.collectionToCommaDelimitedString(accessToken.getScopes()));
        }
        if (ObjectUtils.isNotEmpty(refreshToken)) {
            herodotusAuthorizedClient.setRefreshTokenValue(refreshToken.getTokenValue());
            herodotusAuthorizedClient.setRefreshTokenIssuedAt(DateUtil.toLocalDateTime(refreshToken.getIssuedAt()));
        }
        return herodotusAuthorizedClient;
    }
}
