/*
 * DO NOT ALTER OR REMOVE COPYRIGHT NOTICES OR THIS HEADER.
 *
 * Copyright (c) 2020-2030 郑庚伟 ZHENGGENGWEI (码匠君), <herodotus@aliyun.com> Licensed under the AGPL License
 *
 * This file is part of Herodotus Stirrup.
 *
 * Herodotus Stirrup is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published
 * by the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * Herodotus Stirrup is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <https://www.herodotus.vip>.
 */

package cn.herodotus.stirrup.logic.client.jpa.converter;

import cn.herodotus.stirrup.logic.client.jpa.entity.HerodotusAuthorizedClient;
import org.dromara.hutool.core.date.DateUtil;
import org.springframework.core.convert.converter.Converter;
import org.springframework.dao.DataRetrievalFailureException;
import org.springframework.security.oauth2.client.OAuth2AuthorizedClient;
import org.springframework.security.oauth2.client.registration.ClientRegistration;
import org.springframework.security.oauth2.client.registration.ClientRegistrationRepository;
import org.springframework.security.oauth2.core.OAuth2AccessToken;
import org.springframework.security.oauth2.core.OAuth2RefreshToken;
import org.springframework.util.StringUtils;

import java.time.Instant;
import java.time.LocalDateTime;
import java.util.Collections;
import java.util.Set;

/**
 * <p>Description: HerodotusAuthorizedClient 转 OAuth2AuthorizedClient 转换器 </p>
 *
 * @author : gengwei.zheng
 * @date : 2023/5/22 18:25
 */
public class HerodotusToOAuth2AuthorizedClientConverter implements Converter<HerodotusAuthorizedClient, OAuth2AuthorizedClient> {

    private final ClientRegistrationRepository clientRegistrationRepository;

    public HerodotusToOAuth2AuthorizedClientConverter(ClientRegistrationRepository clientRegistrationRepository) {
        this.clientRegistrationRepository = clientRegistrationRepository;
    }

    @Override
    public OAuth2AuthorizedClient convert(HerodotusAuthorizedClient authorizedClient) {
        String clientRegistrationId = authorizedClient.getClientRegistrationId();
        ClientRegistration clientRegistration = clientRegistrationRepository.findByRegistrationId(clientRegistrationId);
        if (clientRegistration == null) {
            throw new DataRetrievalFailureException(
                    "The ClientRegistration with id '" + clientRegistrationId + "' exists in the data source, "
                            + "however, it was not found in the ClientRegistrationRepository.");
        }

        // AccessToken
        OAuth2AccessToken.TokenType tokenType = null;
        if (OAuth2AccessToken.TokenType.BEARER.getValue().equalsIgnoreCase(authorizedClient.getAccessTokenType())) {
            tokenType = OAuth2AccessToken.TokenType.BEARER;
        }
        String tokenValue = authorizedClient.getAccessTokenValue();
        Instant issuedAt = DateUtil.toInstant(authorizedClient.getAccessTokenIssuedAt());
        Instant expiresAt = DateUtil.toInstant(authorizedClient.getAccessTokenExpiresAt());
        Set<String> scopes = Collections.emptySet();
        String accessTokenScopes = authorizedClient.getAccessTokenScopes();
        if (accessTokenScopes != null) {
            scopes = StringUtils.commaDelimitedListToSet(accessTokenScopes);
        }
        OAuth2AccessToken accessToken = new OAuth2AccessToken(tokenType, tokenValue, issuedAt, expiresAt, scopes);

        // RefreshToken
        OAuth2RefreshToken refreshToken = null;
        String refreshTokenValue = authorizedClient.getRefreshTokenValue();
        if (refreshTokenValue != null) {
            tokenValue = refreshTokenValue;
            issuedAt = null;
            LocalDateTime refreshTokenIssuedAt = authorizedClient.getRefreshTokenIssuedAt();
            if (refreshTokenIssuedAt != null) {
                issuedAt = DateUtil.toInstant(refreshTokenIssuedAt);
            }
            refreshToken = new OAuth2RefreshToken(tokenValue, issuedAt);
        }

        String principalName = authorizedClient.getPrincipalName();
        return new OAuth2AuthorizedClient(clientRegistration, principalName, accessToken, refreshToken);
    }
}
