/*
 * Copyright 2020-2030 码匠君<herodotus@aliyun.com>
 *
 * Dante Engine licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *    http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 * Dante Engine 是 Dante Cloud 系统核心组件库，采用 APACHE LICENSE 2.0 开源协议，您在使用过程中，需要注意以下几点：
 *
 * 1. 请不要删除和修改根目录下的LICENSE文件。
 * 2. 请不要删除和修改 Dante Engine 源码头部的版权声明。
 * 3. 请保留源码和相关描述文件的项目出处，作者声明等。
 * 4. 分发源码时候，请注明软件出处 <https://gitee.com/dromara/dante-cloud>
 * 5. 在修改包名，模块名称，项目代码等时，请注明软件出处 <https://gitee.com/dromara/dante-cloud>
 * 6. 若您的项目无法满足以上几点，可申请商业授权
 */

package cn.herodotus.engine.oauth2.authorization.config;

import cn.herodotus.engine.core.autoconfigure.oauth2.OAuth2AuthorizationProperties;
import cn.herodotus.engine.core.autoconfigure.oauth2.servlet.ServletOAuth2ResourceMatcherConfigurer;
import cn.herodotus.engine.core.foundation.condition.ConditionalOnServletApplication;
import cn.herodotus.engine.oauth2.authorization.processor.SecurityAttributeAnalyzer;
import cn.herodotus.engine.oauth2.authorization.processor.SecurityAttributeStorage;
import cn.herodotus.engine.oauth2.authorization.servlet.OAuth2SessionManagementConfigurerCustomer;
import cn.herodotus.engine.oauth2.authorization.servlet.ServletOAuth2AuthorizationConfigurerManager;
import cn.herodotus.engine.oauth2.authorization.servlet.ServletSecurityAuthorizationManager;
import cn.herodotus.engine.web.core.servlet.template.ThymeleafTemplateHandler;
import jakarta.annotation.PostConstruct;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.boot.autoconfigure.condition.ConditionalOnMissingBean;
import org.springframework.context.annotation.Bean;
import org.springframework.context.annotation.Configuration;
import org.springframework.context.annotation.Import;
import org.springframework.security.config.annotation.method.configuration.EnableMethodSecurity;
import org.springframework.security.oauth2.jwt.JwtDecoder;
import org.springframework.security.oauth2.server.resource.introspection.OpaqueTokenIntrospector;

/**
 * <p>Description: SecurityAttribute 配置 </p>
 *
 * @author : gengwei.zheng
 * @date : 2022/1/23 15:42
 */
@Configuration(proxyBeanMethods = false)
@ConditionalOnServletApplication
@EnableMethodSecurity(proxyTargetClass = true, securedEnabled = true, jsr250Enabled = true)
@Import({
        OAuth2ServletSessionConfiguration.class,
})
public class OAuth2ServletAuthorizationConfiguration {

    private static final Logger log = LoggerFactory.getLogger(OAuth2ServletAuthorizationConfiguration.class);

    @PostConstruct
    public void postConstruct() {
        log.debug("[Herodotus] |- Module [OAuth2 Servlet Authorization] Configure.");
    }

    @Bean
    @ConditionalOnMissingBean
    public SecurityAttributeStorage securityMetadataSourceStorage() {
        return new SecurityAttributeStorage();
    }

    @Bean
    @ConditionalOnMissingBean
    public ServletSecurityAuthorizationManager servletSecurityAuthorizationManager(SecurityAttributeStorage securityAttributeStorage, ServletOAuth2ResourceMatcherConfigurer servletOAuth2ResourceMatcherConfigurer) {
        ServletSecurityAuthorizationManager manager = new ServletSecurityAuthorizationManager(securityAttributeStorage, servletOAuth2ResourceMatcherConfigurer);
        log.trace("[Herodotus] |- Bean [Servlet Security Authorization Manager] Configure.");
        return manager;
    }

    @Bean
    @ConditionalOnMissingBean
    public SecurityAttributeAnalyzer securityAttributeAnalyzer(SecurityAttributeStorage securityAttributeStorage, ServletOAuth2ResourceMatcherConfigurer servletOAuth2ResourceMatcherConfigurer) {
        SecurityAttributeAnalyzer analyzer = new SecurityAttributeAnalyzer(securityAttributeStorage, servletOAuth2ResourceMatcherConfigurer);
        log.trace("[Herodotus] |- Bean [Security Attribute Analyzer] Configure.");
        return analyzer;
    }

    @Bean
    @ConditionalOnMissingBean
    public ServletOAuth2AuthorizationConfigurerManager servletOAuth2AuthorizationFacadeConfigurer(
            ThymeleafTemplateHandler thymeleafTemplateHandler,
            OAuth2AuthorizationProperties oauth2AuthorizationProperties,
            JwtDecoder jwtDecoder,
            OpaqueTokenIntrospector opaqueTokenIntrospector,
            OAuth2SessionManagementConfigurerCustomer sessionManagementConfigurerCustomer,
            ServletOAuth2ResourceMatcherConfigurer servletOAuth2ResourceMatcherConfigurer,
            ServletSecurityAuthorizationManager servletSecurityAuthorizationManager) {
        ServletOAuth2AuthorizationConfigurerManager configurer = new ServletOAuth2AuthorizationConfigurerManager(
                thymeleafTemplateHandler,
                oauth2AuthorizationProperties,
                jwtDecoder,
                opaqueTokenIntrospector,
                sessionManagementConfigurerCustomer,
                servletOAuth2ResourceMatcherConfigurer,
                servletSecurityAuthorizationManager);
        log.trace("[Herodotus] |- Bean [Servlet OAuth2 Resource Server Configurer] Configure.");
        return configurer;
    }
}
