package cn.hchub.groovy;

import lombok.extern.slf4j.Slf4j;
import org.w3c.dom.Document;
import org.w3c.dom.Element;
import org.w3c.dom.Node;
import org.w3c.dom.NodeList;
import org.xml.sax.InputSource;
import org.xml.sax.SAXException;

import javax.xml.parsers.DocumentBuilder;
import javax.xml.parsers.DocumentBuilderFactory;
import javax.xml.parsers.ParserConfigurationException;
import javax.xml.transform.Result;
import javax.xml.transform.Transformer;
import javax.xml.transform.TransformerException;
import javax.xml.transform.TransformerFactory;
import javax.xml.transform.dom.DOMSource;
import javax.xml.transform.stream.StreamResult;
import java.io.IOException;
import java.io.StringReader;
import java.io.StringWriter;

@Slf4j
public class GroovyXMLConfig {

    private GroovyXMLConfig() {

    }

    private static String DDD = "http://apache.org/xml/features/disallow-doctype-decl";
    private static final String SPRING_TAG = "lang:groovy";
    private static final String SPRING_TAG_ATTR_ID = "id";
    private static final String SPRING_TAG_ATTR_SS = "script-source";

    private static final String BEANS_FILE_CONTENT = "<beans xmlns='http://www.springframework.org/schema/beans'\n"
            + "    xmlns:lang='http://www.springframework.org/schema/lang'\n"
            + "    xmlns:xsi='http://www.w3.org/2001/XMLSchema-instance'\n"
            + "    xsi:schemaLocation='http://www.springframework.org/schema/beans http://www.springframework.org/schema/beans/spring-beans.xsd\n"
            + "                        http://www.springframework.org/schema/lang http://www.springframework.org/schema/lang/spring-lang.xsd'>"
            + "</beans>\n";

    private Document document;


    public static GroovyXMLConfig builder() throws ParserConfigurationException, IOException, SAXException {
        GroovyXMLConfig builder = new GroovyXMLConfig();
        builder.initDocument();
        return builder;
    }

    private void initDocument() throws ParserConfigurationException, IOException, SAXException {
        DocumentBuilderFactory factory = DocumentBuilderFactory.newInstance();
        factory.setFeature(DDD, true);
        factory.setValidating(true);
        factory.setNamespaceAware(true);

        DocumentBuilder builder = factory.newDocumentBuilder();
        StringReader reader = new StringReader(BEANS_FILE_CONTENT);
        this.document = builder.parse(new InputSource(reader));
    }

    public GroovyXMLConfig config(String id, String scriptSource) {
        boolean isExist = getElementByAttr(SPRING_TAG, SPRING_TAG_ATTR_ID, id) != null;
        if (isExist) {
            log.warn("tag {} attr {}={} exists", SPRING_TAG, SPRING_TAG_ATTR_ID, id);
            return this;
        }

        Element bean = document.createElement(SPRING_TAG);
        bean.setAttribute(SPRING_TAG_ATTR_ID, id);
        bean.setAttribute(SPRING_TAG_ATTR_SS, scriptSource);
        NodeList list = document.getElementsByTagName("beans");
        Node beans = list.item(0);
        beans.appendChild(bean);
        return this;
    }

    public String build() throws IOException, TransformerException {
        DOMSource source = new DOMSource(this.document);
        try (StringWriter writer = new StringWriter()) {
            Result result = new StreamResult(writer);
            Transformer transformer = TransformerFactory.newInstance().newTransformer();
            transformer.transform(source, result);
            return writer.getBuffer().toString();
        }
    }

    private Node getElementByAttr(String tagName, String attrName, String attrValue) {
        NodeList list = document.getElementsByTagName(tagName);
        for (int i = 0; i < list.getLength(); i++) {
            if (list.item(i).getAttributes().getNamedItem(attrName).getNodeValue().equals(attrValue)) {
                return list.item(i);
            }
        }
        return null;
    }

}
