package cn.hchub.groovy;

import com.google.common.collect.Sets;
import cn.hchub.groovy.script.GroovyScript;
import groovy.lang.GroovyClassLoader;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.BeansException;
import org.springframework.beans.factory.config.BeanPostProcessor;
import org.springframework.beans.factory.groovy.GroovyBeanDefinitionReader;
import org.springframework.beans.factory.support.BeanDefinitionRegistry;
import org.springframework.beans.factory.support.DefaultListableBeanFactory;
import org.springframework.context.ApplicationContext;
import org.springframework.context.ApplicationContextAware;
import org.springframework.context.ConfigurableApplicationContext;
import org.springframework.stereotype.Component;
import org.springframework.util.Assert;
import org.xml.sax.SAXException;

import javax.xml.parsers.ParserConfigurationException;
import javax.xml.transform.TransformerException;
import java.io.IOException;
import java.nio.charset.StandardCharsets;
import java.util.Arrays;
import java.util.Set;
import java.util.stream.Collectors;

@Slf4j
@Component
public class GroovyScriptBeanReader implements ApplicationContextAware {

    private ConfigurableApplicationContext applicationContext;

    private static final GroovyClassLoader groovyClassLoader = new GroovyClassLoader(GroovyScriptBeanReader.class.getClassLoader());

    public Set<String> loadScript(Set<GroovyScript> scripts) throws ParserConfigurationException, IOException, SAXException, TransformerException {
        Assert.notEmpty(scripts, "groovy script config is empty");
        GroovyXMLConfig builder = GroovyXMLConfig.builder();
        for (GroovyScript script : scripts) {
            groovyClassLoader.parseClass(script.getGroovyContent());
            builder.config(script.getBeanName(), StringUtils.join(DatabaseScriptSource.SCRIPT_SOURCE_PREFIX, script.getBeanName()));
        }
        String xml = builder.build();
        this.loadBeanDefinitions(xml);
        this.addBeanPostProcessor();
        Set<String> msgs = scripts.stream().map(script -> script.getId() + ":" + script.getBeanName()).collect(Collectors.toSet());
        return msgs;
    }

    public void unloadScript(Set<String> beanNames) {
        this.destroyBeanDefinition(beanNames);
        this.destroyScriptBeanFactory();
    }

    public Set<BeanElement> getBeans(Class<?> clazz) {
        String[] beanNames = applicationContext.getBeanNamesForType(clazz);
        if (beanNames == null || beanNames.length <= 0) {
            return Sets.newHashSet();
        }
        Set<BeanElement> elements = Arrays.stream(beanNames).map(beanName -> {
            Object bean = applicationContext.getBean(beanName, clazz);
            return BeanElement.builder().beanName(beanName).classSimpleName(bean.getClass().getCanonicalName()).build();
        }).collect(Collectors.toSet());
        return elements;
    }

    private void loadBeanDefinitions(String xml) {
//        XmlBeanDefinitionReader beanDefinitionReader = new XmlBeanDefinitionReader((BeanDefinitionRegistry) this.applicationContext.getBeanFactory());
//        beanDefinitionReader.setResourceLoader(this.applicationContext);
//        beanDefinitionReader.setBeanClassLoader(applicationContext.getClassLoader());
//        beanDefinitionReader.setEntityResolver(new ResourceEntityResolver(this.applicationContext));
//        beanDefinitionReader.loadBeanDefinitions(new GroovyByteArrayResource(xml.getBytes(StandardCharsets.UTF_8), "groovy.xml"));

        GroovyBeanDefinitionReader reader = new GroovyBeanDefinitionReader((BeanDefinitionRegistry) this.applicationContext.getBeanFactory());
        reader.loadBeanDefinitions(new GroovyByteArrayResource(xml.getBytes(StandardCharsets.UTF_8), "groovy_temp.xml"));
    }

    /**
     * 暂时不知道什么用途，待研究TODO
     */
    private void addBeanPostProcessor() {
        String[] postProcessorNames = applicationContext.getBeanFactory().getBeanNamesForType(GroovyScriptFactoryPostProcessor.class, true, false);
        for (String postProcessorName : postProcessorNames) {
            applicationContext.getBeanFactory().addBeanPostProcessor((BeanPostProcessor) applicationContext.getBean(postProcessorName));
        }
    }

    @Override
    public void setApplicationContext(ApplicationContext applicationContext) throws BeansException {
        this.applicationContext = (ConfigurableApplicationContext) applicationContext;
    }


    public void destroyBeanDefinition(Set<String> beanNames) {
        DefaultListableBeanFactory beanFactory = (DefaultListableBeanFactory) applicationContext.getAutowireCapableBeanFactory();
        for (String beanName : beanNames) {
            try {
                beanFactory.removeBeanDefinition(beanName);
                log.info("groovy script destroy bean definition:{}", beanName);
            } catch (Exception e) {
                log.error("groovy script destroy bean definition exception. skip:{}", e, beanName);
            }
        }
    }

    public void destroyScriptBeanFactory() {
        String[] postProcessorNames = applicationContext.getBeanFactory().getBeanNamesForType(GroovyScriptFactoryPostProcessor.class, true, false);
        for (String postProcessorName : postProcessorNames) {
            GroovyScriptFactoryPostProcessor processor = (GroovyScriptFactoryPostProcessor) applicationContext.getBean(postProcessorName);
            processor.destroy();
        }
    }

    public ConfigurableApplicationContext getApplicationContext() {
        return applicationContext;
    }
}
