package cn.happyloves.ali.tools.oss;

import cn.happyloves.ali.tools.oss.properties.OssProperties;
import com.aliyun.oss.OSSClient;
import com.aliyun.oss.OSSException;
import com.aliyun.oss.model.*;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.StringUtils;

import java.io.FileInputStream;
import java.io.InputStream;
import java.net.URL;
import java.text.SimpleDateFormat;
import java.util.*;

/**
 *
 */
@Slf4j
public class OssUtils {
    public static class OssBucket {
        private static final String SUCCESS = "success";
        private static final String FAILED = "failed";

        /**
         * 创建新的存储空间
         *
         * @param ossClient  OSS客户端
         * @param bucketName bucketName
         * @return 返回值
         */
        public static String createBucket(OSSClient ossClient, String bucketName) {
            CreateBucketRequest createBucketRequest = new CreateBucketRequest(bucketName);
            // 设置存储空间的权限为公共读，默认是私有。
            createBucketRequest.setCannedACL(CannedAccessControlList.PublicRead);
            // 设置存储空间的存储类型为低频访问类型，默认是标准类型。
            createBucketRequest.setStorageClass(StorageClass.IA);
            try {
                ossClient.createBucket(createBucketRequest);
                return SUCCESS;
            } catch (OSSException e) {
                log.error("error:{}", e.getMessage());
                return FAILED;
            }
        }

        /**
         * 列举存储空间
         *
         * @param bucketPrefix
         */
        public static List<Bucket> listBucket(OSSClient ossClient, String bucketPrefix) {
            ListBucketsRequest listBucketsRequest = new ListBucketsRequest();
            // 列举指定前缀的存储空间。
            if (StringUtils.isNotBlank(bucketPrefix)) {
                listBucketsRequest.setPrefix(bucketPrefix);
            }
            try {
                return ossClient.listBuckets(listBucketsRequest).getBucketList();
            } catch (OSSException e) {
                log.error("error:{}", e.getMessage());
                return new ArrayList<>();
            }
        }

        /**
         * 获取存储空间信息
         *
         * @param bucketName
         */
        public static Map getBucketInfo(OSSClient ossClient, String bucketName) {
            //判断空间是否存在
            boolean exists = ossClient.doesBucketExist(bucketName);
            if (!exists) {
                return new HashMap(1);
            }
            // 存储空间的信息包括地域（Region或Location）、创建日期（CreationDate）、拥有者（Owner）、权限（Grants）等。
            Map<String, Object> result = new HashMap<>(4);
            BucketInfo info = ossClient.getBucketInfo(bucketName);
            // 获取地域
            result.put("location", info.getBucket().getLocation());
            // 获取创建日期
            result.put("createTime", new SimpleDateFormat("yyyy-MM-dd HH:ss").format(info.getBucket().getCreationDate()));
            // 获取拥有者信息
            result.put("owner", info.getBucket().getOwner());
            // 获取权限信息
            AccessControlList acl = ossClient.getBucketAcl(bucketName);
            result.put("authority", acl.toString());
            return result;
        }

        /**
         * 删除存储空间
         *
         * @param bucketName
         */
        public static boolean deleteBucket(OSSClient ossClient, String bucketName) {
            //判断空间是否存在
            boolean exists = ossClient.doesBucketExist(bucketName);
            if (!exists) {
                return false;
            }
            // 删除存储空间
            ossClient.deleteBucket(bucketName);
            return true;
        }
    }

    public static class OssUpload {
        /**
         * listFiles 列举文件
         * 列举文件。 prefix，则列举存储空间下所有的文件。反之prefix，则列举包含指定前缀的文件。
         */
        public static List<String> listFiles(OSSClient ossClient, OssProperties aliyunOssProperties, String prefix) {
            ObjectListing objectListing = ossClient.listObjects(aliyunOssProperties.getBucketName(), prefix);
            List<OSSObjectSummary> ossObjectSummaries = objectListing.getObjectSummaries();
            List<String> keys = new ArrayList<>();
            for (OSSObjectSummary ossObjectSummary : ossObjectSummaries) {
                log.info("key name:{}", ossObjectSummary.getKey());
                keys.add(ossObjectSummary.getKey());
            }
            return keys;
        }

        /**
         * DeleteSpecifiedPrefixFile 删除指定前缀文件
         * 列举所有包含指定前缀的文件并删除
         */
        public static List<String> deleteSpecifiedPrefixFile(OSSClient ossClient, OssProperties aliyunOssProperties, String prefix) {
            String bucketName = aliyunOssProperties.getBucketName();
            String nextMarker = null;
            List<String> keys = new ArrayList<>();
            ObjectListing objectListing = null;
            do {
                ListObjectsRequest listObjectsRequest = new ListObjectsRequest(bucketName)
                        .withPrefix(prefix)
                        .withMarker(nextMarker);
                objectListing = ossClient.listObjects(listObjectsRequest);
                if (objectListing.getObjectSummaries().size() > 0) {
                    for (OSSObjectSummary ossObjectSummary : objectListing.getObjectSummaries()) {
                        log.info("key name:{}", ossObjectSummary.getKey());
                        keys.add(ossObjectSummary.getKey());
                    }
                    List<String> files = new ArrayList<>(keys.size());
                    keys.forEach(item -> files.add(aliyunOssProperties.getHome().concat("/").concat(item)));
                    DeleteObjectsRequest deleteObjectsRequest = new DeleteObjectsRequest(bucketName).withKeys(files);
                    ossClient.deleteObjects(deleteObjectsRequest);
                }
                nextMarker = objectListing.getNextMarker();
            } while (objectListing.isTruncated());
            return keys;
        }

        /**
         * batchDeleteFile 批量删除文件
         * 删除文件。key等同于ObjectName，表示删除OSS文件时需要指定包含文件后缀在内的完整路径，例如abc/efg/123.jpg。
         */
        public static List<String> batchDeleteFile(OSSClient ossClient, OssProperties aliyunOssProperties, List<String> keys) {
            List<String> files = new ArrayList<>(keys.size());
            keys.forEach(item -> files.add(aliyunOssProperties.getHome().concat("/").concat(item)));
            DeleteObjectsResult deleteObjectsResult = ossClient.deleteObjects(new DeleteObjectsRequest(aliyunOssProperties.getBucketName()).withKeys(files));
            List<String> deletedObjects = deleteObjectsResult.getDeletedObjects();
            return deletedObjects;
        }

        /**
         * deleteFile 单个文件删除
         * 删除文件。如需删除文件夹，请将ObjectName设置为对应的文件夹名称。如果文件夹非空，则需要将文件夹下的所有object删除后才能删除该文件夹。
         */
        public static void deleteFile(OSSClient ossClient, OssProperties aliyunOssProperties, String objectName) {
            ossClient.deleteObject(aliyunOssProperties.getBucketName(), aliyunOssProperties.getHome().concat("/").concat(objectName));
        }


        /**
         * checkFileWhetherExist
         */
        public static boolean checkFileWhetherExist(OSSClient ossClient, OssProperties aliyunOssProperties, String fileName) {
            boolean whetherExist = ossClient.doesObjectExist(aliyunOssProperties.getBucketName(), aliyunOssProperties.getHome().concat("/").concat(fileName));
            return whetherExist;
        }

        /**
         * filePath上传
         */
        public static String uploadByFilePath(OSSClient ossClient, OssProperties aliyunOssProperties, String fileName, String filePath) throws Exception {
            return upload(ossClient, aliyunOssProperties, fileName, new FileInputStream(filePath));
        }

        /**
         * url上传
         */
        public static String uploadByUrl(OSSClient ossClient, OssProperties aliyunOssProperties, String fileName, String url) throws Exception {
            return upload(ossClient, aliyunOssProperties, fileName, new URL(url).openStream());
        }

        /**
         * 上传临时文件（设置过期时间不能访问，文件还是存在）
         */
        public static String upload(OSSClient ossClient, OssProperties aliyunOssProperties, String fileName, InputStream inputStream, long expireTime) throws Exception {
            String home = aliyunOssProperties.getHome();
            String cdn = aliyunOssProperties.getCdn();
            String bucketName = aliyunOssProperties.getBucketName();
            String endpoint = aliyunOssProperties.getEndpoint();
            fileName = home.concat("/").concat(fileName);
            boolean whetherIsExist = checkFileWhetherExist(ossClient, aliyunOssProperties, fileName);
            if (whetherIsExist) {
                throw new RuntimeException("the file is exist!!!");
            }
            ObjectMetadata metadata = new ObjectMetadata();
            metadata.setObjectAcl(CannedAccessControlList.PublicReadWrite);
            PutObjectRequest request = new PutObjectRequest(bucketName, fileName, inputStream, metadata);
            Date expire = new Date(System.currentTimeMillis() + expireTime * 1000);
            ossClient.putObject(request);
            inputStream.close();
            String url = ossClient.generatePresignedUrl(bucketName, fileName, expire).toString();
            log.info("url:{}", url);
            if (StringUtils.isNotBlank(cdn)) {
                return ossClient.generatePresignedUrl(bucketName, fileName, expire).toString().replaceFirst(bucketName.concat(".").concat(endpoint), aliyunOssProperties.getCdn());
            } else {
                return url;
            }
        }


        /**
         * 默认上传
         */
        public static String upload(OSSClient ossClient, OssProperties aliyunOssProperties, String fileName, InputStream inputStream) throws Exception {
            String home = aliyunOssProperties.getHome();
            String bucketName = aliyunOssProperties.getBucketName();
            boolean whetherIsExist = checkFileWhetherExist(ossClient, aliyunOssProperties, fileName);
            if (whetherIsExist) {
                throw new RuntimeException("the file is exist!!!");
            }
            ObjectMetadata metadata = new ObjectMetadata();
            metadata.setObjectAcl(CannedAccessControlList.PublicReadWrite);
            PutObjectResult putObjectResult = ossClient.putObject(bucketName, home.concat("/").concat(fileName), inputStream, metadata);
            inputStream.close();
            String url = generateUrl(aliyunOssProperties, fileName);
            log.info("url:{}", url);
            return url;
        }

        /**
         * 生成地址(如果cdn为空则返回oss域名)
         * oss:https://bucketNmae.oss-cn-beijing.aliyuncs.com/upload/xxx.xxx
         * cdn:https://xxx.xxx.com/upload/xxx.xxx
         */
        private static String generateUrl(OssProperties aliyunOssProperties, String fileName) {
            String cdn = aliyunOssProperties.getCdn();
            String home = aliyunOssProperties.getHome();
            if (StringUtils.isNotBlank(cdn)) {
                return String.format("https://%s/%s/%s", cdn, home, fileName);
            } else {
                return String.format("https://%s.%s/%s/%s", aliyunOssProperties.getBucketName(), aliyunOssProperties.getEndpoint(), home, fileName);
            }
        }

    }
}
