package cn.hamm.airpower.exception;

import cn.hamm.airpower.config.GlobalConfig;
import cn.hamm.airpower.result.Result;
import cn.hamm.airpower.result.ResultException;
import cn.hamm.airpower.result.json.Json;
import cn.hamm.airpower.result.json.JsonData;
import jakarta.validation.ConstraintViolation;
import jakarta.validation.ConstraintViolationException;
import lombok.extern.slf4j.Slf4j;
import org.jetbrains.annotations.NotNull;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.dao.DataIntegrityViolationException;
import org.springframework.dao.InvalidDataAccessResourceUsageException;
import org.springframework.data.mapping.PropertyReferenceException;
import org.springframework.data.redis.RedisConnectionFailureException;
import org.springframework.http.HttpStatus;
import org.springframework.http.converter.HttpMessageNotReadableException;
import org.springframework.transaction.CannotCreateTransactionException;
import org.springframework.validation.BindingResult;
import org.springframework.validation.FieldError;
import org.springframework.web.HttpMediaTypeNotSupportedException;
import org.springframework.web.HttpRequestMethodNotSupportedException;
import org.springframework.web.bind.MethodArgumentNotValidException;
import org.springframework.web.bind.annotation.ControllerAdvice;
import org.springframework.web.bind.annotation.ExceptionHandler;
import org.springframework.web.bind.annotation.ResponseBody;
import org.springframework.web.bind.annotation.ResponseStatus;
import org.springframework.web.servlet.NoHandlerFoundException;

import java.sql.SQLIntegrityConstraintViolationException;
import java.util.List;
import java.util.Objects;
import java.util.Set;

/**
 * <h1>全局异常处理拦截器</h1>
 *
 * @author Hamm
 * @see Result
 */
@SuppressWarnings({"SingleStatementInBlock", "CallToPrintStackTrace"})
@ControllerAdvice
@ResponseStatus(HttpStatus.OK)
@ResponseBody
@Slf4j
public class ExceptionInterceptor {
    @Autowired
    private GlobalConfig globalConfig;

    /**
     * 参数验证失败
     */
    @ExceptionHandler(MethodArgumentNotValidException.class)
    public Json badRequestHandle(@NotNull MethodArgumentNotValidException exception) {
        log.error(exception.getMessage());
        BindingResult result = exception.getBindingResult();
        StringBuilder stringBuilder = new StringBuilder();
        if (!result.hasErrors()) {
            return new Json(Result.PARAM_INVALID);
        }
        if (!result.hasFieldErrors()) {
            return new Json(Result.PARAM_INVALID);
        }
        List<FieldError> errors = result.getFieldErrors();
        for (FieldError error : errors) {
            stringBuilder
                    .append(error.getDefaultMessage())
                    .append("(")
                    .append(error.getField())
                    .append(")");
            break;
        }
        return new Json(Result.PARAM_INVALID, stringBuilder.toString());
    }

    /**
     * 参数校验失败
     */
    @ExceptionHandler(ConstraintViolationException.class)
    public Json badRequestHandle(@NotNull ConstraintViolationException exception) {
        log.error(exception.getMessage());
        StringBuilder stringBuilder = new StringBuilder();
        Set<ConstraintViolation<?>> errors = exception.getConstraintViolations();
        for (ConstraintViolation<?> error : errors) {
            stringBuilder
                    .append(error.getMessage())
                    .append("(")
                    .append(error.getInvalidValue())
                    .append(")");
            break;
        }
        return new Json(Result.PARAM_INVALID, stringBuilder.toString());
    }

    /**
     * 删除时的数据关联校验异常
     */
    @ExceptionHandler({SQLIntegrityConstraintViolationException.class, DataIntegrityViolationException.class})
    public Json deleteUsingDataException(@NotNull Exception exception) {
        log.error(exception.getMessage());
        if (globalConfig.isDebug()) {
            exception.printStackTrace();
        }
        return new Json(Result.FORBIDDEN_DELETE_USED, "数据正在使用中,无法被删除!");
    }

    /**
     * 访问的接口没有实现
     */
    @ExceptionHandler(NoHandlerFoundException.class)
    public Json notFoundHandle(@NotNull Exception exception) {
        log.error(exception.getMessage());
        return new Json(Result.API_SERVICE_UNSUPPORTED);
    }

    /**
     * 请求的数据不是标准JSON
     */
    @ExceptionHandler(HttpMessageNotReadableException.class)
    public Json dataExceptionHandle(@NotNull Exception exception) {
        log.error(exception.getMessage());
        return new Json(Result.REQUEST_CONTENT_TYPE_UNSUPPORTED, "请求参数格式不正确,请检查是否接口支持的JSON");
    }

    /**
     * 不支持的请求方法
     */
    @ExceptionHandler(HttpRequestMethodNotSupportedException.class)
    public Json methodExceptionHandle(@NotNull HttpRequestMethodNotSupportedException exception) {
        log.error(exception.getMessage());
        String supportedMethod = String.join("|", Objects.requireNonNull(exception.getSupportedMethods()));
        return new Json(
                Result.REQUEST_METHOD_UNSUPPORTED,
                exception.getMethod() + "不被支持,请使用" + supportedMethod + "方法请求");
    }

    /**
     * 不支持的数据类型
     */
    @ExceptionHandler(HttpMediaTypeNotSupportedException.class)
    public Json httpMediaTypeNotSupportedExceptionHandle(@NotNull HttpMediaTypeNotSupportedException exception) {
        log.error(exception.getMessage());
        if (globalConfig.isDebug()) {
            exception.printStackTrace();
        }
        return new Json(
                Result.REQUEST_CONTENT_TYPE_UNSUPPORTED,
                Objects.requireNonNull(exception.getContentType()) + "不被支持,请使用JSON请求");
    }

    /**
     * 数据库连接发生错误
     */
    @ExceptionHandler(CannotCreateTransactionException.class)
    public Json databaseExceptionHandle(@NotNull Exception exception) {
        log.error(exception.getMessage());
        if (globalConfig.isDebug()) {
            exception.printStackTrace();
        }
        return new Json(Result.DATABASE_ERROR);
    }

    /**
     * REDIS连接发生错误
     */
    @ExceptionHandler(RedisConnectionFailureException.class)
    public Json redisExceptionHandle(@NotNull Exception exception) {
        log.error(exception.getMessage());
        if (globalConfig.isDebug()) {
            exception.printStackTrace();
        }
        return new Json(Result.REDIS_ERROR);
    }

    /**
     * 自定义业务异常
     */
    @ExceptionHandler(ResultException.class)
    public JsonData customExceptionHandle(@NotNull ResultException result) {
        return new JsonData(result.getData(), result.getMessage(), result.getCode());
    }

    /**
     * JWT校验失败错误
     */
    @ExceptionHandler(value = {cn.hutool.jwt.JWTException.class})
    public Json jwtExceptionHandle(@NotNull Exception exception) {
        log.error(exception.getMessage());
        if (globalConfig.isDebug()) {
            exception.printStackTrace();
        }
        return new Json(Result.UNAUTHORIZED);
    }

    /**
     * 数据字段不存在
     */
    @ExceptionHandler(value = PropertyReferenceException.class)
    public Json propertyReferenceExceptionHandle(@NotNull PropertyReferenceException exception) {
        log.error(exception.getMessage());
        if (globalConfig.isDebug()) {
            exception.printStackTrace();
        }
        return new Json(Result.DATABASE_UNKNOWN_FIELD, "不支持的数据字段" + exception.getPropertyName());
    }

    /**
     * 数据表或字段异常
     */
    @ExceptionHandler(value = InvalidDataAccessResourceUsageException.class)
    public Json invalidDataAccessResourceUsageExceptionHandle(@NotNull Exception exception) {
        log.error(exception.getMessage());
        if (globalConfig.isDebug()) {
            exception.printStackTrace();
        }
        return new Json(Result.DATABASE_TABLE_OR_FIELD_ERROR);
    }

    /**
     * 其他异常
     */
    @ExceptionHandler(value = {Exception.class, RuntimeException.class})
    public Object otherExceptionHandle(@NotNull Exception exception) {
        log.error(exception.getMessage());
        if (globalConfig.isDebug()) {
            exception.printStackTrace();
        }
        return new Json(Result.ERROR);
    }
}