package cn.gybyt.web.plugins;

import cn.gybyt.web.util.SpringUtil;
import org.apache.ibatis.builder.xml.XMLMapperBuilder;
import org.apache.ibatis.session.Configuration;
import org.apache.ibatis.session.SqlSessionFactory;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.core.io.Resource;
import org.springframework.core.io.support.PathMatchingResourcePatternResolver;

import javax.annotation.PostConstruct;
import java.io.IOException;
import java.lang.reflect.Field;
import java.util.*;

/**
 * mapper 刷新插件
 *
 * @classname: MybatisMapperRefresh
 * @author: codetiger
 * @create: 2023/3/16 18:49
 **/
public class GybytMybatisMapperRefreshPlugin {

    /**
     * 日志对象
     */
    private final Logger log = LoggerFactory.getLogger(GybytMybatisMapperRefreshPlugin.class);
    /**
     * sqlSessionFactory 对象
     */
    private SqlSessionFactory sqlSessionFactory;
    /**
     * mapper文件列表
     */
    private Resource[] mapperLocations;
    /**
     * mapper文件位置
     */
    private String packageSearchPath;
    /**
     * 刷新延迟
     */
    private Long refreshInterval;
    /**
     * 存储更改的文件
     */
    private List<String> changeList;
    /**
     * 记录文件是否变化
     */
    private HashMap<String, Long> fileMapping = new HashMap<>();

    public GybytMybatisMapperRefreshPlugin(String packageSearchPath, Long refreshInterval) {
        this.sqlSessionFactory = SpringUtil.getBean(SqlSessionFactory.class);
        this.packageSearchPath = packageSearchPath;
        this.refreshInterval = refreshInterval;
    }

    /**
     * 初始化方法
     */
    public void init() {
        final GybytMybatisMapperRefreshPlugin ruRefresh = this;
        new Thread(() -> {
            while (true) {
                try {
                    ruRefresh.refreshMapper();
                } catch (Exception e1) {
                    log.error("刷新失败");
                }
                try {
                    Thread.sleep(refreshInterval * 1000);
                } catch (InterruptedException e) {
                    log.error("刷新失败");
                }
            }
        }, "Thread-Mybatis-Refresh").start();
    }

    /**
     * 刷新mapper缓存
     */
    private void refreshMapper() {
        try {
            Configuration configuration = this.sqlSessionFactory.getConfiguration();
            try {
                this.scanMapperXml();
            } catch (IOException e) {
                log.error("扫描包路径配置错误");
                return;
            }
            if (this.isChanged()) {
                this.removeConfig(configuration);
                for (Resource configLocation : mapperLocations) {
                    try {
                        XMLMapperBuilder xmlMapperBuilder = new XMLMapperBuilder(configLocation.getInputStream(),
                                configuration, configLocation.toString(), configuration.getSqlFragments());
                        xmlMapperBuilder.parse();
                        if (changeList.contains(configLocation.getFilename())) {
                            log.info("[" + configLocation.getFilename() + "] refresh finished");
                        }
                    } catch (IOException e) {
                        log.error("[" + configLocation.getFilename() + "] refresh error");
                    }
                }
            }
        } catch (Exception e) {
            log.error(e.getMessage());
        }
    }

    /**
     * 扫描xml文件所在的路径
     *
     * @throws IOException
     */
    private void scanMapperXml() throws IOException {
        this.mapperLocations = new PathMatchingResourcePatternResolver().getResources(packageSearchPath);
    }

    /**
     * 清空Configuration中几个重要的缓存
     *
     * @param configuration
     * @throws Exception
     */
    private void removeConfig(Configuration configuration) throws Exception {
        Class<?> classConfig = configuration.getClass();
        clearMap(classConfig, configuration, "mappedStatements");
        clearMap(classConfig, configuration, "caches");
        clearMap(classConfig, configuration, "resultMaps");
        clearMap(classConfig, configuration, "parameterMaps");
        clearMap(classConfig, configuration, "keyGenerators");
        clearMap(classConfig, configuration, "sqlFragments");
        clearSet(classConfig, configuration, "loadedResources");
    }

    /**
     * 清除缓存map
     *
     * @param classConfig
     * @param configuration
     * @param fieldName
     * @throws Exception
     */
    private void clearMap(Class<?> classConfig, Configuration configuration, String fieldName) throws Exception {
        Field field = classConfig.getDeclaredField(fieldName);
        field.setAccessible(true);
        Map mapConfig = (Map) field.get(configuration);
        mapConfig.clear();
    }

    /**
     * 清除缓存set
     *
     * @param classConfig
     * @param configuration
     * @param fieldName
     * @throws Exception
     */
    private void clearSet(Class<?> classConfig, Configuration configuration, String fieldName) throws Exception {
        Field field = classConfig.getDeclaredField(fieldName);
        field.setAccessible(true);
        Set setConfig = (Set) field.get(configuration);
        setConfig.clear();
    }

    /**
     * 判断文件是否发生了变化
     *
     * @return
     * @throws IOException
     */
    private boolean isChanged() throws IOException {
        boolean flag = false;
        changeList = new ArrayList<>();
        // 遍历文件mapper列表
        for (Resource resource : mapperLocations) {
            String resourceName = resource.getFilename();
            // 新增标识
            boolean addFlag = !fileMapping.containsKey(resourceName);
            // 修改文件:判断文件内容是否有变化
            Long compareFrame = fileMapping.get(resourceName);
            long lastFrame = resource.contentLength() + resource.lastModified();
            // 修改标识
            boolean modifyFlag = null != compareFrame && compareFrame != lastFrame;
            // 新增或是修改时,存储文件
            if (addFlag || modifyFlag) {
                // 记录文件内容帧值
                fileMapping.put(resourceName, lastFrame);
                flag = true;
                changeList.add(resource.getFilename());
            }
        }
        return flag;
    }

}
