package cn.gongler.util.text;

/**
 * @author honger
 * date 2022/10/6
 */

import cn.gongler.util.bytes.HexUtil;

import java.math.RoundingMode;
import java.text.NumberFormat;
import java.util.Arrays;
import java.util.Objects;

import static cn.gongler.util.GonglerUtil.WithDefault;

/**
 * 字符串连接器。解决SpringJoiner不能插入非字符传的问题
 */
public class StringLinker implements CharSequence, java.io.Serializable {

    public static StringLinker of() {
        return new StringLinker("");
    }

    public static StringLinker of(CharSequence delimiter) {
        return new StringLinker(delimiter);
    }

    public static StringLinker of(CharSequence prefix,
                                  CharSequence delimiter,
                                  CharSequence suffix) {
        return new StringLinker(delimiter).wrapper(prefix, suffix);
    }

    private final String delimiter;
    private String prefix = "";
    private String suffix = "";

    private String defaultStringIfNull = null;

    /*
     * By default, the string consisting of prefix+suffix, returned by
     * toString(), or properties of value, when no elements have yet been added,
     * i.e. when it is empty.  This may be overridden by the user to be some
     * other value including the empty String.
     */
    private String emptyValue;

    /*
     * StringBuilder value -- at any time, the characters constructed from the
     * prefix, the added element separated by the delimiter, but without the
     * suffix, so that we can more easily add elements without having to jigger
     * the suffix each time.
     */
    private StringBuilder value;

    /**
     * Constructs a {@code StringJoiner} with no characters in it, with no
     * {@code prefix} or {@code suffix}, and a copy of the supplied
     * {@code delimiter}.
     * If no characters are added to the {@code StringJoiner} and methods
     * accessing the value of it are invoked, it will not return a
     * {@code prefix} or {@code suffix} (or properties thereof) in the result,
     * unless {@code setEmptyValue} has first been called.
     *
     * @param delimiter the sequence of characters to be used between each
     *                  element added to the {@code StringJoiner} value
     * @throws NullPointerException if {@code delimiter} is {@code null}
     */
    public StringLinker(CharSequence delimiter) {
        Objects.requireNonNull(delimiter, "The delimiter must not be null");
        this.delimiter = delimiter.toString();
        this.emptyValue = this.prefix + this.suffix;
    }

    /**
     * Constructs a {@code StringJoiner} with no characters in it using copies
     * of the supplied {@code prefix}, {@code delimiter} and {@code suffix}.
     * If no characters are added to the {@code StringJoiner} and methods
     * accessing the string value of it are invoked, it will return the
     * {@code prefix + suffix} (or properties thereof) in the result, unless
     * {@code setEmptyValue} has first been called.
     *
     * @param prefix    the sequence of characters to be used at the beginning
     * @param delimiter the sequence of characters to be used between each
     *                  element added to the {@code StringJoiner}
     * @param suffix    the sequence of characters to be used at the end
     * @throws NullPointerException if {@code prefix}, {@code delimiter}, or
     *                              {@code suffix} is {@code null}
     */
    public StringLinker(CharSequence prefix,
                        CharSequence delimiter,
                        CharSequence suffix) {
        Objects.requireNonNull(prefix, "The prefix must not be null");
        Objects.requireNonNull(delimiter, "The delimiter must not be null");
        Objects.requireNonNull(suffix, "The suffix must not be null");
        // make defensive copies of arguments
        this.prefix = prefix.toString();
        this.delimiter = delimiter.toString();
        this.suffix = suffix.toString();
        this.emptyValue = this.prefix + this.suffix;
    }

    public StringLinker wrapper(CharSequence prefix,
                                CharSequence suffix) {
        Objects.requireNonNull(prefix, "The prefix must not be null");
        Objects.requireNonNull(suffix, "The suffix must not be null");
        // make defensive copies of arguments
        this.prefix = prefix.toString();
        this.suffix = suffix.toString();
        this.emptyValue = this.prefix + this.suffix;
        return this;
    }

    public StringLinker emptyValue(CharSequence emptyValue) {
        this.emptyValue = Objects.requireNonNull(emptyValue,
                "The empty value must not be null").toString();
        return this;
    }

    public StringLinker nullElement(CharSequence defaultElementIfNull) {
        this.defaultStringIfNull = Objects.requireNonNull(defaultElementIfNull).toString();
        return this;
    }

    private StringLinker _add(CharSequence newElement) {
        prepareBuilder().append(newElement == null ? defaultStringIfNull : newElement);
        return this;
    }

    public StringLinker link(Object... newElements) {
        for (Object element : newElements) {
            add(element);
        }
        return this;
    }

    public StringLinker add(Object... newElements) {
        StringBuilder buf = new StringBuilder();
        for (Object element : newElements) {
            buf.append(WithDefault(element, defaultStringIfNull, String::valueOf));
        }
        return add(buf.toString());
    }

    public StringLinker add(Object newElement) {
        return _add(WithDefault(newElement, defaultStringIfNull, String::valueOf));
    }

    /**
     *
     * @param doubleValue doubleValue
     * @param scale 小数位数
     * @return this
     */
    public StringLinker add(Double doubleValue, int scale) {
        NumberFormat nf = NumberFormat.getNumberInstance();
        nf.setMaximumFractionDigits(scale);// 保留两位小数
        nf.setRoundingMode(RoundingMode.HALF_UP);
        return _add(WithDefault(doubleValue, defaultStringIfNull, nf::format));
    }

    public StringLinker addRepeat(CharSequence newElement, int times) {
        for (int i = 0; i < times; i++) {
            _add(newElement);
        }
        return this;
    }

    public StringLinker add(byte[] newElement) {
        return _add(HexUtil.BytesToHex(newElement));
    }

    public StringLinker add(int[] newElement) {
        return _add(Arrays.toString(newElement));
    }

    public StringLinker add(float[] newElement) {
        return _add(Arrays.toString(newElement));
    }

    public StringLinker add(long[] newElement) {
        return _add(Arrays.toString(newElement));
    }

    public StringLinker add(short[] newElement) {
        return _add(Arrays.toString(newElement));
    }

    public StringLinker add(double[] newElement) {
        return _add(Arrays.toString(newElement));
    }

    public StringLinker merge(StringLinker other) {
        Objects.requireNonNull(other);
        if (other.value != null) {
            final int length = other.value.length();
            StringBuilder builder = prepareBuilder();
            builder.append(other.value, other.prefix.length(), length);
        }
        return this;
    }

    private StringBuilder prepareBuilder() {
        if (value != null) {
            value.append(delimiter);
        } else {
            value = new StringBuilder().append(prefix);
        }
        return value;
    }

    @Override
    public int length() {
        // Remember that we never actually append the suffix unless we return
        // the full (present) value or some sub-string or length of it, so that
        // we can add on more if we need to.
        return (value != null ? value.length() + suffix.length() :
                emptyValue.length());
    }

    @Override
    public char charAt(int index) {
        return value.charAt(index);
    }

    @Override
    public CharSequence subSequence(int start, int end) {
        return value.subSequence(start, end);
    }

    @Override
    public String toString() {
        if (value == null) {
            return this.prefix + this.suffix;
        } else {
            if (suffix.equals("")) {
                return value.toString();
            } else {
                int initialLength = value.length();
                String result = value.append(suffix).toString();
                // reset value to pre-append initialLength
                value.setLength(initialLength);
                return result;
            }
        }
    }


    public static String strcat(Object... objs) {
        return StringLinker.of().add(objs).toString();
    }

    public static String strdeli(CharSequence delimiter, Object... objs) {
        return StringLinker.of(",").nullElement("").add(objs).toString();
    }

}
