/*
 */
package cn.gongler.util.sgeo.geo;

import java.awt.geom.Path2D;
import java.time.LocalDateTime;
import java.util.*;
import java.util.stream.Collectors;

/**
 * ScopeInfo, GeoScope合并，并作为不可变实例（新潮流）。 可以作为Map的Key。
 *
 * @author gongler
 * @since 2017.09.06
 */
public class Scope implements Comparable<Scope>, Iterable<IGeoPoint> {

    private static final long serialVersionUID = 1L;//since 2017-09-06

    private final long id;
    private String name;
    private long num1;
    private long num2;
    private String str1 = "";
    private String str2 = "";

    private long groupId;//

    private final Path2D.Double polygon = new Path2D.Double();//无点时围栏内外判断不报错（已验证）: System.out.println(""+new Path2D.Double().contains(1, 1));//false
    private int pointCount = 0;
    private final List<IGeoPoint> geoPoints = new ArrayList<>();//20160922add
    private double area;

    private final LocalDateTime createTime = LocalDateTime.now();

    private Scope(long id) {
        this.id = id;
    }

    public long id() {
        return id;
    }

    public String name() {
        return name;
    }

    public long groupId() {
        return groupId;
    }

    public LocalDateTime createTime() {
        return this.createTime;
    }

    public boolean inside(double geoL, double geoB) {//对应经度、纬度
        return polygon.contains(geoL, geoB);
    }

    public boolean inside(IGeoPoint gps) {//对应经度、纬度
        return inside(gps.gpsLng(), gps.gpsLat());
    }

    public static Scope FirstInside(IGeoPoint gps, Collection<Scope> scopes) {
        for (Scope scope : scopes) {
            if (scope.inside(gps)) {
                return scope;
            }
        }
        return null;
    }

    public static List<Scope> AllInside(IGeoPoint gps, Collection<Scope> scopes) {
        return scopes.stream().filter(a -> a.inside(gps)).collect(Collectors.toList());
    }

    public int pointCount() {
        return pointCount;
    }

    public List<IGeoPoint> geoPoints() {
        return this.geoPoints;
    }

    /**
     * 注释：近似围栏面积（把经纬度直接作为直角坐标系计算），可用于比较一个城市、地区内围栏的大小比较。
     * 球面三角形面积计算参见：https://wenku.baidu.com/view/d905d77e7375a417866f8f83.html
     *
     * @return 返回区域
     */
    public double polygonArea() {
        return area;
    }

    private static double TriangleArea(XY p0, XY p1, XY p2) {
        //0.5*(x1-x0)(y2-y0)-(x2-x0)(y1-y0)) 
        double ret = 0.5D * ((p1.x() - p0.x()) * (p2.y() - p0.y()) - (p2.x() - p0.x()) * (p1.y() - p0.y()));
        return ret;
    }

    private static double TriangleArea(IGeoPoint p1, IGeoPoint p2, IGeoPoint p3) {
        return TriangleArea(XY.of(p1), XY.of(p2), XY.of(p3));
    }

    private Scope flushPolygonArea() {//20170901gongleradd 
        IGeoPoint first = null;
        double areaTmp = 0;
        List<IGeoPoint> list = new ArrayList<>(geoPoints);
        if (!list.isEmpty()) {
            first = list.get(0);
            list.add(first);//close
        }
        //System.out.println("<|:"+TriangleArea(geoPoints.get(0), geoPoints.get(1), geoPoints.get(2)));
        for (int i = list.size() - 1; i > 0; i--) {
            IGeoPoint prev = list.get(i - 1);
            IGeoPoint next = list.get(i);
            areaTmp += TriangleArea(first, prev, next);
        }
        this.area = Math.abs(areaTmp);
        return this;
    }

    @Override
    public int compareTo(Scope o) {
        return Long.compare(this.id(), o.id());
    }

    @Override
    public boolean equals(Object o) {
        if (o instanceof Scope) {
            Scope anather = (Scope) o;
            return Objects.equals(this.id(), anather.id());
        }
        return false;
    }

    @Override
    public int hashCode() {
        int hash = 5;
        hash = 47 * hash + (int) (this.id ^ (this.id >>> 32));
        return hash;
    }

    @Override
    public Iterator<IGeoPoint> iterator() {
        return this.geoPoints.iterator();
    }

    @Override
    public String toString() {
        return "scope" + id() + "," + name + " num1:" + num1 + " num2:" + num2 + " pnt:" + pointCount() + " area:" + polygonArea();
    }

    public static class Builder {

        private static final long serialVersionUID = 1L;

        final Scope instance;

        public Builder(long id) {
            this.instance = new Scope(id);
        }

        public Builder name(String name) {
            instance.name = name;
            return this;
        }

        public Builder group(long groupId) {
            instance.groupId = groupId;
            return this;
        }

        public Builder num1(long num1) {
            instance.num1 = num1;
            return this;
        }

        public Builder num2(long num2) {
            instance.num2 = num2;
            return this;
        }

        public Builder str1(String str1) {
            instance.str1 = str1;
            return this;
        }

        public Builder str2(String str2) {
            instance.str2 = str2;
            return this;
        }

        public Builder add(double geoL, double geoB) {//对应经度、纬度
            if (instance.pointCount == 0) {
                instance.polygon.moveTo(geoL, geoB);
            } else {
                instance.polygon.lineTo(geoL, geoB);
            }
            instance.geoPoints.add(IGeoPoint.of(geoL, geoB));
            instance.pointCount++;
            return this;
        }

        public Builder add(IGeoPoint gps) {//对应经度、纬度
            return add(gps.gpsLng(), gps.gpsLat());
        }

        public Builder addAll(List<IGeoPoint> points) {//对应经度、纬度
            for (IGeoPoint point : points) {
                add(point);
            }
            return this;
        }

        public Scope build() {
            if (instance.pointCount > 0) {//java.awt.geom.IllegalPathStateException: missing initial moveto in path definition

                instance.polygon.closePath();//无点时围栏内外判断不报错（已验证），但是无点关闭(closePath())则有问题（java.awt.geom.IllegalPathStateException: missing initial moveto in path definition）。
            }
            instance.flushPolygonArea();
            return instance;
        }
    }

//    public static void main(String[] args) {
//        System.out.println("" + new Path2D.Double().contains(1, 1));
////        Scope scope = new Scope.Builder(1)
////                .add(0, 0)
////                .add(1, 0)
////                .add(1, 1)
////                //.add(0, 1)
////                .build();
////        System.out.println("" + scope.polygonArea());
////        System.out.println("" + scope.inside(0.5, 0.5));
////        System.out.println("" + scope.inside(1.5, 0.5));
////        Scope scope2 = new Scope.Builder(1).name("test2")
////                .add(11.0,114.556011)
////                .add(11.0,114.556357)
////                .add(2, 2)
////                .build();
////        System.out.println("" + scope2.polygonArea());
////        System.out.println("" + scope2);
//
////        Scope s = new Scope.Builder(1)
////                .add(11.0,1145560)
////                .add(11.0,1145563)
////                .add(11.0,1143984)
//////                .add(11.0,114.398785D*10000)
//////                .add(11.0,114.556703D*10000)
////                .build();
////        System.out.println("scope.polygonArea:" + s.polygonArea());
//        System.out.println("?" + TriangleArea(
//                XY.of(11.0D, 1145560D),
//                XY.of(11.0D, 1145563D),
//                XY.of(11.0D, 1143984D)));
//
//    }

    /**
     * @return the num1
     */
    public long num1() {
        return num1;
    }

    /**
     * @return the num2
     */
    public long num2() {
        return num2;
    }

    /**
     * @return the str1
     */
    public String str1() {
        return str1;
    }

    /**
     * @return the str2
     */
    public String str2() {
        return str2;
    }
}
