/*
 */
package cn.gongler.util.sgeo.geo;

import java.util.ArrayList;
import java.util.Collections;
import java.util.List;

import static java.lang.Math.*;

/**
 * @author gongler
 */
public final class GeoUtil {

    private static final long serialVersionUID = -8078555444382488053L;
    private static final double EARTH_RADIUS_KM = 6371d;//6 378.1 公里
    private static final double EARTH_RADIUS_METER = EARTH_RADIUS_KM * 1000D;//6 378.1 公里

    private GeoUtil() {
    }//禁止实例化

    /**
     * 求2经纬度坐标间距离
     *
     * @param a first point
     * @param b second point
     * @return 返回距离（单位：米）
     * @since 2017.09.30
     */
    public static double DisMeters(IGeoPoint a, IGeoPoint b) {
        return GetDistance(a.gpsLng(), a.gpsLat(), b.gpsLng(), b.gpsLat()) * 1000D;
    }

    /**
     * @param a a点
     * @param b b点
     * @return 返回距离（公里）
     */
    @Deprecated
    public static double GetDistance(IGeoPoint a, IGeoPoint b) {
        return GetDistance(a.gpsLng(), a.gpsLat(), b.gpsLng(), b.gpsLat());
    }

    /**
     * 角度
     *
     * @param geoL1 经度1
     * @param geoB1 纬度1
     * @param geoL2 经度2
     * @param geoB2 纬度2
     * @return 单位为公里
     */
    @Deprecated
    public static double GetDistance(double geoL1, double geoB1, double geoL2, double geoB2) {
        //System.out.println(""+geoL1+", "+geoB1+" to "+geoL2+", "+geoB2);
        return GetDistanceRad(toRadians(geoL1), toRadians(geoB1), toRadians(geoL2), toRadians(geoB2));
    }

    /**
     * @param geoL1 弧度经度
     * @param geoB1 弧度纬度
     * @param geoL2 经度2
     * @param geoB2 纬度2
     * @return 单位为公里
     */
    @Deprecated
    public static double GetDistanceRad(double geoL1, double geoB1, double geoL2, double geoB2) {
        //final double radians = acos(sin(geoB1)*sin(geoB2)+cos(geoB1)*cos(geoB2)*cos(geoL1 - geoL2));
        double val = sin(geoB1) * sin(geoB2) + cos(geoB1) * cos(geoB2) * cos(geoL1 - geoL2);
        val = max(min(val, 1), -1);
        final double radians = acos(val);
        final double distance = radians * EARTH_RADIUS_KM; //6 378.1 公里
        return distance;
    }

    /**
     * 根据圆心和半径生成12边形围栏
     *
     * @param center       圆心经纬度
     * @param radiusMeters 半径（单位：米）
     * @return 返回折线列表
     * @since 2017.09.30
     */
    public static List<IGeoPoint> PolygonFromCircleWith(IGeoPoint center, double radiusMeters) {
        return PolygonFromCircleWith(center, radiusMeters, 12);
    }

    /**
     * 根据圆心和半径生成围栏
     *
     * @param center       圆心经纬度
     * @param radiusMeters 半径（单位：米）
     * @param pointCnt     围栏节点数
     * @return 返回折线列表
     * @since 2017.09.30
     */
    public static List<IGeoPoint> PolygonFromCircleWith(IGeoPoint center, double radiusMeters, int pointCnt) {
        double radiusRadian = radiusMeters / EARTH_RADIUS_METER;//半径距离弧度
        List<IGeoPoint> list = new ArrayList(pointCnt);
        for (int i = 0; i < pointCnt; i++) {
            final double radian = Math.PI * 2 / pointCnt * i;
            double geoLDeta = center.gpsLng() + Math.toDegrees(sin(radian) * radiusRadian / cos(Math.toRadians(center.gpsLat())));//double geoLDeta = center.gpsLng() + Rad2Degree(sin(radian) * radiusRadian / cos(Degree2Rad(center.gpsLat())));
            double geoBDeta = center.gpsLat() - Math.toDegrees(cos(radian) * radiusRadian);//double geoBDeta = center.gpsLat() - Rad2Degree(cos(radian) * radiusRadian);
            IGeoPoint point = IGeoPoint.of(geoLDeta, geoBDeta);
            list.add(point);
        }
        return Collections.unmodifiableList(list);
    }

//    public static void main(String[] args) {
////        // 116.642068  39.547473 116.631663  39.552052
////        System.out.println(GetDistance(106.0932109382, 30.8125512978, 106.09259096299999, 30.8125746302));
////        //System.out.println(GetDistance2(106.0932109382, 30.8125512978, 106.09259096299999, 30.8125746302));
////
////        System.out.println(GetDistance(116.642068, 39.547473, 116.631663, 39.552052));
//
//        IGeoPoint center = IGeoPoint.of(116.642068, 39.547473);
//        System.out.println("center:" + center);
//        List<IGeoPoint> polygon = PolygonFromCircleWith(center, 50, 12);
//        for (IGeoPoint point : polygon) {
//            System.out.println("" + center.metersTo(point) + "//" + point);
//        }
//
//    }
}
