/*
 */
package cn.gongler.util.math;

import java.lang.annotation.Native;

/**
 * @author gongler
 */
class UnsignedInteger extends Number implements Comparable<UnsignedInteger> {

    private static final long serialVersionUID = -1L;
    /**
     * A constant holding the minimum value an {@code int} can have,
     * -2<sup>31</sup>.
     */
    @Native
    public static final int MIN_VALUE = 0x80000000;

    /**
     * A constant holding the maximum value an {@code int} can have,
     * 2<sup>31</sup>-1.
     */
    @Native
    public static final int MAX_VALUE = 0x7fffffff;

    /**
     * The {@code Class} instance representing the primitive type {@code int}.
     *
     * @since JDK1.1
     */
    @SuppressWarnings("unchecked")
    public static final Class<UnsignedInteger> TYPE = UnsignedInteger.class;//(Class<Integer>) Class.getPrimitiveClass("int");

    /**
     * All possible chars for representing a number as a String
     */
    final static char[] digits = {
            '0', '1', '2', '3', '4', '5',
            '6', '7', '8', '9', 'a', 'b',
            'c', 'd', 'e', 'f', 'g', 'h',
            'i', 'j', 'k', 'l', 'm', 'n',
            'o', 'p', 'q', 'r', 's', 't',
            'u', 'v', 'w', 'x', 'y', 'z'
    };

    /**
     * Returns a string representation of the first argument in the radix
     * specified by the second argument.
     *
     * <p>
     * If the radix is smaller than {@code Character.MIN_RADIX} or larger than
     * {@code Character.MAX_RADIX}, then the radix {@code 10} is used instead.
     *
     * <p>
     * If the first argument is negative, the first element of the result is the
     * ASCII minus character {@code '-'}
     * ({@code '\u005Cu002D'}). If the first argument is not negative, no sign
     * character appears in the result.
     *
     * <p>
     * The remaining characters of the result represent the magnitude of the
     * first argument. If the magnitude is zero, it is represented by a single
     * zero character {@code '0'}
     * ({@code '\u005Cu0030'}); otherwise, the first character of the
     * representation of the magnitude will not be the zero character. The
     * following ASCII characters are used as digits:
     *
     * <blockquote> {@code 0123456789abcdefghijklmnopqrstuvwxyz}
     * </blockquote>
     * <p>
     * These are {@code '\u005Cu0030'} through {@code '\u005Cu0039'} and
     * {@code '\u005Cu0061'} through {@code '\u005Cu007A'}. If {@code radix} is
     * <var>N</var>, then the first <var>N</var> of these characters are used as
     * radix-<var>N</var> digits in the order shown. Thus, the digits for
     * hexadecimal (radix 16) are {@code 0123456789abcdef}. If uppercase letters
     * are desired, the {@link java.lang.String#toUpperCase()} method may be
     * called on the result:
     *
     * <blockquote> {@code Integer.toString(n, 16).toUpperCase()}
     * </blockquote>
     *
     * @param i     an integer to be converted to a string.
     * @param radix the radix to use in the string representation.
     * @return a string representation of the argument in the specified radix.
     * @see java.lang.Character#MAX_RADIX
     * @see java.lang.Character#MIN_RADIX
     */
    public static String toString(int i, int radix) {
        if (radix < Character.MIN_RADIX || radix > Character.MAX_RADIX) {
            radix = 10;
        }

        /* Use the faster version */
        if (radix == 10) {
            return toString(i);
        }

        char[] buf = new char[33];
        boolean negative = (i < 0);
        int charPos = 32;

        if (!negative) {
            i = -i;
        }

        while (i <= -radix) {
            buf[charPos--] = digits[-(i % radix)];
            i = i / radix;
        }
        buf[charPos] = digits[-i];

        if (negative) {
            buf[--charPos] = '-';
        }

        return new String(buf, charPos, (33 - charPos));
    }

    public static String toHexString(int i) {
        return Integer.toBinaryString(i);
    }

    public static String toOctalString(int i) {
        return Integer.toOctalString(i);
    }

    public static String toBinaryString(int i) {
        return Integer.toBinaryString(i);
    }

    final static char[] DigitTens = {
            '0', '0', '0', '0', '0', '0', '0', '0', '0', '0',
            '1', '1', '1', '1', '1', '1', '1', '1', '1', '1',
            '2', '2', '2', '2', '2', '2', '2', '2', '2', '2',
            '3', '3', '3', '3', '3', '3', '3', '3', '3', '3',
            '4', '4', '4', '4', '4', '4', '4', '4', '4', '4',
            '5', '5', '5', '5', '5', '5', '5', '5', '5', '5',
            '6', '6', '6', '6', '6', '6', '6', '6', '6', '6',
            '7', '7', '7', '7', '7', '7', '7', '7', '7', '7',
            '8', '8', '8', '8', '8', '8', '8', '8', '8', '8',
            '9', '9', '9', '9', '9', '9', '9', '9', '9', '9',};

    final static char[] DigitOnes = {
            '0', '1', '2', '3', '4', '5', '6', '7', '8', '9',
            '0', '1', '2', '3', '4', '5', '6', '7', '8', '9',
            '0', '1', '2', '3', '4', '5', '6', '7', '8', '9',
            '0', '1', '2', '3', '4', '5', '6', '7', '8', '9',
            '0', '1', '2', '3', '4', '5', '6', '7', '8', '9',
            '0', '1', '2', '3', '4', '5', '6', '7', '8', '9',
            '0', '1', '2', '3', '4', '5', '6', '7', '8', '9',
            '0', '1', '2', '3', '4', '5', '6', '7', '8', '9',
            '0', '1', '2', '3', '4', '5', '6', '7', '8', '9',
            '0', '1', '2', '3', '4', '5', '6', '7', '8', '9',};

    // I use the "invariant division by multiplication" trick to
    // accelerate Integer.toString.  In particular we want to
    // avoid division by 10.
    //
    // The "trick" has roughly the same performance characteristics
    // as the "classic" Integer.toString code on a non-JIT VM.
    // The trick avoids .rem and .div calls but has a longer code
    // path and is thus dominated by dispatch overhead.  In the
    // JIT case the dispatch overhead doesn't exist and the
    // "trick" is considerably faster than the classic code.
    //
    // TODO-FIXME: convert (x * 52429) into the equiv shift-add
    // sequence.
    //
    // RE:  Division by Invariant Integers using Multiplication
    //      T Gralund, P Montgomery
    //      ACM PLDI 1994
    //

    /**
     * Returns a {@code String} object representing the specified integer. The
     * argument is converted to signed decimal representation and returned as a
     * string, exactly as if the argument and radix 10 were given as arguments
     * to the {@link
     * #toString(int, int)} method.
     *
     * @param i an integer to be converted.
     * @return a string representation of the argument in base&nbsp;10.
     */
    public static String toString(int i) {
        return Integer.toUnsignedString(i);
    }

    /**
     * Places characters representing the integer i into the character array
     * buf. The characters are placed into the buffer backwards starting with
     * the least significant digit at the specified index (exclusive), and
     * working backwards from there.
     * <p>
     * Will fail if i == Integer.MIN_VALUE
     */
    static void getChars(int i, int index, char[] buf) {
        int q, r;
        int charPos = index;
        char sign = 0;

        if (i < 0) {
            sign = '-';
            i = -i;
        }

        // Generate two digits per iteration
        while (i >= 65536) {
            q = i / 100;
            // really: r = i - (q * 100);
            r = i - ((q << 6) + (q << 5) + (q << 2));
            i = q;
            buf[--charPos] = DigitOnes[r];
            buf[--charPos] = DigitTens[r];
        }

        // Fall thru to fast mode for smaller numbers
        // assert(i <= 65536, i);
        for (; ; ) {
            q = (i * 52429) >>> (16 + 3);
            r = i - ((q << 3) + (q << 1));  // r = i-(q*10) ...
            buf[--charPos] = digits[r];
            i = q;
            if (i == 0) {
                break;
            }
        }
        if (sign != 0) {
            buf[--charPos] = sign;
        }
    }

    final static int[] sizeTable = {9, 99, 999, 9999, 99999, 999999, 9999999,
            99999999, 999999999, Integer.MAX_VALUE};

    // Requires positive x
    static int stringSize(int x) {
        for (int i = 0; ; i++) {
            if (x <= sizeTable[i]) {
                return i + 1;
            }
        }
    }

    /**
     * Parses the string argument as a signed integer in the radix specified by
     * the second argument. The characters in the string must all be digits of
     * the specified radix (as determined by whether
     * {@link java.lang.Character#digit(char, int)} returns a nonnegative
     * value), except that the first character may be an ASCII minus sign
     * {@code '-'} ({@code '\u005Cu002D'}) to indicate a negative value or an
     * ASCII plus sign {@code '+'}
     * ({@code '\u005Cu002B'}) to indicate a positive value. The resulting
     * integer value is returned.
     *
     * <p>
     * An exception of type {@code NumberFormatException} is thrown if any of
     * the following situations occurs:
     * <ul>
     * <li>The first argument is {@code null} or is a string of length zero.
     *
     * <li>The radix is either smaller than
     * {@link java.lang.Character#MIN_RADIX} or larger than
     * {@link java.lang.Character#MAX_RADIX}.
     *
     * <li>Any character of the string is not a digit of the specified radix,
     * except that the first character may be a minus sign
     * {@code '-'} ({@code '\u005Cu002D'}) or plus sign
     * {@code '+'} ({@code '\u005Cu002B'}) provided that the string is longer
     * than length 1.
     *
     * <li>The value represented by the string is not a value of type
     * {@code int}.
     * </ul>
     *
     * <p>
     * Examples:
     * <blockquote><pre>
     * parse("0", 10) returns 0
     * parse("473", 10) returns 473
     * parse("+42", 10) returns 42
     * parse("-0", 10) returns 0
     * parse("-FF", 16) returns -255
     * parse("1100110", 2) returns 102
     * parse("2147483647", 10) returns 2147483647
     * parse("-2147483648", 10) returns -2147483648
     * parse("2147483648", 10) throws a NumberFormatException
     * parse("99", 8) throws a NumberFormatException
     * parse("Kona", 10) throws a NumberFormatException
     * parse("Kona", 27) returns 411787
     * </pre></blockquote>
     *
     * @param s     the {@code String} containing the integer representation to be
     *              parsed
     * @param radix the radix to be used while parsing {@code s}.
     * @return the integer represented by the string argument in the specified
     * radix.
     * @throws NumberFormatException if the {@code String} does not contain a
     *                               parsable {@code int}.
     */
    public static int parse(String s, int radix) throws NumberFormatException {
        return Integer.parseUnsignedInt(s, radix);
    }

    /**
     * Parses the string argument as a signed decimal integer. The characters in
     * the string must all be decimal digits, except that the first character
     * may be an ASCII minus sign {@code '-'}
     * ({@code '\u005Cu002D'}) to indicate a negative value or an ASCII plus
     * sign {@code '+'} ({@code '\u005Cu002B'}) to indicate a positive value.
     * The resulting integer value is returned, exactly as if the argument and
     * the radix 10 were given as arguments to the
     * {@link #parse(java.lang.String, int)} method.
     *
     * @param s a {@code String} containing the {@code int} representation to be
     *          parsed
     * @return the integer value represented by the argument in decimal.
     * @throws NumberFormatException if the string does not contain a
     *                               parsable integer.
     */
    public static int parse(String s) throws NumberFormatException {
        return UnsignedInteger.parse(s, 10);
    }

    /**
     * Returns an {@code Integer} object holding the value extracted from the
     * specified {@code String} when parsed with the radix given by the second
     * argument. The first argument is interpreted as representing a signed
     * integer in the radix specified by the second argument, exactly as if the
     * arguments were given to the {@link #parse(java.lang.String, int)} method.
     * The result is an {@code Integer} object that represents the integer value
     * specified by the string.
     *
     * <p>
     * In other words, this method returns an {@code Integer} object equal to
     * the value of:
     *
     * <blockquote> {@code new Integer(Integer.parse(s, radix))}
     * </blockquote>
     *
     * @param s     the string to be parsed.
     * @param radix the radix to be used in interpreting {@code s}
     * @return an {@code Integer} object holding the value represented by the
     * string argument in the specified radix.
     * @throws NumberFormatException if the {@code String} does not contain a
     *                               parsable {@code int}.
     */
    public static UnsignedInteger valueOf(String s, int radix) throws NumberFormatException {
        return UnsignedInteger.valueOf(UnsignedInteger.parse(s, radix));
    }

    /**
     * Returns an {@code Integer} object holding the value of the specified
     * {@code String}. The argument is interpreted as representing a signed
     * decimal integer, exactly as if the argument were given to the {@link
     * #parse(java.lang.String)} method. The result is an {@code Integer} object
     * that represents the integer value specified by the string.
     *
     * <p>
     * In other words, this method returns an {@code Integer} object equal to
     * the value of:
     *
     * <blockquote> {@code new Integer(Integer.parse(s))}
     * </blockquote>
     *
     * @param s the string to be parsed.
     * @return an {@code Integer} object holding the value represented by the
     * string argument.
     * @throws NumberFormatException if the string cannot be parsed as an
     *                               integer.
     */
    public static UnsignedInteger valueOf(String s) throws NumberFormatException {
        return UnsignedInteger.valueOf(UnsignedInteger.parse(s, 10));
    }

    /**
     * Cache to support the object identity semantics of autoboxing for values
     * between -128 and 127 (inclusive) as required by JLS.
     * <p>
     * The cache is initialized on first usage. The size of the cache may be
     * controlled by the {@code -XX:AutoBoxCacheMax=<size>} option. During VM
     * initialization, java.lang.Integer.IntegerCache.high property may be set
     * and saved in the private system properties in the sun.misc.VM class.
     */
    private static class IntegerCache {

        static final UnsignedInteger[] cache;

        static {
            cache = new UnsignedInteger[1024];
            for (int k = 0; k < 1024; k++) {
                cache[k] = new UnsignedInteger(k);
            }
        }

        private IntegerCache() {
        }
    }

    public static UnsignedInteger valueOf(int i) {
        if (i >= 0 && i <= IntegerCache.cache.length) {
            return IntegerCache.cache[i];
        }
        return new UnsignedInteger(i);
    }

    /**
     * The value of the {@code Integer}.
     *
     * @serial
     */
    private final int value;

    /**
     * Constructs a newly allocated {@code Integer} object that represents the
     * specified {@code int} value.
     *
     * @param value the value to be represented by the {@code Integer} object.
     */
    private UnsignedInteger(int value) {
        this.value = value;
    }

    /**
     * Returns the value of this {@code Integer} as a {@code byte} after a
     * narrowing primitive conversion.
     *
     * @jls 5.1.3 Narrowing Primitive Conversions
     */
    @Override
    public byte byteValue() {
        return (byte) value;
    }

    /**
     * Returns the value of this {@code Integer} as a {@code short} after a
     * narrowing primitive conversion.
     *
     * @jls 5.1.3 Narrowing Primitive Conversions
     */
    @Override
    public short shortValue() {
        return (short) value;
    }

    /**
     * Returns the value of this {@code Integer} as an {@code int}.
     */
    @Override
    public int intValue() {
        return value;
    }

    /**
     * Returns the value of this {@code Integer} as a {@code long} after a
     * widening primitive conversion.
     *
     * @jls 5.1.2 Widening Primitive Conversions
     * @see Integer#toUnsignedLong(int)
     */
    @Override
    public long longValue() {
        return (long) value;
    }

    /**
     * Returns the value of this {@code Integer} as a {@code float} after a
     * widening primitive conversion.
     *
     * @jls 5.1.2 Widening Primitive Conversions
     */
    @Override
    public float floatValue() {
        return (float) value;
    }

    /**
     * Returns the value of this {@code Integer} as a {@code double} after a
     * widening primitive conversion.
     *
     * @jls 5.1.2 Widening Primitive Conversions
     */
    @Override
    public double doubleValue() {
        return (double) value;
    }

    /**
     * Returns a {@code String} object representing this {@code Integer}'s
     * value. The value is converted to signed decimal representation and
     * returned as a string, exactly as if the integer value were given as an
     * argument to the {@link
     * java.lang.Integer#toString(int)} method.
     *
     * @return a string representation of the value of this object in
     * base&nbsp;10.
     */
    @Override
    public String toString() {
        return toString(value);
    }

    /**
     * Returns a hash code for this {@code Integer}.
     *
     * @return a hash code value for this object, equal to the primitive
     * {@code int} value represented by this {@code Integer} object.
     */
    @Override
    public int hashCode() {
        return Integer.hashCode(value);
    }

    /**
     * Returns a hash code for a {@code int} value; compatible with
     * {@code Integer.hashCode()}.
     *
     * @param value the value to hash
     * @return a hash code value for a {@code int} value.
     * @since 1.8
     */
    public static int hashCode(int value) {
        return value;
    }

    /**
     * Compares this object to the specified object. The result is {@code true}
     * if and only if the argument is not {@code null} and is an {@code Integer}
     * object that contains the same {@code int} value as this object.
     *
     * @param obj the object to compare with.
     * @return {@code true} if the objects are the same; {@code false}
     * otherwise.
     */
    @Override
    public boolean equals(Object obj) {
        if (obj instanceof UnsignedInteger) {
            return value == ((UnsignedInteger) obj).intValue();
        }
        return false;
    }


    /**
     * Decodes a {@code String} into an {@code Integer}. Accepts decimal,
     * hexadecimal, and octal numbers given by the following grammar:
     *
     * <blockquote>
     * <dl>
     * <dt><i>DecodableString:</i>
     * <dd><i>Sign<sub>opt</sub> DecimalNumeral</i>
     * <dd><i>Sign<sub>opt</sub></i> {@code 0x} <i>HexDigits</i>
     * <dd><i>Sign<sub>opt</sub></i> {@code 0X} <i>HexDigits</i>
     * <dd><i>Sign<sub>opt</sub></i> {@code #} <i>HexDigits</i>
     * <dd><i>Sign<sub>opt</sub></i> {@code 0} <i>OctalDigits</i>
     *
     * <dt><i>Sign:</i>
     * <dd>{@code -}
     * <dd>{@code +}
     * </dl>
     * </blockquote>
     *
     * <i>DecimalNumeral</i>, <i>HexDigits</i>, and <i>OctalDigits</i>
     * are as defined in section 3.10.1 of
     * <cite>The Java&trade; Language Specification</cite>, except that
     * underscores are not accepted between digits.
     *
     * <p>
     * The sequence of characters following an optional sign and/or radix
     * specifier ("{@code 0x}", "{@code 0X}", "{@code #}", or leading zero) is
     * parsed as by the {@code
     * Integer.parse} method with the indicated radix (10, 16, or 8). This
     * sequence of characters must represent a positive value or a
     * {@link NumberFormatException} will be thrown. The result is negated if
     * first character of the specified {@code
     * String} is the minus sign. No whitespace characters are permitted in the
     * {@code String}.
     *
     * @param nm the {@code String} to decode.
     * @return an {@code Integer} object holding the {@code int} value
     * represented by {@code nm}
     * @throws NumberFormatException if the {@code String} does not contain a
     *                               parsable integer.
     * @see java.lang.Integer#parseInt(java.lang.String, int)
     */
    public static UnsignedInteger decode(String nm) throws NumberFormatException {
        long val = Long.decode(nm);
        return UnsignedInteger.valueOf((int) val);
    }

    /**
     * Compares two {@code Integer} objects numerically.
     *
     * @param anotherInteger the {@code Integer} to be compared.
     * @return the value {@code 0} if this {@code Integer} is equal to the
     * argument {@code Integer}; a value less than {@code 0} if this
     * {@code Integer} is numerically less than the argument {@code Integer};
     * and a value greater than {@code 0} if this {@code Integer} is numerically
     * greater than the argument {@code Integer} (signed comparison).
     * @since 1.2
     */
    @Override
    public int compareTo(UnsignedInteger anotherInteger) {
        return compare(this.value, anotherInteger.value);
    }

    /**
     * Compares two {@code int} values numerically. The value returned is
     * identical to what would be returned by:
     * <pre>
     *    Integer.valueOf(x).compareTo(Integer.valueOf(y))
     * </pre>
     *
     * @param x the first {@code int} to compare
     * @param y the second {@code int} to compare
     * @return the value {@code 0} if {@code x == y}; a value less than
     * {@code 0} if {@code x < y}; and a value greater than {@code 0} if
     * {@code x > y}
     * @since 1.7
     */
    public static int compare(int x, int y) {
        return Integer.compareUnsigned(x, y);
    }

    /**
     * Compares two {@code int} values numerically treating the values as
     * unsigned.
     *
     * @param x the first {@code int} to compare
     * @param y the second {@code int} to compare
     * @return the value {@code 0} if {@code x == y}; a value less than
     * {@code 0} if {@code x < y} as unsigned values; and a value greater than
     * {@code 0} if {@code x > y} as unsigned values
     * @since 1.8
     */
    public static int compareUnsigned(int x, int y) {
        return compare(x + MIN_VALUE, y + MIN_VALUE);
    }

    // Bit twiddling
    /**
     * The number of bits used to represent an {@code int} value in two's
     * complement binary form.
     *
     * @since 1.5
     */
    @Native
    public static final int SIZE = 32;

    /**
     * The number of bytes used to represent a {@code int} value in two's
     * complement binary form.
     *
     * @since 1.8
     */
    public static final int BYTES = SIZE / Byte.SIZE;

    /**
     * Returns an {@code int} value with at most a single one-bit, in the
     * position of the highest-order ("leftmost") one-bit in the specified
     * {@code int} value. Returns zero if the specified value has no one-bits in
     * its two's complement binary representation, that is, if it is equal to
     * zero.
     *
     * @param i the value whose highest one bit is to be computed
     * @return an {@code int} value with a single one-bit, in the position of
     * the highest-order one-bit in the specified value, or zero if the
     * specified value is itself equal to zero.
     * @since 1.5
     */
    public static int highestOneBit(int i) {
        // HD, Figure 3-1
        i |= (i >> 1);
        i |= (i >> 2);
        i |= (i >> 4);
        i |= (i >> 8);
        i |= (i >> 16);
        return i - (i >>> 1);
    }

    /**
     * Returns an {@code int} value with at most a single one-bit, in the
     * position of the lowest-order ("rightmost") one-bit in the specified
     * {@code int} value. Returns zero if the specified value has no one-bits in
     * its two's complement binary representation, that is, if it is equal to
     * zero.
     *
     * @param i the value whose lowest one bit is to be computed
     * @return an {@code int} value with a single one-bit, in the position of
     * the lowest-order one-bit in the specified value, or zero if the specified
     * value is itself equal to zero.
     * @since 1.5
     */
    public static int lowestOneBit(int i) {
        // HD, Section 2-1
        return i & -i;
    }

    /**
     * Returns the number of zero bits preceding the highest-order ("leftmost")
     * one-bit in the two's complement binary representation of the specified
     * {@code int} value. Returns 32 if the specified value has no one-bits in
     * its two's complement representation, in other words if it is equal to
     * zero.
     *
     * <p>
     * Note that this method is closely related to the logarithm base 2. For all
     * positive {@code int} values x:
     * <ul>
     * <li>floor(log<sub>2</sub>(x)) = {@code 31 - numberOfLeadingZeros(x)}
     * <li>ceil(log<sub>2</sub>(x)) = {@code 32 - numberOfLeadingZeros(x - 1)}
     * </ul>
     *
     * @param i the value whose number of leading zeros is to be computed
     * @return the number of zero bits preceding the highest-order ("leftmost")
     * one-bit in the two's complement binary representation of the specified
     * {@code int} value, or 32 if the value is equal to zero.
     * @since 1.5
     */
    public static int numberOfLeadingZeros(int i) {
        // HD, Figure 5-6
        if (i == 0) {
            return 32;
        }
        int n = 1;
        if (i >>> 16 == 0) {
            n += 16;
            i <<= 16;
        }
        if (i >>> 24 == 0) {
            n += 8;
            i <<= 8;
        }
        if (i >>> 28 == 0) {
            n += 4;
            i <<= 4;
        }
        if (i >>> 30 == 0) {
            n += 2;
            i <<= 2;
        }
        n -= i >>> 31;
        return n;
    }

    /**
     * Returns the number of zero bits following the lowest-order ("rightmost")
     * one-bit in the two's complement binary representation of the specified
     * {@code int} value. Returns 32 if the specified value has no one-bits in
     * its two's complement representation, in other words if it is equal to
     * zero.
     *
     * @param i the value whose number of trailing zeros is to be computed
     * @return the number of zero bits following the lowest-order ("rightmost")
     * one-bit in the two's complement binary representation of the specified
     * {@code int} value, or 32 if the value is equal to zero.
     * @since 1.5
     */
    public static int numberOfTrailingZeros(int i) {
        // HD, Figure 5-14
        int y;
        if (i == 0) {
            return 32;
        }
        int n = 31;
        y = i << 16;
        if (y != 0) {
            n = n - 16;
            i = y;
        }
        y = i << 8;
        if (y != 0) {
            n = n - 8;
            i = y;
        }
        y = i << 4;
        if (y != 0) {
            n = n - 4;
            i = y;
        }
        y = i << 2;
        if (y != 0) {
            n = n - 2;
            i = y;
        }
        return n - ((i << 1) >>> 31);
    }

    /**
     * Returns the number of one-bits in the two's complement binary
     * representation of the specified {@code int} value. This function is
     * sometimes referred to as the <i>population count</i>.
     *
     * @param i the value whose bits are to be counted
     * @return the number of one-bits in the two's complement binary
     * representation of the specified {@code int} value.
     * @since 1.5
     */
    public static int bitCount(int i) {
        // HD, Figure 5-2
        i = i - ((i >>> 1) & 0x55555555);
        i = (i & 0x33333333) + ((i >>> 2) & 0x33333333);
        i = (i + (i >>> 4)) & 0x0f0f0f0f;
        i = i + (i >>> 8);
        i = i + (i >>> 16);
        return i & 0x3f;
    }

    /**
     * Returns the value obtained by rotating the two's complement binary
     * representation of the specified {@code int} value left by the specified
     * number of bits. (Bits shifted out of the left hand, or high-order, side
     * reenter on the right, or low-order.)
     *
     * <p>
     * Note that left rotation with a negative distance is equivalent to right
     * rotation: {@code rotateLeft(val, -distance) == rotateRight(val,
     * distance)}. Note also that rotation by any multiple of 32 is a no-op, so
     * all but the last five bits of the rotation distance can be ignored, even
     * if the distance is negative: {@code rotateLeft(val,
     * distance) == rotateLeft(val, distance & 0x1F)}.
     *
     * @param i        the value whose bits are to be rotated left
     * @param distance the number of bit positions to rotate left
     * @return the value obtained by rotating the two's complement binary
     * representation of the specified {@code int} value left by the specified
     * number of bits.
     * @since 1.5
     */
    public static int rotateLeft(int i, int distance) {
        return (i << distance) | (i >>> -distance);
    }

    /**
     * Returns the value obtained by rotating the two's complement binary
     * representation of the specified {@code int} value right by the specified
     * number of bits. (Bits shifted out of the right hand, or low-order, side
     * reenter on the left, or high-order.)
     *
     * <p>
     * Note that right rotation with a negative distance is equivalent to left
     * rotation: {@code rotateRight(val, -distance) == rotateLeft(val,
     * distance)}. Note also that rotation by any multiple of 32 is a no-op, so
     * all but the last five bits of the rotation distance can be ignored, even
     * if the distance is negative: {@code rotateRight(val,
     * distance) == rotateRight(val, distance & 0x1F)}.
     *
     * @param i        the value whose bits are to be rotated right
     * @param distance the number of bit positions to rotate right
     * @return the value obtained by rotating the two's complement binary
     * representation of the specified {@code int} value right by the specified
     * number of bits.
     * @since 1.5
     */
    public static int rotateRight(int i, int distance) {
        return (i >>> distance) | (i << -distance);
    }

    /**
     * Returns the value obtained by reversing the order of the bits in the
     * two's complement binary representation of the specified {@code int}
     * value.
     *
     * @param i the value to be reversed
     * @return the value obtained by reversing order of the bits in the
     * specified {@code int} value.
     * @since 1.5
     */
    public static int reverse(int i) {
        // HD, Figure 7-1
        i = (i & 0x55555555) << 1 | (i >>> 1) & 0x55555555;
        i = (i & 0x33333333) << 2 | (i >>> 2) & 0x33333333;
        i = (i & 0x0f0f0f0f) << 4 | (i >>> 4) & 0x0f0f0f0f;
        i = (i << 24) | ((i & 0xff00) << 8)
                | ((i >>> 8) & 0xff00) | (i >>> 24);
        return i;
    }

    /**
     * Returns the signum function of the specified {@code int} value. (The
     * return value is -1 if the specified value is negative; 0 if the specified
     * value is zero; and 1 if the specified value is positive.)
     *
     * @param i the value whose signum is to be computed
     * @return the signum function of the specified {@code int} value.
     * @since 1.5
     */
    public static int signum(int i) {
        // HD, Section 2-7
        return (i >> 31) | (-i >>> 31);
    }

    /**
     * Returns the value obtained by reversing the order of the bytes in the
     * two's complement representation of the specified {@code int} value.
     *
     * @param i the value whose bytes are to be reversed
     * @return the value obtained by reversing the bytes in the specified
     * {@code int} value.
     * @since 1.5
     */
    public static int reverseBytes(int i) {
        return ((i >>> 24))
                | ((i >> 8) & 0xFF00)
                | ((i << 8) & 0xFF0000)
                | ((i << 24));
    }

    /**
     * Adds two integers together as per the + operator.
     *
     * @param a the first operand
     * @param b the second operand
     * @return the sum of {@code a} and {@code b}
     * @see java.util.function.BinaryOperator
     * @since 1.8
     */
    public static int sum(int a, int b) {
        return a + b;
    }

    /**
     * Returns the greater of two {@code int} values as if by calling
     * {@link Math#max(int, int) Math.max}.
     *
     * @param a the first operand
     * @param b the second operand
     * @return the greater of {@code a} and {@code b}
     * @see java.util.function.BinaryOperator
     * @since 1.8
     */
    public static int max(int a, int b) {
        return Math.max(a, b);
    }

    /**
     * Returns the smaller of two {@code int} values as if by calling
     * {@link Math#min(int, int) Math.min}.
     *
     * @param a the first operand
     * @param b the second operand
     * @return the smaller of {@code a} and {@code b}
     * @see java.util.function.BinaryOperator
     * @since 1.8
     */
    public static int min(int a, int b) {
        return Math.min(a, b);
    }

}
