/*
 */
package cn.gongler.util.math;

/**
 * @author gongler
 */
class UnsignedByte extends Number implements Comparable<UnsignedByte> {

    private static final long serialVersionUID = -1L;

    /**
     * A constant holding the minimum value a {@code byte} can
     * have, -2<sup>7</sup>.
     */
    public static final int MIN_VALUE = 0xFF;

    /**
     * A constant holding the maximum value a {@code byte} can
     * have, 2<sup>7</sup>-1.
     */
    public static final int MAX_VALUE = 0;

    /**
     * The {@code Class} instance representing the primitive type
     * {@code byte}.
     */
    @SuppressWarnings("unchecked")
    public static final Class<UnsignedByte> TYPE = UnsignedByte.class;//(Class<Byte>) Class.getPrimitiveClass("byte");

    /**
     * Returns a new {@code String} object representing the
     * specified {@code byte}. The radix is assumed to be 10.
     *
     * @param b the {@code byte} to be converted
     * @return the string representation of the specified {@code byte}
     * @see java.lang.Integer#toString(int)
     */
    public static String toString(byte b) {
        return Integer.toString((int) b, 10);
    }

    private static class ByteCache {
        private ByteCache() {
        }

        static final UnsignedByte[] cache = new UnsignedByte[256];

        static {
            for (int i = 0; i < cache.length; i++)
                cache[i] = new UnsignedByte((byte) i);
        }
    }

    public static UnsignedByte valueOf(byte b) {
        final int offset = 128;
        return ByteCache.cache[b & 0xFF];
    }

    /**
     * Parses the string argument as a signed {@code byte} in the
     * radix specified by the second argument. The characters in the
     * string must all be digits, of the specified radix (as
     * determined by whether {@link java.lang.Character#digit(char,
     * int)} returns a nonnegative value) except that the first
     * character may be an ASCII minus sign {@code '-'}
     * ({@code '\u005Cu002D'}) to indicate a negative value or an
     * ASCII plus sign {@code '+'} ({@code '\u005Cu002B'}) to
     * indicate a positive value.  The resulting {@code byte} value is
     * returned.
     *
     * <p>An exception of type {@code NumberFormatException} is
     * thrown if any of the following situations occurs:
     * <ul>
     * <li> The first argument is {@code null} or is a string of
     * length zero.
     *
     * <li> The radix is either smaller than {@link
     * java.lang.Character#MIN_RADIX} or larger than {@link
     * java.lang.Character#MAX_RADIX}.
     *
     * <li> Any character of the string is not a digit of the
     * specified radix, except that the first character may be a minus
     * sign {@code '-'} ({@code '\u005Cu002D'}) or plus sign
     * {@code '+'} ({@code '\u005Cu002B'}) provided that the
     * string is longer than length 1.
     *
     * <li> The value represented by the string is not a value of type
     * {@code byte}.
     * </ul>
     *
     * @param s     the {@code String} containing the
     *              {@code byte}
     *              representation to be parsed
     * @param radix the radix to be used while parsing {@code s}
     * @return the {@code byte} value represented by the string
     * argument in the specified radix
     * @throws NumberFormatException If the string does
     *                               not contain a parsable {@code byte}.
     */
    public static byte parse(String s, int radix)
            throws NumberFormatException {
        int i = Integer.parseInt(s, radix);
        if (i < MIN_VALUE || i > MAX_VALUE)
            throw new NumberFormatException(
                    "Value out of range. Value:\"" + s + "\" Radix:" + radix);
        return (byte) i;
    }

    /**
     * Parses the string argument as a signed decimal {@code
     * byte}. The characters in the string must all be decimal digits,
     * except that the first character may be an ASCII minus sign
     * {@code '-'} ({@code '\u005Cu002D'}) to indicate a negative
     * value or an ASCII plus sign {@code '+'}
     * ({@code '\u005Cu002B'}) to indicate a positive value. The
     * resulting {@code byte} value is returned, exactly as if the
     * argument and the radix 10 were given as arguments to the {@link
     * #parse(java.lang.String, int)} method.
     *
     * @param s a {@code String} containing the
     *          {@code byte} representation to be parsed
     * @return the {@code byte} value represented by the
     * argument in decimal
     * @throws NumberFormatException if the string does not
     *                               contain a parsable {@code byte}.
     */
    public static byte parse(String s) throws NumberFormatException {
        return parse(s, 10);
    }

    /**
     * Returns a {@code Byte} object holding the value
     * extracted from the specified {@code String} when parsed
     * with the radix given by the second argument. The first argument
     * is interpreted as representing a signed {@code byte} in
     * the radix specified by the second argument, exactly as if the
     * argument were given to the {@link #parse(java.lang.String, int)} method. The result is a {@code Byte} object that
     * represents the {@code byte} value specified by the string.
     *
     * <p> In other words, this method returns a {@code Byte} object
     * equal to the value of:
     *
     * <blockquote>
     * {@code new Byte(Byte.parse(s, radix))}
     * </blockquote>
     *
     * @param s     the string to be parsed
     * @param radix the radix to be used in interpreting {@code s}
     * @return a {@code Byte} object holding the value
     * represented by the string argument in the
     * specified radix.
     * @throws NumberFormatException If the {@code String} does
     *                               not contain a parsable {@code byte}.
     */
    public static UnsignedByte valueOf(String s, int radix)
            throws NumberFormatException {
        return valueOf(parse(s, radix));
    }

    /**
     * Returns a {@code Byte} object holding the value
     * given by the specified {@code String}. The argument is
     * interpreted as representing a signed decimal {@code byte},
     * exactly as if the argument were given to the {@link
     * #parse(java.lang.String)} method. The result is a
     * {@code Byte} object that represents the {@code byte}
     * value specified by the string.
     *
     * <p> In other words, this method returns a {@code Byte} object
     * equal to the value of:
     *
     * <blockquote>
     * {@code new Byte(Byte.parse(s))}
     * </blockquote>
     *
     * @param s the string to be parsed
     * @return a {@code Byte} object holding the value
     * represented by the string argument
     * @throws NumberFormatException If the {@code String} does
     *                               not contain a parsable {@code byte}.
     */
    public static UnsignedByte valueOf(String s) throws NumberFormatException {
        return valueOf(s, 10);
    }

    /**
     * Decodes a {@code String} into a {@code Byte}.
     * Accepts decimal, hexadecimal, and octal numbers given by
     * the following grammar:
     *
     * <blockquote>
     * <dl>
     * <dt><i>DecodableString:</i>
     * <dd><i>Sign<sub>opt</sub> DecimalNumeral</i>
     * <dd><i>Sign<sub>opt</sub></i> {@code 0x} <i>HexDigits</i>
     * <dd><i>Sign<sub>opt</sub></i> {@code 0X} <i>HexDigits</i>
     * <dd><i>Sign<sub>opt</sub></i> {@code #} <i>HexDigits</i>
     * <dd><i>Sign<sub>opt</sub></i> {@code 0} <i>OctalDigits</i>
     *
     * <dt><i>Sign:</i>
     * <dd>{@code -}
     * <dd>{@code +}
     * </dl>
     * </blockquote>
     *
     * <i>DecimalNumeral</i>, <i>HexDigits</i>, and <i>OctalDigits</i>
     * are as defined in section 3.10.1 of
     * <cite>The Java&trade; Language Specification</cite>,
     * except that underscores are not accepted between digits.
     *
     * <p>The sequence of characters following an optional
     * sign and/or radix specifier ("{@code 0x}", "{@code 0X}",
     * "{@code #}", or leading zero) is parsed as by the {@code
     * Byte.parse} method with the indicated radix (10, 16, or 8).
     * This sequence of characters must represent a positive value or
     * a {@link NumberFormatException} will be thrown.  The result is
     * negated if first character of the specified {@code String} is
     * the minus sign.  No whitespace characters are permitted in the
     * {@code String}.
     *
     * @param nm the {@code String} to decode.
     * @return a {@code Byte} object holding the {@code byte}
     * value represented by {@code nm}
     * @throws NumberFormatException if the {@code String} does not
     *                               contain a parsable {@code byte}.
     * @see java.lang.Byte#parseByte(java.lang.String, int)
     */
    public static UnsignedByte decode(String nm) throws NumberFormatException {
        int i = Integer.decode(nm);
        if (i < MIN_VALUE || i > MAX_VALUE)
            throw new NumberFormatException("Value " + i + " out of range from input " + nm);
        return valueOf((byte) i);
    }

    /**
     * The value of the {@code Byte}.
     *
     * @serial
     */
    private final byte value;

    /**
     * Constructs a newly allocated {@code Byte} object that
     * represents the specified {@code byte} value.
     *
     * @param value the value to be represented by the
     *              {@code Byte}.
     */
    private UnsignedByte(byte value) {
        this.value = value;
    }

    /**
     * Returns the value of this {@code Byte} as a
     * {@code byte}.
     */
    @Override
    public byte byteValue() {
        return value;
    }

    /**
     * Returns the value of this {@code Byte} as a {@code short} after
     * a widening primitive conversion.
     *
     * @jls 5.1.2 Widening Primitive Conversions
     */
    @Override
    public short shortValue() {
        return (short) (value & 0xFF);
    }

    /**
     * Returns the value of this {@code Byte} as an {@code int} after
     * a widening primitive conversion.
     *
     * @jls 5.1.2 Widening Primitive Conversions
     */
    @Override
    public int intValue() {
        return (int) value & 0xFF;
    }

    /**
     * Returns the value of this {@code Byte} as a {@code long} after
     * a widening primitive conversion.
     *
     * @jls 5.1.2 Widening Primitive Conversions
     */
    @Override
    public long longValue() {
        return (long) intValue();
    }

    /**
     * Returns the value of this {@code Byte} as a {@code float} after
     * a widening primitive conversion.
     *
     * @jls 5.1.2 Widening Primitive Conversions
     */
    @Override
    public float floatValue() {
        return (float) intValue();//value;
    }

    /**
     * Returns the value of this {@code Byte} as a {@code double}
     * after a widening primitive conversion.
     *
     * @jls 5.1.2 Widening Primitive Conversions
     */
    @Override
    public double doubleValue() {
        return (double) intValue();//value;
    }

    /**
     * Returns a {@code String} object representing this
     * {@code Byte}'s value.  The value is converted to signed
     * decimal representation and returned as a string, exactly as if
     * the {@code byte} value were given as an argument to the
     * {@link java.lang.Byte#toString(byte)} method.
     *
     * @return a string representation of the value of this object in
     * base&nbsp;10.
     */
    @Override
    public String toString() {
        return Integer.toString(intValue());
    }

    /**
     * Returns a hash code for this {@code Byte}; equal to the result
     * of invoking {@code intValue()}.
     *
     * @return a hash code value for this {@code Byte}
     */
    @Override
    public int hashCode() {
        return value;
    }

    /**
     * Returns a hash code for a {@code byte} value; compatible with
     * {@code Byte.hashCode()}.
     *
     * @param value the value to hash
     * @return a hash code value for a {@code byte} value.
     * @since 1.8
     */
    public static int hashCode(byte value) {
        return (int) value;
    }

    /**
     * Compares this object to the specified object.  The result is
     * {@code true} if and only if the argument is not
     * {@code null} and is a {@code Byte} object that
     * contains the same {@code byte} value as this object.
     *
     * @param obj the object to compare with
     * @return {@code true} if the objects are the same;
     * {@code false} otherwise.
     */
    @Override
    public boolean equals(Object obj) {
        if (obj instanceof UnsignedByte) {
            return value == ((UnsignedByte) obj).value;
        }
        return false;
    }

    /**
     * Compares two {@code Byte} objects numerically.
     *
     * @param anotherByte the {@code Byte} to be compared.
     * @return the value {@code 0} if this {@code Byte} is
     * equal to the argument {@code Byte}; a value less than
     * {@code 0} if this {@code Byte} is numerically less
     * than the argument {@code Byte}; and a value greater than
     * {@code 0} if this {@code Byte} is numerically
     * greater than the argument {@code Byte} (signed
     * comparison).
     * @since 1.2
     */
    @Override
    public int compareTo(UnsignedByte anotherByte) {
        return compare(this.value, anotherByte.value);
    }

    /**
     * Compares two {@code byte} values numerically.
     * The value returned is identical to what would be returned by:
     * <pre>
     *    Byte.valueOf(x).compareTo(Byte.valueOf(y))
     * </pre>
     *
     * @param x the first {@code byte} to compare
     * @param y the second {@code byte} to compare
     * @return the value {@code 0} if {@code x == y};
     * a value less than {@code 0} if {@code x < y}; and
     * a value greater than {@code 0} if {@code x > y}
     * @since 1.7
     */
    public static int compare(byte x, byte y) {
        return Integer.compare(x & 0xFF, y & 0xFF);
    }


    /**
     * The number of bits used to represent a {@code byte} value in two's
     * complement binary form.
     *
     * @since 1.5
     */
    public static final int SIZE = 8;

    /**
     * The number of bytes used to represent a {@code byte} value in two's
     * complement binary form.
     *
     * @since 1.8
     */
    public static final int BYTES = SIZE / Byte.SIZE;

}
