/*
 */
package cn.gongler.util.io;

import cn.gongler.util.bytes.BytesBuilder;

import java.io.*;

/**
 * 允许输出16进制数据到文本文件。允许指定行宽
 *
 * @author gongler
 * @since 2018.03.27
 */
public class HexOutputStream extends OutputStream {

    private static final long serialVersionUID = -1L;
    public final static int DEFAULT_LINE_WIDTH = 16;

    private final Writer hexWriter;
    private int lineWidth = DEFAULT_LINE_WIDTH;
    private boolean includeBlankchar = false;

    private int columnCnt = 0;
    private int lineCnt = 0;
    private long lineBytePos = 0;

    private final BytesBuilder cache = BytesBuilder.of();

    public HexOutputStream(Writer hexWriter) {
        this.hexWriter = new BufferedWriter(hexWriter);
    }

    public HexOutputStream(File hexFile) throws IOException {
        this.hexWriter = new BufferedWriter(new FileWriter(hexFile));
    }

    public HexOutputStream lineWidth(int lineWidth) {
        this.lineWidth = lineWidth;
        return this;
    }

    public HexOutputStream blankChar(boolean includeBlankChar) {
        this.includeBlankchar = includeBlankChar;
        return this;
    }

    @Override
    public void write(int b) throws IOException {
        writeHalfByte(b >>> 4);
        writeHalfByte(b);
        cache.addByte(b);
        if (includeBlankchar) {
            hexWriter.write(' ');
        }
        columnCnt++;
        if (columnCnt == lineWidth) {
            newLine();
        }
    }

    /**
     * 2015-5-5，为了允许外部强制换行而新增（需求：凯里消费文件头和数据长度不再是统一的64字节：而是头不变，数据变为96字节）。
     *
     * @throws IOException 异常
     */
    public void newLine() throws IOException {
        if (columnCnt > 0) {//刚换完行，将不再换行。
            hexWriter.append("#L").append(String.valueOf(++lineCnt))
                    .append(", ").append(Long.toHexString(lineBytePos)).append("(").append(String.valueOf(lineBytePos)).append(")")
                    .append(" ").append(byteChars(cache.toBytes()))
            ;
            //cache.clear();
            lineBytePos += columnCnt;
            columnCnt = 0;
            hexWriter.write('\r');
            hexWriter.write('\n');
        }
    }

    private static String byteChars(byte[] bytes) {
        StringBuilder buf = new StringBuilder();
        for (byte b : bytes) {
            char ch = (char) (b & 0xFF);
            buf.append(ch >= ' ' && ch <= 0x7E ? ch : ".");
        }
        return buf.toString();
    }

    private static char HexValToAsc(int halfByte) throws IOException {
        final char[] table = new char[]{'0', '1', '2', '3', '4', '5', '6', '7', '8', '9', 'A', 'B', 'C', 'D', 'E', 'F'};
        return table[halfByte & 0x0f];

    }

    private void writeHalfByte(int halfByte) throws IOException {
        hexWriter.write(HexValToAsc(halfByte));
    }

    @Override
    public void close() throws IOException {
        if (hexWriter != null) {
            hexWriter.close();
        }
    }

//    public static void main(String[] args) throws FileNotFoundException, IOException {
//        HexOutputStream out = new HexOutputStream(new FileWriter("d:/test/XY_999999_121105154454.CS.TXT")).lineWidth(64).blankChar(true);
//        byte[] buf = Files.readAllBytes(Paths.get("d:/test/XY_999999_121105154454.CS"));
//        out.write(buf);
//        out.close();
//    }

}
