/*
 *
 *
 *
 */
package cn.gongler.util.function;

import cn.gongler.util.GonglerUtil;

import java.util.Objects;
import java.util.function.Function;

import static cn.gongler.util.GonglerUtil.CallWithThrowAny;

/**
 * @param <T> T
 * @param <R> result
 * @author gongler
 */
@FunctionalInterface
public interface ExceptionFunction<T, R> {

    /**
     * Applies this function to the given argument.
     *
     * @param t the function argument
     * @return the function result
     * @throws java.lang.Exception Exception
     */
    R apply(T t) throws Exception;

    default R applyWithCatchAny(T t) {
        return applyWithCatchAny(t, null);
    }

    default R applyWithCatchAny(T t, R defaultVal) {
        try {
            return this.apply(t);
        } catch (Throwable e) {
            e.printStackTrace();
            return defaultVal;
        }
    }

    default R applyWithThrowAny(T t) {
        return CallWithThrowAny(() -> this.apply(t));
    }

    /**
     * Returns a composed function that first applies the {@code before}
     * function to its input, and then applies this function to the result. If
     * evaluation of either function throws an exception, it is relayed to the
     * caller of the composed function.
     *
     * @param <V>    the type of input to the {@code before} function, and to the
     *               composed function
     * @param before the function to apply before this function is applied
     * @return a composed function that first applies the {@code before}
     * function and then applies this function
     * @throws NullPointerException if before is null
     */
    default <V> ExceptionFunction<V, R> compose(ExceptionFunction<? super V, ? extends T> before) {
        Objects.requireNonNull(before);
        return (V v) -> apply(before.apply(v));
    }

    /**
     * Returns a composed function that first applies this function to its
     * input, and then applies the {@code after} function to the result. If
     * evaluation of either function throws an exception, it is relayed to the
     * caller of the composed function.
     *
     * @param <V>   the type of output of the {@code after} function, and of the
     *              composed function
     * @param after the function to apply after this function is applied
     * @return a composed function that first applies this function and then
     * applies the {@code after} function
     * @throws NullPointerException if after is null
     */
    default <V> ExceptionFunction<T, V> andThen(ExceptionFunction<? super R, ? extends V> after) {
        Objects.requireNonNull(after);
        return (T t) -> after.apply(apply(t));
    }

    /**
     * Returns a function that always returns its input argument.
     *
     * @param <T> the type of the input and output objects to the function
     * @return a function that always returns its input argument
     */
    static <T> ExceptionFunction<T, T> identity() {
        return t -> t;
    }

    default Function<T, R> toFunction() {
        return t -> GonglerUtil.CallWithThrowAny(() -> this.apply(t));
    }
}
